package tables2html;

use strict;
use CheckPoint;
use Constants;
use Options;
use Sections2html;
use checkpoint2html;
use object2html;
use Strings;


require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(table2html);

#*************************************************************************
#***                        Tables to HTML                             ***
#*************************************************************************
# Routine: table2html
#
# Description:
# Overall routine to generate the different tables to HTML code
#
# Parameters:
# $Policy   Current policy name
# $template	Empty template to be filled for the html page
# $table	Name of table to be generated into template
#
# Returns:
# NrObjects The number of objects in the table
#
# $template	Filled template with the requested sections

sub table2html {
	use strict;
	my (@ObjList, $AllObjects);
	my ($obj, $table, $name, $type, $class);
	my (@list, $list);
	my ($CPtable, $CPclass);
	my ($key, $var, $ok);
	my $sub2html;
	
	my $Policy   = shift;
	my $template = shift;
	my $table    = shift;
	my $TABLE    = uc($table);

	# Initialize special marker signs ...
	my $Check      = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/check.png>';
	my $CheckTick  = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/checktick.png>';
	my $Mark       = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/mark.png>';
	my $MarkDot    = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/markdot.png>';
	my $Spacer     = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/spacer.png>';
	my $None       = '<I>(none)</I>';

	# Determine if all objects need to be listed ... 
	$AllObjects = Option($Policy, 'ALLOBJECTS');

	# Determine Check Point table name to be searched ...	
	$CPtable = $HTMLTable{$TABLE}{TABLE};

	if ($HTMLTable{$TABLE}{LIST}) {
		# Create list of TYPE object ...
		$type = $HTMLTable{$TABLE}{LIST};
		if (ref(GetKey($CPtable))) {
			# table is not empty ...
			@list=GetMembers(GetKey($CPtable));
			foreach $list (@list) {
				if (GetKey($list, 'collection:Name') eq $Policy) {
					@ObjList = GetMembers(GetKey($list, $type))
				}
			}
		}
	} else {
		# Create list of objects ...
		if (ref(GetKey($CPtable))) {
			# table is not empty ...
			foreach $name (keys %{GetKey($CPtable)}) {
				if ($AllObjects || (GetKey($CPtable, $name, 'is_used', $Policy) eq 'true')) {
					$CPclass = lc(GetKey($CPtable, $name, 'AdminInfo:ClassName'));
					$type = GetType($CPtable, $name, $HTMLTable{$TABLE}{CLASS}{$CPclass});
					if ($type) { push @ObjList, $type.':'.$name }
				}
			}
		}
	}

	# Create table if there are objects to be displayed ;
	# The exception is the NAT rulebase, as there does not
	# have to be actual NAT rules. They can be automatic.
	if ((@ObjList)||($TABLE eq 'NAT')) {
		
		# Define global variables in template file ...		
		$template->param(	'SECTION_'.$TABLE	=> 1,
							$TABLE.'_REF'		=> $table
						);
		
		# Call the corresponding subroutine ...	
		no strict 'refs';
		$sub2html = lc($table).'2html';
		if (defined(&$sub2html)) { &$sub2html($Policy, $template, $CPtable, \@ObjList) }
	}

	# Return the number of objects in the list
	return @ObjList;

##########################################################################
# End of main routine table2html
##########################################################################


	##########################################################################
	# (sub)Routine: GetType
	#
	# Description:
	# This routine determines the type of an object ...
	##########################################################################
	sub GetType {
		my $CPtable = shift;
		my $name    = shift;
		my $class   = shift;
		my $type;
		my ($key, $val, $var);
		
		$type = '';
		if (ref($class)) {
			foreach $key (keys %{$class}) {
				# Get value of key within object ...
				$val = lc(stripquotes(GetKey($CPtable, $name, $key)));
				foreach $var (keys %{$class->{$key}}) {
					if (lc($var) eq $val) { 
						$type = $class->{$key}{$val};
					}
				}
			}
		} else {
			$type = $class
		}
	
		return $type;
	}
	
	##########################################################################
	# Routine: security2html
	#
	# Description:
	# Routine to create the normal rules section
	##########################################################################
	sub security2html {
		use strict;
		my (@list, $list);
		my (@rules, $rule);
		my ($rnr, $rulenr);
		my (@trules, @install);
		my ($rb_simplified);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		@rules       = @{$obj};
	
		# Find Simplified mode ...
		@list=GetMembers(GetKey($CPtable));
		foreach $list (@list) {
			if (GetKey($list, 'collection:Name') eq $Policy) {
				$rb_simplified = (GetKey($list, "use_VPN_communities") eq 'true');
			}
		}
		
		# RULEBASE properties to HTML ...
		$template->param( RUL_SIMPLIFIED	=> $rb_simplified );
		
		#----------------------------------------------------------------
		#---                      RULES to HTML                       ---
		#----------------------------------------------------------------
		$rulenr=0;
		$rnr=-1;
		foreach $rule (@rules) {
			if (GetKey($rule, 'AdminInfo:ClassName') eq 'security_header_rule') {
				$rnr += 1;
				# Comment line in rulebase
				$trules[$rnr]{RUL_HEADER}	= 1;
				$trules[$rnr]{RUL_HDR_TEXT}	= stripquotes(GetKey($rule, 'header_text'));
	
			} elsif (GetKey($rule, "AdminInfo:ClassName" eq 'security_rule')) {
				# Actual rule
				$rnr += 1;
				$rulenr += 1;
				$trules[$rnr]{RUL_SIMPLIFIED} = $rb_simplified;
				$trules[$rnr]{RUL_HEADER}	= 0;
				$trules[$rnr]{RUL_NO}		= $rulenr;
				$trules[$rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');
	
				$trules[$rnr]{RUL_SOURCE}	= HTMLMemberList($Policy, GetKey($rule, 'src'), $_HREF);
				$trules[$rnr]{RUL_DEST}		= HTMLMemberList($Policy, GetKey($rule, 'dst'), $_HREF);
				$trules[$rnr]{RUL_VPN}    	= HTMLMemberList($Policy, GetKey($rule, 'through'), $_HREF);
				$trules[$rnr]{RUL_SERVICE}	= HTMLMemberList($Policy, GetKey($rule, 'services'), $_HREF);
				$trules[$rnr]{RUL_ACTION}	= HTMLMemberList($Policy, GetKey($rule, 'action'), $_HREF);
				$trules[$rnr]{RUL_TRACK}	= HTMLMemberList($Policy, GetKey($rule, 'track'), $_HREF);;
				$trules[$rnr]{RUL_WHEN}		= HTMLMemberList($Policy, GetKey($rule, 'time'), $_HREF);;
				@install=GetMembers(GetKey($rule, 'install'));
				if (GetKey($install[0], 'Name') eq 'Any') {
					$trules[$rnr]{RUL_INSTALL}	= [txt2html('Policy Targets', 'any_object', 'black')]
				} else {
					$trules[$rnr]{RUL_INSTALL}	= HTMLMemberList($Policy, GetKey($rule, 'install'), $_HREF);
				}
	
				$trules[$rnr]{RUL_COMMENTS}	= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
			}
		}
		$template->param(RULES=>\@trules);
		
		return $template;
	}
	
	
	##########################################################################
	# Routine: nat2html
	#
	# Description:
	# Routine to create the NAT rules section
	##########################################################################
	sub nat2html {
		use strict;
		my (@rules, $rule);
		my ($rnr, $rulenr);
		my (%fwtargets, $fw);
		my ($AllObjects);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		@rules       = @{$obj};
	
		# Determine if all objects need to be listed ... 
		$AllObjects = Option($Policy, 'ALLOBJECTS');
	
		# Determine installation targets ...
		if (lc(GetKey("policies_collections:".$Policy.":all_internal_modules")) eq 'true') {
			$fwtargets{All}=1;
		} else {
			$fwtargets{All}=0;
			foreach $fw (GetMembers(GetKey("policies_collections:".$Policy.":installable_targets"))) { 
				$fwtargets{GetKey($fw, "Name")}=1
			}
		}
	
		#----------------------------------------------------------------
		#---                    NATRULES to HTML                      ---
		#----------------------------------------------------------------
		$rulenr=0;
	
		# Insert NAT rules BEFORE
		my (@trules, @install);
	
		$rnr=-1;
		foreach $rule (@rules) {
			if (GetKey($rule, 'rule_block_number') eq '0') {
				$rnr += 1;
				$rulenr += 1;
				$trules[$rnr]{RUL_NO}		= $rulenr;
				$trules[$rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');
		
				$trules[$rnr]{ORG_SOURCE}		= HTMLMemberList($Policy, GetKey($rule, 'src_adtr'), $_HREF);
				$trules[$rnr]{ORG_DESTINATION}	= HTMLMemberList($Policy, GetKey($rule, 'dst_adtr'), $_HREF);
				$trules[$rnr]{ORG_SERVICE}		= HTMLMemberList($Policy, GetKey($rule, 'services_adtr'), $_HREF);
				if (GetKey($rule.':services_adtr', 'adtr_method') eq 'adtr_method_static') {
					$trules[$rnr]{TR_SOURCE}	= HTMLMemberList($Policy, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_STATIC);
				} else {
					$trules[$rnr]{TR_SOURCE}	= HTMLMemberList($Policy, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_HIDE);
				}
				$trules[$rnr]{TR_DESTINATION}	= HTMLMemberList($Policy, GetKey($rule, 'dst_adtr_translated'), $_HREF+$_NAT_STATIC);
				$trules[$rnr]{TR_SERVICE}		= HTMLMemberList($Policy, GetKey($rule, 'services_adtr_translated'), $_HREF+$_NAT_STATIC);
				@install=GetMembers(GetKey($rule, 'install'));
				if (GetKey($install[0], 'Name') eq 'Any') {
					$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
				} else {
					$trules[$rnr]{INSTALL}	    = HTMLMemberList($Policy, GetKey($rule, 'install'), $_HREF);
				}
				$trules[$rnr]{RUL_COMMENTS}		= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
			}
		}
		$template->param(NAT_RULES_FIRST=>\@trules);
			
		# Insert automatic NAT rules ...
		# Auto NAT rule order:
		# 1. first hosts(H) than networks(N)
		# 2. first static(S) than hide(H)
		# 3. alphabetical
	
		# Initialize tables for this section ...
		my @trules; 
		my (@HSlist, @HHlist, @NSlist, @NHlist);
		my ($obj, $table, $name);
	
		# Dynamic NAT rules can be stored in ALL network objects ...
		$table='network_objects';
	
		if (ref(GetKey($table))) {
			foreach $name (keys %{GetKey($table)}) {
				$obj=GetKey($table.':'.$name);
				if (GetKey($obj, 'add_adtr_rule') eq 'true') {
					if (	$AllObjects || (GetKey($obj, 'the_firewalling_obj:Name') eq 'All') ||
							$fwtargets{All} || $fwtargets{GetKey($obj, 'the_firewalling_obj:Name')} ) {
						if (GetKey($obj, 'netmask') ne '') {  # Network object
							if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_hide') {
								push @NHlist, $name;
							} else {
								push @NSlist, $name;
							}
						} else {                # Host object
							if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_hide') {
								push @HHlist, $name;
							} else {
								push @HSlist, $name;
							}
						}
					}
				}
			}
		}
		
		# Build automatic NAT rules
		$rnr=-1;
		foreach $name (sort(@HSlist), sort(@HHlist), sort(@NSlist), sort(@NHlist)) {
			# Host NAT ...
			$obj=GetKey($table, $name);
	
			if (GetKey($obj, 'netmask')) {
				# Network to Network: no NAT
				$rnr += 1;
				$rulenr += 1;
				$trules[$rnr]{RUL_NO}		= $rulenr;
	
				$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($Policy, $table, $name, $_HREF);
				$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($Policy, $table, $name, $_HREF);
				$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($Policy, 'globals', 'Any');
				$trules[$rnr]{TR_SOURCE}		= [txt2html('Original', 'Original', 'black')];
				$trules[$rnr]{TR_DESTINATION}	= [txt2html('Original', 'Original', 'black')];
				$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
				if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
					$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
				} else {
					$trules[$rnr]{INSTALL}	    = HTMLObjectList($Policy, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
				}
				$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);
			}
			
			# From inside to outside NAT ...
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_NO} = $rulenr;
	
			$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($Policy, $table, $name, $_HREF);
			$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($Policy, 'globals', 'Any');
			$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($Policy, 'globals', 'Any');
			$trules[$rnr]{TR_SOURCE}		= HTMLObjectList($Policy, $table, $name, ($_NAT+$_HREF));
			$trules[$rnr]{TR_DESTINATION}	= [txt2html('Original', 'Original', 'black')];
			$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
			if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
				$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{INSTALL}	    = HTMLObjectList($Policy, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
			}
			$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);
	
			if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
				# Static NAT only: From outside to inside NAT ...
				$rnr += 1;
				$rulenr += 1;
				$trules[$rnr]{RUL_NO}		= $rulenr;
	
				$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($Policy, 'globals', 'Any');
				$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($Policy, $table, $name, ($_NAT+$_HREF));
				$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($Policy, 'globals', 'Any');
				$trules[$rnr]{TR_SOURCE}		= [txt2html('Original', 'Original', 'black')];
				$trules[$rnr]{TR_DESTINATION}	= HTMLObjectList($Policy, $table, $name, $_HREF);
				$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
				if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
					$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
				} else {
					$trules[$rnr]{INSTALL}	    = HTMLObjectList($Policy, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
				}
				$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);
			}
		}
	
		$template->param(NAT_AUTO_RULES=>\@trules);
	
		# Insert NAT rules LAST
	
		# Initialize tables for this section ...
		my (@trules); 
	
		$rnr=-1;
		foreach $rule (@rules) {
			if (GetKey($rule, 'rule_block_number') eq '1') {
				$rnr += 1;
				$rulenr += 1;
				$trules[$rnr]{RUL_NO}		= $rulenr;
		
				$trules[$rnr]{ORG_SOURCE}		= HTMLMemberList($Policy, GetKey($rule, 'src_adtr'), $_HREF);
				$trules[$rnr]{ORG_DESTINATION}	= HTMLMemberList($Policy, GetKey($rule, 'dst_adtr'), $_HREF);
				$trules[$rnr]{ORG_SERVICE}		= HTMLMemberList($Policy, GetKey($rule, 'services_adtr'), $_HREF);
				if (GetKey($rule.':services_adtr', 'adtr_method') eq 'adtr_method_static') {
					$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($Policy, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_STATIC);
				} else {
					$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($Policy, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_HIDE);
				}
				$trules[$rnr]{TR_DESTINATION}	= HTMLMemberList($Policy, GetKey($rule, 'dst_adtr_translated'), $_HREF+$_NAT_STATIC);
				$trules[$rnr]{TR_SERVICE}		= HTMLMemberList($Policy, GetKey($rule, 'services_adtr_translated'), $_HREF+$_NAT_STATIC);
				@install=GetMembers(GetKey($rule, 'install'));
				if (GetKey($install[0], 'Name') eq 'Any') {
					$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
				} else {
					$trules[$rnr]{INSTALL}	    = HTMLMemberList($Policy, GetKey($rule, 'install'), $_HREF);
				}
				$trules[$rnr]{RUL_COMMENTS}		= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
			}
		}
		$template->param(NAT_RULES_LAST=>\@trules);
	}
	
	
	##########################################################################
	# Routine: desktop2html
	#
	# Description:
	# Routine to create the desktop security section
	##########################################################################
	sub desktop2html {
		use strict;
		my (@rules, $rule);
		my ($in_rnr, $out_rnr, $rulenr);
		my (@in_rules, @out_rules, @install);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		@rules       = @{$obj};
	
		#----------------------------------------------------------------
		#---            DESKTOP SECURITY RULES to HTML                ---
		#----------------------------------------------------------------
		$rulenr=0;
		$in_rnr=-1;
		$out_rnr=-1;
		foreach $rule (@rules) {
			if (GetKey((GetMembers(GetKey($rule, 'install')))[0], 'Name') eq 'Dst') {
				# INBOUND rule
				$rulenr += 1;
				$in_rnr += 1;
				$in_rules[$in_rnr]{RUL_NO}		= $rulenr;
				$in_rules[$in_rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');
	
				$in_rules[$in_rnr]{RUL_SOURCE}		= HTMLMemberList($Policy, GetKey($rule, 'src'), $_HREF);
				$in_rules[$in_rnr]{RUL_DESKTOP}	= HTMLMemberList($Policy, GetKey($rule, 'dst'), $_HREF);
				$in_rules[$in_rnr]{RUL_SERVICE}	= HTMLMemberList($Policy, GetKey($rule, 'services'), $_HREF);
				$in_rules[$in_rnr]{RUL_ACTION}		= HTMLMemberList($Policy, GetKey($rule, 'action'), $_HREF);
				$in_rules[$in_rnr]{RUL_TRACK}		= HTMLMemberList($Policy, GetKey($rule, 'track'), $_HREF);;
	
				$in_rules[$in_rnr]{RUL_COMMENTS}	= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
			} else {
				# OUTBOUND rule
				$rulenr += 1;
				$out_rnr += 1;
				$out_rules[$out_rnr]{RUL_NO}		= $rulenr;
				$out_rules[$out_rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');
	
				$out_rules[$out_rnr]{RUL_DESKTOP}	= HTMLMemberList($Policy, GetKey($rule, 'src'), $_HREF);
				$out_rules[$out_rnr]{RUL_DEST}		= HTMLMemberList($Policy, GetKey($rule, 'dst'), $_HREF);
				$out_rules[$out_rnr]{RUL_SERVICE}	= HTMLMemberList($Policy, GetKey($rule, 'services'), $_HREF);
				$out_rules[$out_rnr]{RUL_ACTION}	= HTMLMemberList($Policy, GetKey($rule, 'action'), $_HREF);
				$out_rules[$out_rnr]{RUL_TRACK}		= HTMLMemberList($Policy, GetKey($rule, 'track'), $_HREF);;
	
				$out_rules[$out_rnr]{RUL_COMMENTS}	= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
			}
		}
		$template->param(IN_RULES =>\@in_rules, OUT_RULES=>\@out_rules);
		
		return $template;
	}
	
	
	##########################################################################
	# Routine: nodes2html
	#
	# Description:
	# Routine to create the NODES table
	##########################################################################
	sub nodes2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $section);
		my ($var, %var, $val, $type);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                    NODES to HTML                         ---
		#----------------------------------------------------------------
		$cnt = -1;
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable, $name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
			
			#------------------------------------------------
			# TOPOLOGY tab of Nodes
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Nodes'), 'Nodes', 'Topology');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_PROP}}, Topology($Policy, $obj, $name, $type, $section);
			}
			
			#------------------------------------------------
			# NAT tab of Nodes
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Nodes'), 'Nodes', 'NAT');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_PROP}}, NAT($Policy, $obj, $name, $type, $section);
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_PROP}}) { push @{$tobj[$cnt]{OBJ_PROP}}, { VOID => 1 } }
	
			#------------------------------------------------
			# ADVANCED column of Nodes
			#------------------------------------------------
			# Web Server configuration
			my @section;
			if (GetKey($obj, 'additional_products:is_web_server') eq 'true') {
				$section = Section($Policy, Object2Page($Policy, 'Nodes'), 'Nodes', 'Web Server');
				if ($section) {
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{ 	SF_BEG => [{	SECTION  => html('Web Server'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{ SUBSECT_BEG => html('Ports configurations'),
						  S1 =>	[{ 	CP3 => 1, 
									OBJ_VAL1 => (GetKey($obj, 'additional_products:web_server_prop:web_server_ports:0:has_standard_port') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Server uses Standard Port:'),
									OBJ_VAL3 => html('80') }],
						  SUBSECT_END => 1 }
					);
									
					%var = %{obj2html($Policy, GetKey($obj, 'additional_products:web_server_prop:protected_by_obj:Table'), GetKey($obj, 'additional_products:web_server_prop:protected_by_obj:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{ SUBSECT_BEG => html('Protection'),
						  S1 =>	[{ 	CP2 => 1,
									OBJ_VAL1 => html('Protected by:'),
									OBJ_VAL2 => $var }],
						  SUBSECT_END => 1 }
					);
		
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{ SUBSECT_BEG => html('Attacks defense'),
						  S1 => [{	OBJ_VAL1 => (GetKey($obj, 'additional_products:web_server_prop:cross_sites_scripting') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Cross Site Scripting:') }] }
					);
					$var = 'OFF';
					if (GetKey($obj, 'additional_products:web_server_prop:cross_sites_scripting') eq 'true') {
						$val = GetKey($obj, 'additional_products:web_server_prop:strip');
						push @{$tobj[$cnt]{OBJ_ADV}}, (
							{ S1 => [{	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => ($val eq 'strip_tags')?$MarkDot:$Mark,
										OBJ_VAL3 => html('Block SCRIPT Tags') },
									 {	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => ($val eq 'strip_HTML')?$MarkDot:$Mark,
										OBJ_VAL3 => html('Block all HTML Tags') },
									 {	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => ($val eq 'strip_all_tags')?$MarkDot:$Mark,
										OBJ_VAL3 => html('Block all Tags') }] }
						);
										
					}
					push @{$tobj[$cnt]{OBJ_ADV}}, { SUBSECT_END => 1 };
	
					push @{$tobj[$cnt]{OBJ_ADV}}, { SF_END => 1 };
				}
			}
					
			#------------------------------------------------
			# FireWall-1 GX tab of Node
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Nodes'), 'Nodes', 'Firewall-1 GX');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, Firewall1_GX($Policy, $obj, $name, $type, $section);
			}
			
			#------------------------------------------------
			# SNMP tab of Node
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Nodes'), 'Nodes', 'Advanced');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, Advanced($Policy, $obj, $name, $type, $section);
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ADV}}) { push @{$tobj[$cnt]{OBJ_ADV}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENT column
			#------------------------------------------------
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(NODES=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: interoperable2html
	#
	# Description:
	# Routine to create the INTEROPERABLE table
	##########################################################################
	sub interoperable2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $section);
		my ($var, %var, $val, $type);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                INTEROPERABLE to HTML                     ---
		#----------------------------------------------------------------
		$cnt = -1;
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
			
			#------------------------------------------------
			# TOPOLOGY tab/column of Interoperable Devices
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Interoperable'), 'Interoperable', 'Topology');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_PROP}}, Topology($Policy, $obj, $name, $type, $section);
			}
			
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_PROP}}) { push @{$tobj[$cnt]{OBJ_PROP}}, { VOID => 1 } }
	
			#------------------------------------------------
			# VPN tab of Interoperable Devices
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Interoperable'), 'Interoperable', 'VPN');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, VPN($Policy, $obj, $name, $type, $section);
			}
			
			#------------------------------------------------
			# FireWall-1 GX tab Interoperable Devices
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Interoperable'), 'Interoperable', 'Firewall-1 GX');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, Firewall1_GX($Policy, $obj, $name, $type, $section);
			}
			
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ADV}}) { push @{$tobj[$cnt]{OBJ_ADV}}, { VOID => 1 } }
		}
		$template->param(INTEROPERABLE=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: networks2html
	#
	# Description:
	# Routine to create the NETWORKS table
	##########################################################################
	sub networks2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name, $section);
		my ($type, $current_type, $var, %var);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                   NETWORKS to HTML                       ---
		#----------------------------------------------------------------
		$cnt   = -1;
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
			$tobj[$cnt]{OBJ_NETMASK}    = GetKey($obj, 'netmask');
	
			#------------------------------------------------
			# Advanced column
			#------------------------------------------------
			# NAT configuration
			if (GetKey($obj, 'add_adtr_rule') eq 'true') {
				$section = Section($Policy, Object2Page($Policy, 'Networks'), 'Networks', 'NAT');
				if ($section) {
					push @{$tobj[$cnt]{OBJ_NAT}}, {
						SF_BEG => [{	SECTION  => html('NAT'),
										OBJ_NAME => $name,
										STATE	 => ($section eq $_OPEN) }]
					};
					if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
						push @{$tobj[$cnt]{OBJ_NAT}}, (
							{ S1 => 	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Translation method:'), 
											OBJ_VAL2 => html('Static') },
										 { 	CP2 => 1, 
											OBJ_VAL1 => html('Translate to IP address:'), 
											OBJ_VAL2 => GetKey($obj, 'valid_ipaddr') }] }
						);
					} else {
						push @{$tobj[$cnt]{OBJ_NAT}}, (
							{ S1 => 	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Translation method:'), 
											OBJ_VAL2 => html('Hide') }] },
							{ S1 => 	[{ 	OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') eq '0.0.0.0')?$MarkDot:$Mark, 
											OBJ_VAL2 => html('Hide behind Gateway'),
											OBJ_VAL3 => $Spacer },
										 { 	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?$MarkDot:$Mark, 
											OBJ_VAL2 => html('Hide behind IP address'),
											OBJ_VAL3 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?html(GetKey($obj, 'valid_ipaddr')):$Spacer }] }
						);
					}
					%var = %{obj2html($Policy, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_NAT}}, (
						{ S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('Install on Gateway:'), 
										OBJ_VAL2 => $var }] }
					);
					push @{$tobj[$cnt]{OBJ_NAT}}, { SF_END => 1 };
				}            
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_NAT}}) { push @{$tobj[$cnt]{OBJ_NAT}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENT column
			#------------------------------------------------
			$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(NETWORKS=>\@tobj);
	}
	
	##########################################################################
	# Routine: domains2html
	#
	# Description:
	# Routine to create the DOMAINS table
	##########################################################################
	sub domains2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type);
		my (%var, $var);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                     DOMAINS to HTML                      ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}     = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}     = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(DOMAINS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: accesspoints2html
	#
	# Description:
	# Routine to create the ACCESSPOINTS table
	##########################################################################
	sub accesspoints2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $sct);
		my (%var, $var, $section);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                 ACCESSPOINTS to HTML                     ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			$sct = 0;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_APN}  = GetKey($obj, 'apn');
	
			#------------------------------------------------
			# ENDUSERDOMAIN column
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'AccessPoints'), 'AccessPoints', 'Intra Tunnel');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, {
					SF_BEG => [{	SECTION  => html('Intra Tunnel Policy Enforcement'),
									OBJ_NAME => $name,
									STATE	 => ($section eq $_OPEN) }]
				};
	
				push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, { 
					S1 => [{	OBJ_VAL1 => (GetKey($obj, 'enforce_domain') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Enforce End User Domain') }]
				};
				if (GetKey($obj, 'enforce_domain') eq 'true') {
					%var = %{obj2html($Policy, GetKey($obj, 'domain:Table'), GetKey($obj, 'domain:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, (
						{ S1 =>	[{	CP2 => 1, 
									OBJ_VAL1 => html('End User Domain:'), 
									OBJ_VAL2 => $var }] },
						{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'enforce_other_apns') eq 'true')?$CheckTick:$Check, 
									OBJ_VAL2 => html('Block MS to MS traffic between this and other APN End User domains') },
								 {	OBJ_VAL1 => (GetKey($obj, 'enforce_this_apn') eq 'true')?$CheckTick:$Check, 
									OBJ_VAL2 => html('Block MS to MS traffic within this End User domain') }] }
					);
				}			
				push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, { SF_END => 1 };
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}) { push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, { VOID => 1 } }
			
			#------------------------------------------------
			# COMMENT column
			#------------------------------------------------
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(ACCESSPOINTS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: osedevices2html
	#
	# Description:
	# Routine to create the OSEDEVICES table
	##########################################################################
	sub osedevices2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $section);
		my ($var, %var, @var, $val);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                  OSEDEVICES to HTML                      ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
			
			$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_IPADDRESS} = GetKey($obj, 'ipaddr');
			
			#------------------------------------------------
			# TOPOLOGY tab/column of OSE Device
			#------------------------------------------------
			push @{$tobj[$cnt]{OBJ_PROP}}, Topology($Policy, $obj, $name, $type);
			
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_PROP}}) { push @{$tobj[$cnt]{OBJ_PROP}}, { VOID => 1 } }
	
			#------------------------------------------------
			# SNMP tab of OSE Device
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'OSEDevices'), 'OSEDevices', 'SNMP');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, Advanced($Policy, $obj, $name, $type);
			}
				
			#------------------------------------------------
			# SETUP column/tab of OSE Device
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'OSEDevices'), 'OSEDevices', 'Setup');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_ADV}}, {
					SF_BEG => [{	SECTION  => html('Setup'),
									OBJ_NAME => $name,
									STATE	 => ($section eq $_OPEN) }]
				};
				if ($type =~ /Cisco/i) {
					push @{$tobj[$cnt]{OBJ_ADV}}, { 
						SUBSECT_BEG => html('OSE Device Access'), 
						S1 => 	[{	CP2 => 1,
									OBJ_VAL1 => html('Access List No.:'),
									OBJ_VAL2 => number(GetKey($obj, 'alnum')) },
								 {	CP2 => 1,
									OBJ_VAL1 => html('Version:'),
									OBJ_VAL2 => number(GetKey($obj, 'version')) }],
						SUBSECT_END => 1
					};
				} elsif ($type =~ /Nortel/i) {
					push @{$tobj[$cnt]{OBJ_ADV}}, { 
						SUBSECT_BEG => html('OSE Device Access'), 
						S1 => 	[{	CP2 => 1,
									OBJ_VAL1 => html('Volume:'),
									OBJ_VAL2 => number(GetKey($obj, 'slot_num')) },
								 {	CP2 => 1,
									OBJ_VAL1 => html('Config File:'),
									OBJ_VAL2 => html(GetKey($obj, 'config_file')) },
								 {	CP2 => 1,
									OBJ_VAL1 => html('Version:'),
									OBJ_VAL2 => number(GetKey($obj, 'version')) }],
						SUBSECT_END => 1
					};
				}
		
				if ($type =~ /Cisco|3Com/i) {
					push @{$tobj[$cnt]{OBJ_ADV}}, { 
						SUBSECT_BEG => html('Interface Directions'), 
						S1 => 	[{	CP2 => 1,
									OBJ_VAL1 => html('Rules:'),
									OBJ_VAL2 => $IntDirect{GetKey($obj, 'rules_dir')} },
								 {	CP2 => 1,
									OBJ_VAL1 => html('Spoof rules:'),
									OBJ_VAL2 => $IntDirect{GetKey($obj, 'spoof_dir')} }],
						SUBSECT_END => 1
					};
				}
			
				if ($type =~ /3Com/i) {
					push @{$tobj[$cnt]{OBJ_ADV}}, { 
						SUBSECT_BEG => html('Service Independent Filters'), 
						S1 => 	[{	OBJ_VAL1 => (GetKey($obj, 'deny_record_route') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny Route Recording') },
								 {	OBJ_VAL1 => (GetKey($obj, 'deny_src_route') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny Src Routing') },
								 {	OBJ_VAL1 => (GetKey($obj, 'deny_tiny_fragment') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny Tiny Fragments') },
								 {	OBJ_VAL1 => (GetKey($obj, 'deny_time_stamp') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny Time Stamping') },
								 {	OBJ_VAL1 => (GetKey($obj, 'deny_ip_tunnel') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny IP Tunnels') },
								 {	OBJ_VAL1 => (GetKey($obj, 'deny_src_spoofing') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Deny Src Spoofing (3Com)') },
								 {	OBJ_VAL1 => (GetKey($obj, 'generate_icmp') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Generate ICMP Errors') }],
						SUBSECT_END => 1
					};
				}
				push @{$tobj[$cnt]{OBJ_ADV}}, { SF_END => 1 };
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ADV}}) { push @{$tobj[$cnt]{OBJ_ADV}}, { VOID => 1 } }
	
	
			#------------------------------------------------
			# COMMENT column
			#------------------------------------------------
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(OSEDEVICES=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: objectgroups2html
	#
	# Description:
	# Routine to create the ObjectGROUPS table
	##########################################################################
	sub objectgroups2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                    GROUPS to HTML                        ---
		#----------------------------------------------------------------
		$cnt   = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
			
			$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
			if ($type =~ /Simple/i) {
				$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($Policy, $obj, $_HREF);
				$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
			} elsif ($type =~ /Exclusion/i) {
				$tobj[$cnt]{OBJ_MEMBERS}    = HTMLObjectList($Policy, GetKey($obj, 'base:Table'), GetKey($obj, 'base:Name'), $_HREF);
				$tobj[$cnt]{OBJ_EXCLUSION}  = HTMLObjectList($Policy, GetKey($obj, 'exception:Table'), GetKey($obj, 'exception:Name'), $_HREF);
			} elsif ($type =~ /User Authority/i) {
				$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($Policy, GetKey($obj, 'members'), $_HREF);
				$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
			} elsif ($type =~ /GSN Handover/i) {
				$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($Policy, $obj, $_HREF);
				$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
			}
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(OBJECTGROUPS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: logicalservers2html
	#
	# Description:
	# Routine to create the LOGICALSERVERS table
	##########################################################################
	sub logicalservers2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name, $section);
		my ($type, $current_type);
		my ($var, $val, %var);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                LOGICALSERVERS to HTML                    ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
	
			#------------------------------------------------
			# GENERAL tab/column of LogicalServers
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'LogicalServers'), 'LogicalServers', 'Server');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_GENERAL}}, {
					SF_BEG => [{	SECTION  => html('Server'),
									OBJ_NAME => $name,
									STATE	 => ($section eq $_OPEN) }]
				};
				%var = %{obj2html($Policy, GetKey($obj, 'servers:Table'), GetKey($obj, 'servers:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @{$tobj[$cnt]{OBJ_GENERAL}}, (
					{ S1 =>	[{	OBJ_VAL1 => html("Server's type:"),
								OBJ_VAL2 => (GetKey($obj, 'servers_type') eq 'http')?$MarkDot:$Mark,
								OBJ_VAL3 => html('HTTP') },
							 {	C1 => 1,
								OBJ_VAL2 => (GetKey($obj, 'servers_type') eq 'other')?$MarkDot:$Mark,
								OBJ_VAL3 => html('Other') }] },
					{ S1 =>	[{	CP2 => 1,
								OBJ_VAL1 => html('Servers group:'),
								OBJ_VAL2 =>  $var }] }
				);
				if (GetKey($obj, 'use_cache') eq 'true') {
					push @{$tobj[$cnt]{OBJ_GENERAL}}, { 
						S1 => 	[{	OBJ_VAL1 => $CheckTick,
									OBJ_VAL2 => html('Persistent server mode:'),
									OBJ_VAL3 => (GetKey($obj, 'cache_type') eq 'by_service')?$MarkDot:$Mark,
									OBJ_VAL4 => html('Persistency by service') },
								 {	C1 => 2,
									OBJ_VAL3 => (GetKey($obj, 'cache_type') eq 'by_server')?$MarkDot:$Mark,
									OBJ_VAL4 => html('Persistency by server') }]
					};
				} else {
					push @{$tobj[$cnt]{OBJ_GENERAL}}, { 
						S1 => 	[{	OBJ_VAL1 => $Check,
									OBJ_VAL2 => html('Persistent server mode') }]
					};
				}
		
				push @{$tobj[$cnt]{OBJ_GENERAL}}, { 
					SUBSECT_BEG => html('Balance Method:'),
					S1 => 	[{	OBJ_VAL1 => ((GetKey($obj, 'method') eq '"server load"')?$MarkDot:$Mark).
											html(' Server Load'),
								OBJ_VAL2 => ((GetKey($obj, 'method') eq '"round trip"')?$MarkDot:$Mark).
											html(' Round Trip'),
								OBJ_VAL3 => ((GetKey($obj, 'method') eq '"round robin"')?$MarkDot:$Mark).
											html(' Round Robin') },
							 {	OBJ_VAL1 => ((GetKey($obj, 'method') eq 'random')?$MarkDot:$Mark).
											html(' Random'),
								OBJ_VAL2 => ((GetKey($obj, 'method') eq 'domain')?$MarkDot:$Mark).
											html(' Domain') }],
					SUBSECT_END => 1
				};
			}
			
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(LOGICALSERVERS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: addressranges2html
	#
	# Description:
	# Routine to create the ADDRESSRANGES table
	##########################################################################
	sub addressranges2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $section);
		my (%var, $var);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                ADDRESSRANGES to HTML                    ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}    = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}    = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_FIRSTIP} = GetKey($obj, 'ipaddr_first');
			$tobj[$cnt]{OBJ_LASTIP}  = GetKey($obj, 'ipaddr_last');
	
			# NAT configuration
			if (GetKey($obj, 'add_adtr_rule') eq 'true') {
				$section = Section($Policy, Object2Page($Policy, 'AddressRanges'), 'AddressRanges', 'NAT');
				if ($section) {
					push @{$tobj[$cnt]{OBJ_ADV}}, {
						SF_BEG => [{	SECTION  => html('NAT'),
										OBJ_NAME => $name,
										STATE	 => ($section eq $_OPEN) }]
					};
					if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
						push @{$tobj[$cnt]{OBJ_ADV}}, (
							{ S1 => 	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Translation method:'), 
											OBJ_VAL2 => html('Static') },
										 { 	CP2 => 1, 
											OBJ_VAL1 => html('Translate to IP address:'), 
											OBJ_VAL2 => GetKey($obj, 'valid_ipaddr') }] }
						);
					} else {
						push @{$tobj[$cnt]{OBJ_ADV}}, (
							{ S1 => 	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Translation method:'), 
											OBJ_VAL2 => html('Hide') }] },
							{ S1 => 	[{ 	OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') eq '0.0.0.0')?$MarkDot:$Mark, 
											OBJ_VAL2 => html('Hide behind Gateway'),
											OBJ_VAL3 => $Spacer },
										 { 	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?$MarkDot:$Mark, 
											OBJ_VAL2 => html('Hide behind IP address'),
											OBJ_VAL3 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?html(GetKey($obj, 'valid_ipaddr')):$Spacer }] }
						);
					}
					%var = %{obj2html($Policy, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{ S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('Install on Gateway:'), 
										OBJ_VAL2 => $var }] }
					);
				}            
				push @{$tobj[$cnt]{OBJ_ADV}}, { SF_END => 1 };
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ADV}}) { push @{$tobj[$cnt]{OBJ_ADV}}, { VOID => 1 } }
	
			$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(ADDRESSRANGES=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: dynamicobjects2html
	#
	# Description:
	# Routine to create the DYNAMICOBJECTS table
	##########################################################################
	sub dynamicobjects2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                DYNAMICOBJECTS to HTML                    ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{OBJ_TEXT}     = stripquotes($name);
			$tobj[$cnt]{OBJ_NAME}     = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(DYNAMICOBJECTS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: voipdomains2html
	#
	# Description:
	# Routine to create the VOIPDOMAINS table
	##########################################################################
	sub voipdomains2html {
		use strict;
		my (@ResourceList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $spec, $section);
	
		my (@var, %var, $var, $val);
		my (@tmp, $none);
		my ($Usage, $share, $server);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                  VOIPDOMAINS to HTML                     ---
		#----------------------------------------------------------------
		$cnt  = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable.':'.$name);
			$spec = GetKey($obj, 'spec_select') || GetKey($obj, 'resource_with_cvp');
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# GENERAL tab/column in VoIP Domain
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'VoIPDomains'), 'VoIPDomains', 'General');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_GENERAL}}, {
					SF_BEG => [{	SECTION  => html('General'),
									OBJ_NAME => $name,
									STATE	 => ($section eq $_OPEN) }]
				};
	
				my @section;
				%var = %{obj2html($Policy, GetKey($obj, 'zone_range:Table'), GetKey($obj, 'zone_range:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { 	CP2 => 1, 
									OBJ_VAL1 => html('Related endpoints domain:'), 
									OBJ_VAL2 => $var };
		
				%var = %{obj2html($Policy, GetKey($obj, 'server:Table'), GetKey($obj, 'server:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { 	CP2 => 1, 
									OBJ_VAL1 => html('VoIP installed at:'), 
									OBJ_VAL2 => $var };
		
				push @{$tobj[$cnt]{OBJ_GENERAL}}, { S1 => \@section };
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_GENERAL}}) { push @{$tobj[$cnt]{OBJ_GENERAL}}, { VOID => 1 } }
	
			#------------------------------------------------
			# ROUTING Mode tab/column in VoIP Domain
			#------------------------------------------------
			if ($type =~ /H.323/) {
				$section = Section($Policy, Object2Page($Policy, 'VoIPDomains'), 'VoIPDomains', 'Routing Mode');
				if ($section) {
					my @section;
					if ($type =~ /Gatekeeper/) {
						@var = GetMembers(GetKey($obj, 'h323_gatekeeper_protocols'));
					} elsif ($type =~ /Gateway/) {
						@var = GetMembers(GetKey($obj, 'h323_gateway_protocols'));
					}
					$val = 0;
					foreach $var (@var) { 
						if ($var eq 'H.225')    { $val += 1 }
						if ($var eq 'H.245')    { $val += 2 }
						if ($var eq '"RTP/RTCP"') { $val += 4 }
					}
					if ($type =~ /Gatekeeper/) {
						push @section, { 	OBJ_VAL1 => ($val & 1)?$CheckTick:$Check, 
											OBJ_VAL2 => html('Direct') };
					}
					push @section, 	{ 	OBJ_VAL1 => ($val & 2)?$CheckTick:$Check, 
										OBJ_VAL2 => html('Call Setup (Q.931)') },
									{ 	OBJ_VAL1 => ($val & 4)?$CheckTick:$Check, 
										OBJ_VAL2 => html('Call Setup (Q.931) and Call Control (H.245)') };
		
					push @{$tobj[$cnt]{OBJ_ROUTING}}, {
						SF_BEG => [{	SECTION  => html('Routing Mode'),
										OBJ_NAME => $name,
										STATE	 => ($section eq $_OPEN) }],
						S1 => \@section,
						SF_END => 1
					};
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OBJ_ROUTING}}) { push @{$tobj[$cnt]{OBJ_ROUTING}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENTS tab/column in VoIP Domain
			#------------------------------------------------
			$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(VOIPDOMAINS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: services2html
	#
	# Description:
	# Routine to create the SERVICES table
	##########################################################################
	sub services2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $var, $section);
		my (%var, $var, $val, $proto);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                   SERVICES to HTML                       ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{SVC_TEXT}       = stripquotes($name);
			$tobj[$cnt]{SVC_NAME}       = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# POLICY LOAD column of the SERVICES
			#------------------------------------------------
			if ($type ne 'GTP' && $type ne 'Citrix TCP') {
				push @{$tobj[$cnt]{SVC_KEEP}}, { 
					S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'reload_proof') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Keep') }] };
			}
	
			#------------------------------------------------
			# SPECIFICATION column of the SERVICES
			#------------------------------------------------
			$tobj[$cnt]{SVC_DCERPC} = ($type eq 'DCE-RPC');
				
			if ($type eq 'TCP' || $type eq 'UDP') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'port')) }] };
			} elsif ($type eq 'Compound TCP') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'port')) },
							 {	CP2 => 1, 
								OBJ_VAL1 => html('Compound Service:'), 
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'svc_type'))) }] };
			} elsif ($type eq 'Citrix TCP') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'port')) }] };
			} elsif ($type eq 'RPC') {
				push @{$tobj[$cnt]{SVC_SPEC}}, {
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Program Number:'), 
								OBJ_VAL2 => number(GetKey($obj, 'port')) }] };
			} elsif ($type eq 'ICMP') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Type:'), 
								OBJ_VAL2 => number(GetKey($obj, 'icmp_type')) },
							 {	CP2 => 1, 
								OBJ_VAL1 => html('Code:'), 
								OBJ_VAL2 => number(GetKey($obj, 'icmp_code')) }] };
			} elsif ($type eq 'Other') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('IP Protocol:'), 
								OBJ_VAL2 => number(GetKey($obj, 'protocol')) }] };
			} elsif ($type eq 'DCE-RPC') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('Interface UUID:'), 
								OBJ_VAL2 => stripquotes(GetKey($obj, 'uuid')) }] };
			} elsif ($type eq 'GTP') {
				push @{$tobj[$cnt]{SVC_SPEC}}, { 
					S1 => 	[{	CP2 => 1, 
								OBJ_VAL1 => html('GTP version:'), 
								OBJ_VAL2 => (GetKey($obj, 'type') eq 'gtp')?'v0':'v1' },
							 {	CP2 => 1, INACTIVE => (GetKey($obj, 'type') ne 'gtp'),
								OBJ_VAL1 => html('Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'port')) },
							 {	CP2 => 1, INACTIVE => (GetKey($obj, 'type') eq 'gtp'),
								OBJ_VAL1 => html('Control Plane Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'control_port')) },
							 {	CP2 => 1, INACTIVE => (GetKey($obj, 'type') eq 'gtp'),
								OBJ_VAL1 => html('User Plane Port:'), 
								OBJ_VAL2 => number(GetKey($obj, 'user_port')) }] };
			}
	
			#------------------------------------------------
			# ADVANCED settings of SERVICES
			#------------------------------------------------
			if ($type eq 'TCP' || $type eq 'UDP' || $type eq 'Other' || $type eq 'GTP') {
				# These services do have an Advanced section ...
				$section = Section($Policy, Object2Page($Policy, 'Services'), 'Services', 'Advanced');
				if ($section) {
					push @{$tobj[$cnt]{SVC_ADV}}, (
						{ 	SF_BEG => 	[{	SECTION  => html('Advanced'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }] }
					);
			
					if ($type eq 'TCP' || $type eq 'UDP' || $type eq 'Other') {
						
						%var = %{obj2html($Policy, GetKey($obj, 'proto_type:Table'), GetKey($obj, 'proto_type:Name'), $_HREF)};
						$proto = GetKey($obj, 'proto_type')?$var{OBJ_TEXT}:'';
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	CP2 => 1, INACTIVE => ($type ne 'Other'),
											OBJ_VAL1 => html('Match:'),
											OBJ_VAL2 => html(GetKey($obj, 'exp'))||'-' },
										 {	CP2 => 1, INACTIVE => ($type eq 'Other'),
											OBJ_VAL1 => html('Source port:'),
											OBJ_VAL2 => number(GetKey($obj, 'src_port')) },
										 {	CP2 => 1,
											OBJ_VAL1 => html('Protocol Type:'),
											OBJ_VAL2 => $proto||'-' },
										 {	OBJ_VAL1 => chr(0) }] }
						);
					}
	
					# Enable for TCP resource ...
					if ($type eq 'TCP') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'enable_tcp_resource') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Enable for TCP resource') }] }
						);
					}
					# Accept Replies ...
					if ($type eq 'UDP' || $type eq 'Other') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'replies') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Accept Replies') }] }
						);
					}
					# Accept replies from any port ...
					if ($type eq 'UDP') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	INACTIVE => (GetKey($obj, 'replies') eq 'false'),
											OBJ_VAL1 => (GetKey($obj, 'replies_from_any_port') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Accept replies from any port') }] }
						);
					}
					# Match for 'Any'...
					if ($type eq 'TCP' || $type eq 'UDP' || $type eq 'Other') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'include_in_any') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html("Match for 'Any'") }] }
						);
					}
	
					# Session Timeout ...
					if ($type eq 'TCP') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	CP2 => 1,
											OBJ_VAL1 => html('Session Timeout:'),
											OBJ_VAL2 => number(GetKey($obj, 'timeout')),
											OBJ_VAL3 => html('seconds') }] }
						);
					}
					# Virtual Session Timeout ...
					if ($type eq 'UDP' || $type eq 'Other') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	CP2 => 1,
											OBJ_VAL1 => html('Virtual Session Timeout:'),
											OBJ_VAL2 => number(GetKey($obj, 'timeout')),
											OBJ_VAL3 => html('seconds') }] }
						);
					}
					# Synchronize connections on Cluster ...
					if ($type eq 'TCP' || $type eq 'UDP' || $type eq 'Other') {
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'sync_on_cluster') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Synchronize connections on Cluster') }] }
						);
					}
					# Start synchronizing after x seconds after connection initiation ...
					if ($type eq 'TCP') {
						if ((GetKey($obj, 'sync_on_cluster') eq 'true') &&
							($proto eq '-' || $proto =~ /HTTP/ || $proto =~ /None/)) {
							push @{$tobj[$cnt]{SVC_ADV}}, (
								{	S1	=>	[{	CP3 => 1,
												OBJ_VAL1 => (GetKey($obj, 'use_delayed_sync') eq 'true')?$CheckTick:$Check,
												OBJ_VAL2 => html('Start synchronizing'),
												OBJ_VAL3 => number(GetKey($obj, 'delayed_sync_value')),
												OBJ_VAL4 => html('seconds after connection initiation') }] }
							);
						}
					}
	
					if ($type eq 'GTP') {
						# GTP devices
						%var = %{obj2html($Policy, GetKey($obj, 'apn_obj:Table'), GetKey($obj, 'apn_obj:Name'), $_HREF)};
						$var = ($var{OBJ_TEXT})?$var{OBJ_ICON}.' '.$var{OBJ_TEXT}:'';
						push @{$tobj[$cnt]{SVC_ADV}}, (
							{	SUBSECT_BEG	=>	html('Allowed PDP Contents') },
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'imsi_any') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Any') },
										 {	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'imsi_any') eq 'false')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Allow specific IMSI Prefix:'),
											OBJ_VAL3 =>	number(GetKey($obj, 'imsi')) },
										 {	OBJ_VAL1 => '' },
										 {	OBJ_VAL1 => (GetKey($obj, 'apn_any') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Any') },
										 {	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'apn_any') eq 'false')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Allow specific Access Point Names:'),
											OBJ_VAL3 =>	$var||'-' },
										 {	OBJ_VAL1 => '' },
										 {	OBJ_VAL1 => (GetKey($obj, 'sel_mode_any') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Any') },
										 {	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'sel_mode_any') eq 'false')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Allow specific Access Point Names:'),
											OBJ_VAL3 =>	$APN_selmode{GetKey($obj, 'sel_mode')} },
										 {	OBJ_VAL1 => '' },
										 {	OBJ_VAL1 => (GetKey($obj, 'ms_isdn_any') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Any') },
										 {	CP3 => 1,
											OBJ_VAL1 => (GetKey($obj, 'ms_isdn_any') eq 'false')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Allow specific Access Point Names:'),
											OBJ_VAL3 =>	number(GetKey($obj, 'ms_isdn')) },
										 {	OBJ_VAL1 => '' }] },
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'static_eua') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Allow usage of static IP addresses') }] },
							{	SUBSECT_END => 1 }
						);
					}
			
					# Close off ADVANCED settings
					push @{$tobj[$cnt]{SVC_ADV}}, { SF_END => 1 };
				}
			}
			
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{SVC_KEEP}}) { push @{$tobj[$cnt]{SVC_KEEP}}, { VOID => 1 } }
			if (!defined @{$tobj[$cnt]{SVC_SPEC}}) { push @{$tobj[$cnt]{SVC_SPEC}}, { VOID => 1 } }
			if (!defined @{$tobj[$cnt]{SVC_ADV}})  { push @{$tobj[$cnt]{SVC_ADV}},  { VOID => 1 } }
			
			#------------------------------------------------
			# COMMENTS column of the SERVICES
			#------------------------------------------------
			$tobj[$cnt]{SVC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(SERVICES=>\@tobj);
	}
	
	##########################################################################
	# Routine: servicegroups2html
	#
	# Description:
	# Routine to create the SERVICEGROUPS table
	##########################################################################
	sub servicegroups2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                 SERVICEGROUPS to HTML                    ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable, $name);
	
			$tobj[$cnt]{SVC_TEXT}       = stripquotes($name);
			$tobj[$cnt]{SVC_NAME}       = HTMLObjectList($Policy, $obj, $name);
			$tobj[$cnt]{SVC_MEMBERS}    = HTMLMemberList($Policy, $obj, $_HREF);
			$tobj[$cnt]{SVC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(SERVICEGROUPS=>\@tobj);
	}
	
	##########################################################################
	# Routine: resources2html
	#
	# Description:
	# Routine to create the RESOURCES table
	##########################################################################
	sub resources2html {
		use strict;
		my (@ResourceList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $spec, $section);
	
		my (@var, %var, $var, $val, $v1, $v2);
		my (@tmp, $none);
		my ($UseResource, $share, $server);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                   RESOURCES to HTML                      ---
		#----------------------------------------------------------------
		$cnt  = -1;
		$none = '';
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable.':'.$name);
			$spec = GetKey($obj, 'spec_select') || GetKey($obj, 'resource_with_cvp') || '';
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{RSC_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{RSC_TEXT} = stripquotes($name);
			$tobj[$cnt]{RSC_TYPE} = $type;
			$tobj[$cnt]{RSC_NAME} = HTMLObjectList($Policy, $obj, $name);
			
			#------------------------------------------------
			# GENERAL tab/column in Resource
			#------------------------------------------------
			if ($type eq 'URI' || $type eq 'SMTP' || $type eq 'TCP' || $type eq 'FTP') {
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', $type.'_General');
				if ($section) {
					push @{$tobj[$cnt]{RSC_GENERAL}}, (
						{ 	SF_BEG =>	[{	SECTION => html('General'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
										
					if ($type eq 'URI') {
						# Resource Usage ...
						if 		(GetKey($obj, 'url_logging') eq 'true') 		{ $UseResource = 'URL' }
						elsif 	(GetKey($obj, 'url_logging') eq 'false') 		{ $UseResource = 'URI' }
						elsif 	(GetKey($obj, 'url_logging') eq 'enhance_ufp') 	{ $UseResource = 'UFP' };
						
						push @{$tobj[$cnt]{RSC_GENERAL}}, (
							{	TXT =>	[{	TEXT  => html('Use this resource to:'),
											CLASS => 'note_xspc' }], 
								S1 => 	[{	OBJ_VAL1 => ($UseResource eq 'URL')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Optimize URL logging') },
										 { 	OBJ_VAL1 => ($UseResource eq 'URI')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Enforce URI capabilities') },
										 { 	OBJ_VAL1 => ($UseResource eq 'UFP')?$MarkDot:$Mark,
											OBJ_VAL2 => html('Enhance UFP performance') }]
							}
						);
	
						if ($UseResource eq 'URI') {
							# Two columns ...
							push @{$tobj[$cnt]{RSC_GENERAL}}, { COL_BEG => 1 };
							
							# Connection Methods are stored in a list ...
							@var = GetMembers(GetKey($obj, 'connection_method'));
							%var = ();
							foreach $var (@var) { $var{$var} = 'true' }
			
							push @{$tobj[$cnt]{RSC_GENERAL}}, (
								{ 	SUBSECT_BEG => html('Connection'), 
									S1 => 	[{	OBJ_VAL1 => ($var{Transparent})?$CheckTick:$Check,
												OBJ_VAL2 => html('Transparent') },
											 { 	OBJ_VAL1 => ($var{Proxy})?$CheckTick:$Check,
												OBJ_VAL2 => html('Proxy') },
											 { 	OBJ_VAL1 => ($var{Tunneling})?$CheckTick:$Check,
												OBJ_VAL2 => html('Tunneling') }],
									SUBSECT_END => 1 }
							);
			
							push @{$tobj[$cnt]{RSC_GENERAL}}, { COL_NXT => 1 };
			
							# URI Match Specification Type ...
							push @{$tobj[$cnt]{RSC_GENERAL}}, (
								{ 	SUBSECT_BEG => html('URI Match Type'), 
									S1 => 	[{ 	OBJ_VAL1 => (($spec eq '"wild cards"')?$MarkDot:$Mark),
												OBJ_VAL2 => html('Wildcards') },
											 { 	OBJ_VAL1 => (($spec eq 'file')?$MarkDot:$Mark),
												OBJ_VAL2 => html('File') },
											 { 	OBJ_VAL1 => (($spec eq 'ufp')?$MarkDot:$Mark),
												OBJ_VAL2 => html('UFP') }],
									SUBSECT_END => 1 }
							);
															
							push @{$tobj[$cnt]{RSC_GENERAL}}, { COL_END => 1 };
						}
					}
				
					if ($type eq 'SMTP') {
						# Mail delivery ...
						push @{$tobj[$cnt]{RSC_GENERAL}}, (
							{	SUBSECT_BEG => html('Mail delivery'), 
								S1 => 	[{	CP2 => 1, 	OBJ_VAL1 => html('Server:'),
														OBJ_VAL2 => html(GetKey($obj, 'default_server')) }],
								S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'resolve_recipient_domain') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Deliver messages using DNS/MX records') },
										 {	OBJ_VAL1 => (GetKey($obj, 'check_rulebase_again') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Check Rule Base with new destination') }],
								SUBSECT_END => 1 }
						);
		
						# Error notification delivery ...
						push @{$tobj[$cnt]{RSC_GENERAL}}, (
							{ 	SUBSECT_BEG => html('Error Mail delivery'), 
								S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'err_notify') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Notify sender on error') }] }
						);
						if (GetKey($obj, 'err_notify') eq 'true') {
							push @{$tobj[$cnt]{RSC_GENERAL}}, (
								{ 	S1 => 	[{	CP2 => 1, 	OBJ_VAL1 => html('Server:'),
															OBJ_VAL2 => html(GetKey($obj, 'error_server')) }],
									S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'resolve_sender_domain') eq 'true')?$CheckTick:$Check,
												OBJ_VAL2 => html('Deliver messages using DNS/MX records') },
											 {	OBJ_VAL1 => (GetKey($obj, 'err_check_rulebase_again') eq 'true')?$CheckTick:$Check,
												OBJ_VAL2 => html('Check Rule Base with new error destination') }],
								SUBSECT_END => 1 }
							);
						}
					}
			
					if ($type eq 'TCP') {
						# TCP Resource type ...
						push @{$tobj[$cnt]{RSC_GENERAL}}, (
							{	SUBSECT_BEG => html('Type'), 
								S1 =>	[{ 	OBJ_VAL1 => (GetKey($obj, 'resource_with_cvp') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('UFP'), 
											OBJ_VAL3 => (GetKey($obj, 'resource_with_cvp') eq 'false')?$MarkDot:$Mark,
											OBJ_VAL4 => html('CVP') }],
								SUBSECT_END => 1 }
						);
					}
			
					if ((($type eq 'URI') && ($UseResource ne 'URL')) || 
						($type eq 'SMTP') || ($type eq 'TCP') || ($type eq 'FTP')) {
			
						# Exception Track ...
						$var = GetKey($obj, 'except_track:Name');
						$val = 0;
						if ($var eq 'None')                 { $val = 1 }
						elsif ($var eq '"Exception Log"')   { $val = 2 }
						elsif ($var eq '"Exception Alert"') { $val = 4 }
						push @{$tobj[$cnt]{RSC_GENERAL}}, (
							{ 	SUBSECT_BEG => html('Exception Track'), 
								S1 => 	[{ 	OBJ_VAL1 => (($val & 1)?$MarkDot:$Mark).html(' None'), 
											OBJ_VAL2 => (($val & 2)?$MarkDot:$Mark).html(' Log'), 
											OBJ_VAL3 => (($val & 4)?$MarkDot:$Mark).html(' Alert') }],
								SUBSECT_END => 1 }
						);
					}
					push @{$tobj[$cnt]{RSC_GENERAL}}, { SF_END => 1 };
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{RSC_GENERAL}}) { push @{$tobj[$cnt]{RSC_GENERAL}}, { VOID => 1 } }
	
			#------------------------------------------------
			# MATCH tab/column in Resource
			#------------------------------------------------
			if ( (($type eq 'URI') && ($spec ne '"wild cards"')) || ($type eq 'TCP') ) {
				# In these cases there is no MATCH tab at all ...
			} else {
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', $type.'_Match');
				if ($section) {
					push @{$tobj[$cnt]{RSC_MATCH}}, (
						{ 	SF_BEG =>	[{	SECTION => html('Match'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
				
					if (($type eq 'URI') && ($UseResource eq 'URI') && ($spec eq '"wild cards"')) {
						# URI resource with wild cards ...
			
						# Schemes used ...
						$val = GetKey($obj, 'multi_scheme_val');
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ 	SUBSECT_BEG => html('Schemes'), 
								S1 => 	[{ 	OBJ_VAL1 => (($val & $Scheme{http})?$CheckTick:$Check).html(' http'),
											OBJ_VAL2 => (($val & $Scheme{ftp})?$CheckTick:$Check).html(' ftp'),
											OBJ_VAL3 => (($val & $Scheme{gopher})?$CheckTick:$Check).html(' gopher'),
											OBJ_VAL4 => '' },
										 { 	OBJ_VAL1 => (($val & $Scheme{mailto})?$CheckTick:$Check).html(' mailto'),
											OBJ_VAL2 => (($val & $Scheme{news})?$CheckTick:$Check).html(' news'),
											OBJ_VAL3 => (($val & $Scheme{wais})?$CheckTick:$Check).html(' wais'),
											OBJ_VAL4 => html('Other: '.stripquotes(GetKey($obj, 'other_scheme'))) }],
								SUBSECT_END => 1 }
						);
			
						# Methods used ...
						$val = GetKey($obj, 'multi_method_val');
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ 	SUBSECT_BEG => html('Methods'), 
								S1 => 	[{ 	OBJ_VAL1 => (($val & $MethodHTTP{get})?$CheckTick:$Check).html(' GET'),
											OBJ_VAL2 => (($val & $MethodHTTP{post})?$CheckTick:$Check).html(' POST'),
											OBJ_VAL3 => (($val & $MethodHTTP{head})?$CheckTick:$Check).html(' HEAD'),
											OBJ_VAL4 => (($val & $MethodHTTP{put})?$CheckTick:$Check).html(' PUT'),
											OBJ_VAL5 => html('Other: '.stripquotes(GetKey($obj, 'other_method'))) }],
								SUBSECT_END => 1 }
						);
			
				
						# Host, Path and Query ...
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ 	SUBSECT_BEG => html('Wild cards'), 
								S1 => 	[{ CP2 => 1, OBJ_VAL1 => html('Host:'), 
													OBJ_VAL2 => stripquotes(GetKey($obj, 'host')) },
										 { CP2 => 1, OBJ_VAL1 => html('Path:'), 
													OBJ_VAL2 => stripquotes(GetKey($obj, 'path')) },
										 { CP2 => 1, OBJ_VAL1 => html('Query:'), 
													OBJ_VAL2 => stripquotes(GetKey($obj, 'query')) }],
								SUBSECT_END => 1 }
						);
								 
					} elsif ($type eq 'SMTP') {
						# SMTP Resource ....
						my @section;
						
						@var = GetMembers(GetKey($obj, 'match_from'));
						$val = 'Sender:';
						foreach $var (@var) {
							push @section, { CP2 => 1, OBJ_VAL1 => $val, OBJ_VAL2 => stripquotes($var) };
							$val = '';
						}
						@var = GetMembers(GetKey($obj, 'match_to'));
						$val = 'Recipient:';
						foreach $var (@var) {
							push @section, { CP2 => 1, OBJ_VAL1 => $val, OBJ_VAL2 => stripquotes($var) };
							$val = '';
						}
						push @{$tobj[$cnt]{RSC_MATCH}}, { 	
							SUBSECT_BEG => html('SMTP inspection'), 
							S1 => \@section,
							SUBSECT_END => 1 };
							
					} elsif ($type eq 'FTP') {
						# FTP Resource ....
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ 	SUBSECT_BEG => html('Wild cards'),
								S1 =>	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Path:'), 
											OBJ_VAL2 => html(stripquotes(GetKey($obj, 'path'))) }],
								SUBSECT_END => 1 }
						);
						
						# Methods used in MATCH tab ...
						$val = GetKey($obj, 'method_val');
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ 	SUBSECT_BEG => html('Methods'),
								S1 =>	[{ 	OBJ_VAL1 => ($val & $MethodFTP{GET})?$CheckTick:$Check,
											OBJ_VAL2 => html(' GET'),
											OBJ_VAL3 => ($val & $MethodFTP{PUT})?$CheckTick:$Check,
											OBJ_VAL4 => html(' PUT') }],
								SUBSECT_END => 1 }
						);
			
					} elsif ($type eq 'CIFS') {
						# CIFS Resource (allowed share list) ...
						my @section;
			
						push @section, { 	HEAD => 1,
											OBJ_VAL1 => html('Server Name'), 
											OBJ_VAL2 => html('Share Name') };
			
						@var = GetMembers(GetKey($obj, 'allowed_shares_list'));
						foreach $var (@var) {
							($server, $share) = split(/#/, stripquotes($var));
							push @section, { 	OBJ_VAL1 => $server, 
												OBJ_VAL2 => $share };
						}
			
						push @{$tobj[$cnt]{RSC_MATCH}}, { 	
							SUBSECT_BEG => html('Allowed Disk\Print Shares:'), 
							T1 => \@section, C2 => 1,
							SUBSECT_END => 1 };
			
						# CIFS log settings ...			
						push @{$tobj[$cnt]{RSC_MATCH}}, (
							{ S1 =>	[{ 	OBJ_VAL1 => (GetKey($obj, 'log_mapped_shares') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Log mapped shares') },
									 { 	OBJ_VAL1 => (GetKey($obj, 'log_access_violation') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Log access violation') },
									 { 	OBJ_VAL1 => (GetKey($obj, 'block_remote_registry') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Block remote registry access') }] }
						);
						
					} elsif ($type eq 'URI for QoS') {
						# URI for QOS Resource (used for floodgate) ...
						push @{$tobj[$cnt]{RSC_MATCH}}, 
							{ 	S1 =>	[{ 	CP2 => 1, 
											OBJ_VAL1 => html('Search for URL:'), 
											OBJ_VAL2 => html(stripquotes(GetKey($obj, 'path'))) }] };
							
					}
	
					push @{$tobj[$cnt]{RSC_MATCH}}, { SF_END => 1 };
				}
			}
					
			#------------------------------------------------
			# UFP tab/column in Resource
			#------------------------------------------------
			if (	(($type eq 'URI') && ($spec eq 'ufp')) ||
					(($type eq 'TCP') && ($spec eq 'false'))) {
				# In this case, the MATCH tab is UFP only ...
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', $type.'_UFP');
				if ($section) {
					push @{$tobj[$cnt]{RSC_MATCH}}, (
						{ 	SF_BEG =>	[{	SECTION => html('UFP'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
					# UFP options ...
					%var = %{obj2html($Policy, GetKey($obj, 'ufpserver:Table'), GetKey($obj, 'ufpserver:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{RSC_MATCH}}, 
						{ 	S1 => 	[{	CP2 => 1, 
										OBJ_VAL1 => html('UFP server:'), 
										OBJ_VAL2 => $var },
									 {	CP2 => 1, 
										OBJ_VAL1 => html('UFP caching control:'), 
										OBJ_VAL2 => $UFPCaching{GetKey($obj, 'cacheufp')} }] };
		
					# UFP-Categories ...
					my @section;
					@var = GetMembers(GetKey($obj, 'ufp_categories'));
					if (@var) { 
						foreach $var (@var) { 
							if (GetKey($var, 'selected') eq 'true') {
								push @tmp, stripquotes(GetKey($var, 'name'));
							}
						}
						foreach $var (sort @tmp) {
							push @section, { CP1 => 1, OBJ_VAL1 => html($var) };
						}
					} else {
						push @section, {  CP1 => 1, OBJ_VAL1 => $None };
					}
					push @{$tobj[$cnt]{RSC_MATCH}}, { 
						SUBSECT_BEG => html('Categories'), 
						S1 => \@section,
						SUBSECT_END => 1 };
		
					# UFP server failure ...
					push @{$tobj[$cnt]{RSC_MATCH}}, (
						{	SUBSECT_BEG => html('UFP server failure'), 
							S1 => [{	OBJ_VAL1 => (GetKey($obj, 'ufp_fail_allowed') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Ignore UFP server after connection failure') }] }
					);
												
					if (GetKey($obj, 'ufp_fail_allowed') eq 'true') {
						push @{$tobj[$cnt]{RSC_MATCH}}, 
							{ S1 => [{ CP2 => 1, 	OBJ_VAL1 => html('Number of failures before ignoring the UFP server:'), 
													OBJ_VAL2 => number(GetKey($obj, 'ufp_skip_threshold')) },
									 { CP2 => 1, 	OBJ_VAL1 => html('Timeout before reconnect to UFP server (seconds):'), 
													OBJ_VAL2 => number(GetKey($obj, 'ufp_skip_time')) }] };
					}
					push @{$tobj[$cnt]{RSC_MATCH}}, { SUBSECT_END => 1 };
					
					push @{$tobj[$cnt]{RSC_MATCH}}, { SF_END => 1 };
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{RSC_MATCH}}) { push @{$tobj[$cnt]{RSC_MATCH}}, { VOID => 1 } }
	
	
			#------------------------------------------------
			# ACTIONS tab/column in Resource
			#------------------------------------------------
			if (($type eq 'URI') && ($UseResource eq 'URI')) {
				# URI Resources all have the same ACTIONS tab/column
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', 'URI_Action');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG =>	[{	SECTION => html('Action'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
				
					# Replacement URI ...
					push @{$tobj[$cnt]{RSC_ACTIONS}}, 
						{ S1 =>	[{ 	CP2 => 1, 
									OBJ_VAL1 => html('Replacement URI:'), 
									OBJ_VAL2 => html(stripquotes(GetKey($obj, 'redirect_to'))) }] };
		
					# HTML Weeding & Response Scanning ...
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{	COL_BEG => 1 },
						{ 	SUBSECT_BEG => html('HTML Weeding'),
							S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'response:strip_script_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip SCRIPT Tags') },
									 {	OBJ_VAL1 => (GetKey($obj, 'response:strip_applet_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip Applet Tags') },
									 {	OBJ_VAL1 => (GetKey($obj, 'response:strip_active_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip ActiveX Tags') },
									 {	OBJ_VAL1 => (GetKey($obj, 'response:http_erase_ftp_link') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip FTP Links') },
									 {	OBJ_VAL1 => (GetKey($obj, 'response:http_erase_port_cmd') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip Port Strings') }],
							SUBSECT_END => 1 },
						{	COL_NXT => 1 },
						{ 	SUBSECT_BEG => html('Response Scanning'),
							S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'response:strip_java') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Block JAVA Code') }],
							SUBSECT_END => 1 },
						{	COL_END => 1 }
					);
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { SF_END => 1 };
				}
			} 
			
			if ($type eq 'SMTP') {
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', 'SMTP_Action1');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG =>	[{	SECTION => html('Action1'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
				
					# Rewriting rules ...
					my @section;
					@var = GetMembers(GetKey($obj, 'from'));
					push @section, { 	CP2 => 1, CP4 => 1,
										OBJ_VAL1 => html('Sender:'), 
										OBJ_VAL2 => html(stripquotes($var[0])), 
										OBJ_VAL3 => html(' -> '),
										OBJ_VAL4 => html(stripquotes($var[1])) };
		
					@var = GetMembers(GetKey($obj, 'to'));
					push @section, { 	CP2 => 1, CP4 => 1,
										OBJ_VAL1 => html('Recipient:'), 
										OBJ_VAL2 => html(stripquotes($var[0])), 
										OBJ_VAL3 => html(' -> '),
										OBJ_VAL4 => html(stripquotes($var[1])) };
		
					@var = GetMembers(GetKey($obj, 'user_field'));
					if ($var[0]) {
						push @section, { 	CP2 => 1, CP4 => 1,
											OBJ_VAL1 => html(stripquotes($var[0]).':'), 
											OBJ_VAL2 => html(stripquotes($var[1])), 
											OBJ_VAL3 => html(' -> '),
											OBJ_VAL4 => html(stripquotes($var[2])) };
					}
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { 
						SUBSECT_BEG => html('Rewriting Rules'), 
						S1 => \@section,
						SUBSECT_END => 1 };
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { SF_END => 1 };
				}
	
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', 'SMTP_Action2');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG =>	[{	SECTION => html('Action2'),
											OBJ_NAME => $name,
											STATE	 => ($section eq $_OPEN) }]
						}
					);
					# Strip code ...
					@var = GetMembers(GetKey($obj, 'content_type'));
					$v1 = stripquotes($var[0]);
					@var = GetMembers(GetKey($obj, 'forbiddenfiles'));
					$v2 = stripquotes($var[0]);
					push @{$tobj[$cnt]{RSC_ACTIONS}},
						{ 	SUBSECT_BEG => html('Attachment handling'), 
							S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('Strip MIME of type:'), 
										OBJ_VAL2 => $v1 },
									 { 	CP2 => 1, 
										OBJ_VAL1 => html('Strip file by name:'), 
										OBJ_VAL2 => $v2 }],
							SUBSECT_END => 1 };
		
					# Other parameters ...
					push @{$tobj[$cnt]{RSC_ACTIONS}}, 
						{ 	S1 => 	[{ CP2 => 1, 
							OBJ_VAL1 => html('Do not send mail larger than:'), 
							OBJ_VAL2 => number(GetKey($obj, 'maxsize')),
							OBJ_VAL3 => html('KB') }] };
		
					# Other parameters ...
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SUBSECT_BEG => html('Allowed Characters'),
							S1 => 	[{ 	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'allowed_chars') eq '"8 bit"')?$MarkDot:$Mark,
										OBJ_VAL2 => html('8-bit') },
									 { 	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'allowed_chars') eq '"8 bit"')?$Mark:$MarkDot,
										OBJ_VAL2 => html('7-bit (no CTRL chars)') }],
							SUBSECT_END => 1 }
					);
		
					# Weeding ...
					push @{$tobj[$cnt]{RSC_ACTIONS}}, 
						{ 	SUBSECT_BEG => html('Weeding'),
							S1 => 	[{	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'smtp_strip_script_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip SCRIPT Tags') },
									 {	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'smtp_strip_applet_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip Applet Tags') },
									 {	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'smtp_strip_active_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip ActiveX Tags') },
									 {	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'smtp_strip_ftp_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip FTP Links') },
									 {	CP1 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'smtp_strip_port_tags') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Strip Port Strings') }],
							SUBSECT_END => 1
						};
						
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { SF_END => 1 };
				}
			}
	
			#------------------------------------------------
			# CVP tab/column in Resource
			#------------------------------------------------
			if (($type eq 'URI') && ($UseResource eq 'URI')) {
				# CVP Options in URI resource ...
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', 'URI_CVP');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG => [{	SECTION => html('CVP'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'response:av_use') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Use CVP (Content Vectoring Protocol)') }] }
					);
					
					if (GetKey($obj, 'response:av_use') eq 'true') {
						%var = %{obj2html($Policy, GetKey($obj, 'response:av_server:Table'), GetKey($obj, 'response:av_server:Name'), $_HREF)};
						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
						push @{$tobj[$cnt]{RSC_ACTIONS}}, (
							{	S1 =>	[{	CP2 => 1, 
											OBJ_VAL1 => html('CVP server:'),
											OBJ_VAL2 => $var }] },
							{	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'response:av_setting') eq 'cure')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('CVP server is allowed to modify content') },
										 {	OBJ_VAL1 => (GetKey($obj, 'response:av_headers') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('Send HTTP headers to CVP server') },
										 {	OBJ_VAL1 => (GetKey($obj, 'response:http_cvp_on_request') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('Send HTTP requests to CVP server') },
										 {	OBJ_VAL1 => (GetKey($obj, 'response:http_cvp_only_unsafe') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('Send only unsafe file types to CVP server') }] },
							{	SUBSECT_BEG => html('Reply Order') },
							{	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'response:reply_mode') eq 'reply_first')?$MarkDot:$Mark,
											OBJ_VAL2 =>	html('Return data after content is approved') },
										 {	OBJ_VAL1 => (GetKey($obj, 'response:reply_mode') eq 'reply_first')?$Mark:$MarkDot,
											OBJ_VAL2 =>	html('Return data before content is approved') }] },
							{	SUBSECT_END => 1 }
						);
					}
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { SF_END => 1 };
				}
		
			} elsif (($type eq 'SMTP') || ($type eq 'FTP') || ($type eq 'TCP')) {
				# CVP Options in non-URI resource ...
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', $type.'_CVP');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG => [{	SECTION  => html('CVP'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'av_use') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Use CVP (Content Vectoring Protocol)') }] }
					);
					
					if (GetKey($obj, 'av_use') eq 'true') {
						%var = %{obj2html($Policy, GetKey($obj, 'av_server:Table'), GetKey($obj, 'av_server:Name'), $_HREF)};
						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
						push @{$tobj[$cnt]{RSC_ACTIONS}}, (
							{	S1 =>	[{	CP2 => 1, 
											OBJ_VAL1 => html('CVP server:'),
											OBJ_VAL2 => $var }] },
							{	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'av_setting') eq 'cure')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('CVP server is allowed to modify content') },
										 {	OBJ_VAL1 => (GetKey($obj, 'av_headers') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 =>	html('Send SMTP headers to CVP server') }] },
							{	SUBSECT_BEG => html('Reply Order') },
							{	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'reply_mode') eq 'reply_first')?$MarkDot:$Mark,
											OBJ_VAL2 =>	html('Return data after content is approved') },
										 {	OBJ_VAL1 => (GetKey($obj, 'reply_mode') eq 'reply_first')?$Mark:$MarkDot,
											OBJ_VAL2 =>	html('Return data before content is approved') }] },
							{	SUBSECT_END => 1 }
						);
					}
					push @{$tobj[$cnt]{RSC_ACTIONS}}, { SF_END => 1 };
				}
			}
	
			#------------------------------------------------
			# SOAP tab/column of Resource
			#------------------------------------------------
			if (	($type eq 'URI') && ($spec eq '"wild cards"') &&
					((GetKey($obj, 'multi_scheme_val') & $Scheme{http}) || (GetKey($obj, 'other_scheme') eq '"*"')) ) {
				# if URI spec type is wildcards AND schemes include http ...
	
				$section = Section($Policy, Object2Page($Policy, 'Resources'), 'Resources', 'URI_SOAP');
				if ($section) {
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SF_BEG => [{	SECTION  => html('SOAP'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SUBSECT_BEG => html('SOAP Options') },
						{	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'request:soap_accept_all_conn') eq 'accept_all_conns')?$MarkDot:$Mark,
										OBJ_VAL2 =>	html('Allow all SOAP requests') },
									 {	OBJ_VAL1 => (GetKey($obj, 'request:soap_accept_all_conn') eq 'accept_by_file_id')?$MarkDot:$Mark,
										OBJ_VAL2 =>	html('Allow SOAP requests as specified in the following file.;').
													html('(This will not block non-SOAP HTTP traffic):') }] }
					);
					if (GetKey($obj, 'request:soap_accept_all_conn') eq 'accept_by_file_id') {
						push @{$tobj[$cnt]{RSC_ACTIONS}}, (
							{	S1 =>	[{	CP3 => 1,
											OBJ_VAL1 => $Spacer,
											OBJ_VAL2 => html('File ID:'),
											OBJ_VAL3 => stripquotes(GetKey($obj, 'request:soap_file_id')) }] }
						);
					}
					push @{$tobj[$cnt]{RSC_ACTIONS}}, {	SUBSECT_END => 1 };
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, (
						{ 	SUBSECT_BEG => html('Track'),
							S1 =>	[{ 	CP2 => 1,
										OBJ_VAL1 => html('Track SOAP connections:'),
										OBJ_VAL2 => $Track{GetKey($obj, 'request:soap_track')} }],
							SUBSECT_END => 1 }
					);
	
					push @{$tobj[$cnt]{RSC_ACTIONS}}, {	SF_END => 1 };
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{RSC_ACTIONS}}) { push @{$tobj[$cnt]{RSC_ACTIONS}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENTS tab/column of Resource
			#------------------------------------------------
			$tobj[$cnt]{RSC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
				
		}
			
		$template->param(RESOURCES=>\@tobj);
	}
	
	##########################################################################
	# Routine: servers2html
	#
	# Description:
	# Routine to create the SERVERS table
	##########################################################################
	sub servers2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name);
		my ($type, $current_type, $spec, $section);
	
		my (@var, %var, $var, $val);
		my (@tmp, $tmp, $opsec);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                    SERVERS to HTML                       ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj = GetKey($CPtable.':'.$name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{SRV_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{SRV_TEXT} = stripquotes($name);
			$tobj[$cnt]{SRV_NAME} = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# General column/tab ...
			#------------------------------------------------
			if (($type eq 'RADIUS') || ($type eq 'TACACS') || ($type eq 'SecuRemote DNS') || ($type eq 'Certificate Authority')) {
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', $type.'_General');
				if ($section) {
					my @section;
					push @{$tobj[$cnt]{SRV_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION => html('General'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					if (($type eq 'RADIUS') || ($type eq 'TACACS') || ($type eq 'SecuRemote DNS')) {
						# Host ...
						%var = %{obj2html($Policy, GetKey($obj, 'server:Table'), GetKey($obj, 'server:Name'), $_HREF)};
						push @section, {	CP2 => 1, 
											OBJ_VAL1 => html('Host:'), 
											OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
					
						if ($type eq 'RADIUS') {
							# Service ...
							%var = %{obj2html($Policy, GetKey($obj, 'service:Table'), GetKey($obj, 'service:Name'), $_HREF)};
							push @section, { 	CP2 => 1, 
												OBJ_VAL1 => html('Service:'), 
												OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
				
							# Version & Priority ...
							$var = stripquotes(GetKey($obj, 'version'));
							push @section, {	CP2 => 1, 
												OBJ_VAL1 => html('Version:'), 
												OBJ_VAL2 => ($var =~ /1\.0/)?'RADIUS Ver. 1.0 Compatible':'RADIUS Ver. 2.0 Compatible' };
							push @section, { 	CP2 => 1, 
												OBJ_VAL1 => html('Priority:'), 
												OBJ_VAL2 => number(GetKey($obj, 'priority')) };
						}
							 
						if ($type eq 'TACACS') {
							# Tacacs type ...
							$val = stripquotes(GetKey($obj, 'tacacs_server_type'));
							push @section, { 	OBJ_VAL1 => html('Type:'), 
												OBJ_VAL2 => (($val eq 'TACACS')?$MarkDot:$Mark).html(' TACACS  ').
															$Spacer.
															(($val ne 'TACACS')?$MarkDot:$Mark).html(' TACACS +') 
							};
				
							if ($val eq 'TACACS') {
								# Service ...
								%var = %{obj2html($Policy, GetKey($obj, 'service:Table'), GetKey($obj, 'service:Name'), $_HREF)};
								push @section, { 	CP2 => 1, 
													OBJ_VAL1 => html('Service:'), 
													OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
							}
						}
					}
						
					if ($type eq 'Certificate Authority') {
						# Certificate Authority type ...
						$var = stripquotes(GetKey($obj, 'ca_type'));
						if ($var eq 'internal') { $var = html('Local management server') }
						push @section, {	CP2 => 1, 
											OBJ_VAL1 => html('Certificate Authority:'), 
											OBJ_VAL2 => $var 
						};
					}
	
					push @{$tobj[$cnt]{SRV_GENERAL}}, { S1 => \@section, SF_END => 1 };
				}
			}
	
			#------------------------------------------------
			# In Group column/tab ...
			#------------------------------------------------
			if ($type eq 'RADIUS Group') {
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'RADIUSGrp_Members');
				if ($section) {
					my @section;
					push @{$tobj[$cnt]{SRV_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION => html('Members'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					my @section;
					# Find Members ...
					@var = GetMembers($obj);
					if (@var) {
						foreach $var (@var) {
							%var = %{obj2html($Policy, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
							push @section, {	CP1 => 1, 
												OBJ_VAL1 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
						}
					} else {
						push @section, {	OBJ_VAL1 => $None };
					}
					push @{$tobj[$cnt]{SRV_GENERAL}}, { 
						S1 => \@section,
						SF_END => 1 
					}
				}
			}
	
			#------------------------------------------------
			# Account Unit usage column/tab ...
			#------------------------------------------------
			if ($type eq 'LDAP Account Unit') {
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'LDAP_Account');
				if ($section) {
					# Account Unit usage ...
					@var = GetMembers(GetKey($obj, 'ldap_usage'));
					%var = ();
					foreach $var (@var) { $var{$var} = 'true' }
		
					# Profile, but no icons for LDAP profile/policy ...
					%var = %{obj2html($Policy, GetKey($obj, 'ldap_policy:Table'), GetKey($obj, 'ldap_policy:Name'), $_HREF)};
		
					push @{$tobj[$cnt]{SRV_GENERAL}}, (
						{ SF_BEG => [{	SECTION => html('Account Unit usage'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }] },
						{ S1 =>	[{	OBJ_VAL1 => ($var{'crl'})?$CheckTick:$Check,
									OBJ_VAL2 => html('CRL retrieval') },
								 {	OBJ_VAL1 => ($var{'user'})?$CheckTick:$Check,
									OBJ_VAL2 => html('User management') }] },
						{ S1 =>	[{	CP2 => 1,
									OBJ_VAL1 => html('Profile:'),
									OBJ_VAL2 => $var{OBJ_TEXT} }] },
						{ SF_END => 1 }
					);
				}
			}
			
			
			# Check for empty cell and correct if necessary ...
			if (!defined $tobj[$cnt]{SRV_GENERAL}) { 
				push @{$tobj[$cnt]{SRV_GENERAL}}, { VOID => 1 }
			}
	
			#------------------------------------------------
			# SecuRemote DNS Domains ...
			#------------------------------------------------
			if ($type eq 'SecuRemote DNS') {
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'SecuRemote DNS_Domains');
				if ($section) {
					my @section;
					push @section, {	HEAD => 1,
										OBJ_VAL1 => html('Name'), 
										OBJ_VAL2 => html('Max. Label Count') };
					@var = GetMembers(GetKey($obj, 'domain'));
					foreach $var (@var) {
						push @section, { 	OBJ_VAL1 => html(GetKey($var, 'domain')), 
											OBJ_VAL2 => number(GetKey($var, 'dns_label_count'))};
					}
	
					push @{$tobj[$cnt]{SRV_OTHER}}, (
						{ SF_BEG => [{	SECTION => html('Domains'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }] },
						{ TXT => [{	TEXT => html('Domain Patterns:'),
									CLASS => 'note_xspc' }] },
						{ T1 => \@section, C2 => 1 },
						{ SF_END => 1 }
					);
				}
			} 
			
			#------------------------------------------------
			# LDAP Servers ...
			#------------------------------------------------
			if ($type eq 'LDAP Account Unit') {
				# LDAP Servers
				my %server;
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'LDAP_Servers');
				if ($section) {
					push @{$tobj[$cnt]{SRV_OTHER}}, (
						{ SF_BEG => [{	SECTION => html('Servers'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }] },
						{ TXT => [{	TEXT => html('LDAP servers:'),
									CLASS => 'note_xspc' }] }
					);
					
					my @section;
					push @section, { 	HEAD => 1,
										OBJ_VAL1 => html('Host'), 
										OBJ_VAL2 => html('Port'), 
										OBJ_VAL3 => html('Priority') };
					@var = GetMembers(GetKey($obj, 'ldap_servers'));
					foreach $var (@var) {
						# LDAP Server ...
						%var = %{obj2html($Policy, GetKey($var, 'server:Table'), GetKey($var, 'server:Name'), $_HREF)};
						$tmp = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
						push @section, { 	OBJ_VAL1 => $tmp,
											OBJ_VAL2 => number(GetKey($var, 'ldap_ssl_port')), 
											OBJ_VAL3 => number(GetKey($var, 'priority')) };
						$server{GetKey($var, 'AdminInfo:chkpf_uid')}{Name}  = GetKey($var, 'server:Name');
						$server{GetKey($var, 'AdminInfo:chkpf_uid')}{Table} = GetKey($var, 'server:Table');
					}
					push @{$tobj[$cnt]{SRV_OTHER}}, { T1 => \@section, C3 => 1 };
		
					# Early version compatibility server
					my @section;
					$tmp = GetKey($obj, 'bc_server');
					foreach $val (keys %server) {
						if ($val eq $tmp) {
							%var = %{obj2html($Policy, $server{$val}{Table}, $server{$val}{Name}, $_HREF)};
							push @{$tobj[$cnt]{SRV_OTHER}}, { 
								S1 =>	[{	CP2 => 1, 
											OBJ_VAL1 => html('Early Versions server:'), 
											OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }] };
						}
					}
					push @{$tobj[$cnt]{SRV_OTHER}}, { SF_END => 1 };
				}
				
			#------------------------------------------------
			# LDAP Objects Management ...
			#------------------------------------------------
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'LDAP_ObjMgmt');
				if ($section) {
					push @{$tobj[$cnt]{SRV_OTHER}}, (
						{ 	SF_BEG => [{	SECTION  => html('Objects Management'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					my @section;
					$tmp = GetKey($obj, 'editable_server');
					foreach $val (keys %server) {
						if ($val eq $tmp) {
							%var = %{obj2html($Policy, $server{$val}{Table}, $server{$val}{Name}, $_HREF)};
							push @{$tobj[$cnt]{SRV_OTHER}}, { 
								S1 =>	[{	CP2 => 1, 
											OBJ_VAL1 => html('Manage objects on:'), 
											OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }] };
						}
					}
		
					# Branches ...
					push @{$tobj[$cnt]{SRV_OTHER}}, { SUBSECT_BEG => html('Branches in use') };
					my @section;
					@var = GetMembers(GetKey($obj, 'branches'));
					if (@var) {
						foreach $var (@var) {
							push @section, { 	CP1 => 1, 
												OBJ_VAL1 => html(stripquotes($var)) }
						}
					} else {
						push @section, { 	OBJ_VAL1 => $None };
					}
					push @{$tobj[$cnt]{SRV_OTHER}}, { S1 => \@section };
					push @{$tobj[$cnt]{SRV_OTHER}}, { SF_END => 1 };
				}
			} 
			
			#------------------------------------------------
			# Local Management Server ...
			#------------------------------------------------
			if ($type eq 'Certificate Authority') {
				# Certificate Authority CRL Retrieval ...
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'LDAP_LMServer');
				if ($section) {
					push @{$tobj[$cnt]{SRV_OTHER}}, (
						{ 	SF_BEG => [{	SECTION  => html('Local management server'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							TXT => 	[{	TEXT => html('Retrieve CRL From:'), 
										CLASS => 'note_xspc' }],
							S1 => 	[{	CP3 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'crl_ldap') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('LDAP Server(s)') },
									 {	CP3 => 1, 
										OBJ_VAL1 => (GetKey($obj, 'crl_http') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('HTTP Server(s)') }],
							SF_END => 1
						}
					);
				}
				
				# Certificate Authority CRL Caching ...
				$section = Section($Policy, Object2Page($Policy, 'Servers'), 'Servers', 'LDAP_Advanced');
				if ($section) {
					push @{$tobj[$cnt]{SRV_OTHER}}, (
						{ 	SF_BEG => [{	SECTION  => html('Advanced'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					push @{$tobj[$cnt]{SRV_OTHER}}, {
						SUBSECT_BEG => html('CRL Caching'),
						S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'crl_cache_type') ne 'None')?$CheckTick:$Check,
									OBJ_VAL2 => html('Cache CRL on the module') }] 
					};
					if (GetKey($obj, 'crl_cache_type') ne 'None') {
						push @{$tobj[$cnt]{SRV_OTHER}}, {
							S1 =>	[{	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj, 'crl_cache_type') eq 'exp_Date')?$MarkDot:$Mark,
										OBJ_VAL3 => html('Fetch new CRL when expires') }]
						};
						push @{$tobj[$cnt]{SRV_OTHER}}, {
							S1 =>	[{	CP4 => 1, 
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj, 'crl_cache_type') eq 'Timeout')?$MarkDot:$Mark,
										OBJ_VAL3 => html('Fetch new CRL after:'),
										OBJ_VAL4 => (GetKey($obj, 'crl_cache_type') eq 'Timeout')?html(GetKey($obj, 'crl_cache_timeout')):'',
										OBJ_VAL5 => html('seconds')	}]
						}
					}
					push @{$tobj[$cnt]{SRV_OTHER}}, { SUBSECT_END => 1 };
		
		
					# Certificate Authority
					push @{$tobj[$cnt]{SRV_OTHER}}, { 
						SUBSECT_BEG => html('Certificate Authority'), 
						S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'permissions_type') eq 'accept_only')?$CheckTick:$Check,
									OBJ_VAL2 => html('Allow only certificates from the listed branches') }]
					};
					if (GetKey($obj, 'permissions_type') eq 'accept_only') {
						my @section;
						@var = GetMembers(GetKey($obj, 'permissions_strings'));
						if (@var) {
							foreach $var (@var) {
								push @section, { 	CP1 => 1, 
													OBJ_VAL1 => html(stripquotes($var)) }
							}
						} else {
							push @section, {	OBJ_VAL1 => $None };
						}
						push @{$tobj[$cnt]{SRV_OTHER}}, { S1 => \@section };
					}
					push @{$tobj[$cnt]{SRV_OTHER}}, { SUBSECT_END => 1 };
					push @{$tobj[$cnt]{SRV_OTHER}}, { SF_END => 1 };
				}
			}
			
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{SRV_OTHER}}) { push @{$tobj[$cnt]{SRV_OTHER}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENTS tab/column of Resource
			#------------------------------------------------
			$tobj[$cnt]{SRV_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
				
		}
			
		$template->param(SERVERS=>\@tobj);
	}
	
	
	##########################################################################
	# Routine: opsec2html
	#
	# Description:
	# Routine to create the OPSEC Application table
	##########################################################################
	sub opsec2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name, $section);
		my ($type, $current_type, $spec);
	
		my (@var, %var, $var, $val);
		my (@tmp, $tmp, $opsec);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---              OPSEC Applications to HTML                  ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable.':'.$name);
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OPSEC_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{OPSEC_TEXT} = stripquotes($name);
			$tobj[$cnt]{OPSEC_NAME} = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# General column/tab
			#------------------------------------------------
			if ($type eq 'OPSEC Application') {
				$section = Section($Policy, Object2Page($Policy, 'OPSEC'), 'OPSEC', 'OPSEC_General');
				if ($section) {
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('General'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					
					# Host ...
					%var = %{obj2html($Policy, GetKey($obj, 'host:Table'), GetKey($obj, 'host:Name'), $_HREF)};
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { 
						S1 => 	[{	CP2 => 1, 
									OBJ_VAL1 => html('Host:'), 
									OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }]
					};
		
					# Application properties ...
					my @section;
					# Vendor, Product & Version ...
					$var = GetKey(GetKey($obj, 'product:Table').':'.GetKey($obj, 'product:Name'));
					push @section, {	CP2 => 1, 
										OBJ_VAL1 => html('Vendor:'),
										OBJ_VAL2 => html(stripquotes(GetKey($var, 'product_info:vendor'))) };
					$opsec = $obj;
					if (stripquotes(GetKey($var, 'product_info:vendor')) ne 'User defined') {
						push @section, 	{	CP2 => 1, 
											OBJ_VAL1 => html('Product:'), 
											OBJ_VAL2 => html(stripquotes(GetKey($var, 'product_info:product'))) },
										{ 	CP2 => 1, 
											OBJ_VAL1 => html('Version:'), 
											OBJ_VAL2 => html(stripquotes(GetKey($var, 'product_info:version'))) };
						$opsec = $var;
					}
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { 
						SUBSECT_BEG => html('Application properties'), 
						S1 => \@section,
						SUBSECT_END => 1 };
		
					# Server and Client entities special table format ...
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { 
						COL_BEG => 1,
						SUBSECT_BEG => html('Server Entities'),
						S1 => 	[{	OBJ_VAL1 => (GetKey($opsec, 'CVP') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('CVP') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'UFP') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('UFP') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'AMON') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('AMON') }]
					};
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { 
						COL_NXT => 1,
						SUBSECT_BEG => html('Client Entities'),
						S1 => 	[{	OBJ_VAL1 => (GetKey($opsec, 'ELA') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('ELA') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'LEA') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('LEA') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'SAM') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('SAM') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'CPMI') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('CPMI') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'OMI') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('OMI') },
								 {	OBJ_VAL1 => (GetKey($opsec, 'UAC') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('UAA') }]
					};
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { COL_END => 1 };
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { SF_END => 1 };
				}
				
			} elsif ($type =~ /Group/) {
				# Group Members ...
				$section = Section($Policy, Object2Page($Policy, 'OPSEC'), 'OPSEC', 'OPSEC_GrpMembers');
				if ($section) {
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('Members'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					my @section;
					@var = GetMembers(GetKey($obj, 'members'));
					if (@var) {
						foreach $var (@var) {
							%var = %{obj2html($Policy, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
							push @section, { 	C1 => 1, CP1 => 1,
												OBJ_VAL1 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
						}
					} else {
						push @section, {	C1 => 1, 
											OBJ_VAL1 => $None };
					}
					push @{$tobj[$cnt]{OPSEC_GENERAL}}, { 
						S1 => \@section,
						SF_END => 1 
					};
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OPSEC_GENERAL}}) { push @{$tobj[$cnt]{OPSEC_GENERAL}}, { VOID => 1 } }
			
			#------------------------------------------------
			# Options column
			#------------------------------------------------
			if ($type eq 'OPSEC Application') {
				# CVP/UFP/AMON Options ...
	
				foreach $var ('CVP', 'UFP', 'AMON') {
					# Service used ...
					$section = Section($Policy, Object2Page($Policy, 'OPSEC'), 'OPSEC', 'OPSEC_'.$var.'_Options');
					if ((GetKey($opsec, $var) eq 'true') && $section) {
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, (
							{ 	SF_BEG => [{	SECTION  => html($var.' Options'), 
												OBJ_NAME => $name, 
												STATE    => ($section eq $_OPEN) }]
							}
						);
						%var = %{obj2html($Policy, GetKey($obj, $var.'_settings:service:Table'), GetKey($obj, $var.'_settings:service:Name'), $_HREF)};
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
							S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('Service:'), 
										OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }]
						};
	
						# UFP Options ...
						if ($var eq 'UFP') {
							# Dictionary ...
							push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
								SUBSECT_BEG => html('Dictionary'), 
								S1 =>	[{	CP2 => 1,
											OBJ_VAL1 => html('Dictionary ID:'),
											OBJ_VAL2 => html(GetKey($obj, 'UFP_settings:dictionary_id')) },
										 {	CP2 => 1,
											OBJ_VAL1 => html('Description:'),
											OBJ_VAL2 => html(GetKey($obj, 'UFP_settings:description')) }],
								SUBSECT_END => 1
							};
			
							# Categories ...
							my @section;
							@var = GetMembers(GetKey($obj, 'UFP_settings:categories'));
							@tmp = ();
							foreach $var (@var) { push @tmp, stripquotes(GetKey($var, 'name')) }
							if (@tmp) {
								foreach $var (sort @tmp) {
									push @section, {	CP1 => 1, 
														OBJ_VAL1 => html($var) };
								}
							} else {
								push @section, { 	OBJ_VAL1 => $None };
							}
							push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
								SUBSECT_BEG => html('Categories in Dictionary'), 
								S1 => \@section,
								SUBSECT_END => 1 };
						}	
							
						if ($var eq 'AMON') {
							# Identifier ...
							push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
								S1 => 	[{ 	CP2 => 1, 	OBJ_VAL1 => html('Identifier:'),
														OBJ_VAL2 => html(stripquotes(GetKey($obj, 'AMON_class_name'))) }]
							}
						}
					
						if (($var eq 'CVP') || ($var eq 'UFP')) {
							# Early version compatibility mode
							$val = GetKey($obj, $var.'_settings:backwards_compatibility_security_method');
							push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
								S1 => 	[{ 	OBJ_VAL1 => $val?$CheckTick:$Check,
											OBJ_VAL2 => html('Use early version compatibility mode') }]
							};
							if ($val) {
								push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
									S1 => 	[{ 	OBJ_VAL1 => $Spacer,
												OBJ_VAL2 => ($val eq 'opsec')?$MarkDot:$Mark,
												OBJ_VAL3 => html('Clear (opsec)') }, 
											 {	OBJ_VAL1 => $Spacer,
												OBJ_VAL2 => ($val eq 'auth_opsec')?$MarkDot:$Mark,
												OBJ_VAL3 => html('OPSEC Authentication (auth_opsec)') }, 
											 {	OBJ_VAL1 => $Spacer,
												OBJ_VAL2 => ($val eq 'ssl_opsec')?$MarkDot:$Mark,
												OBJ_VAL3 => html('OPSEC SSL (ssl_opsec)') }, 
											 {	OBJ_VAL1 => $Spacer,
												OBJ_VAL2 => ($val eq 'ssl_clear_opsec')?$MarkDot:$Mark,
												OBJ_VAL3 => html('OPSEC SSL Clear (ssl_clear_opsec)') }]
								};
							}
						}
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { SF_END => 1 };
					}
				}
				
			} elsif (($type eq 'CVP Group') || ($type eq 'UFP Group')) {
				# Work distribution method ...
				$section = Section($Policy, Object2Page($Policy, 'OPSEC'), 'OPSEC', 'OPSEC_GrpWD');
				if ($section) {
					push @{$tobj[$cnt]{OPSEC_OPTIONS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Work distribution method'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					$var = 'load sharing';
					if ($type eq 'CVP Group') {
						$var = stripquotes(GetKey($obj, 'method'));
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
							S1 =>	[{	OBJ_VAL1 => ($var eq 'load sharing')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Load sharing'),
										OBJ_VAL3 => ($var ne 'load sharing')?$MarkDot:$Mark,
										OBJ_VAL4 => html('Chaining') }]
						};
					}
					
					if ($var ne 'load sharing') {
						# Chaining ...
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
							S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'drop_on_unsafe') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Abort chaining upon Unsafe reply') }]
						};
					} else {
						# Load sharing ...
						push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { 
							S1 =>	[{	CP2 => 1,
										OBJ_VAL1 => html('Load sharing method:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'load_sharing_method'))) },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Load sharing suspend timeout:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'load_sharing_suspend_timeout'))),
										OBJ_VAL3 => html('minutes') }]
						};
					}
					push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { SF_END => 1 };
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{OPSEC_OPTIONS}}) { push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { VOID => 1 } }
	
			#------------------------------------------------
			# COMMENTS tab/column of Resource
			#------------------------------------------------
			$tobj[$cnt]{OPSEC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(OPSEC=>\@tobj);
	}
	
		
	##########################################################################
	# Routine: times2html
	#
	# Description:
	# Routine to create the TIME object table
	##########################################################################
	sub	times2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name, $section);
		my ($type, $current_type);
		my (@var, %var, $var, $val);
		my (@tmp, $tmp);
		my ($h1, $m1, $h2, $m2, $h, $m, $s);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                  Time objects to HTML                    ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable.':'.$name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{TIME_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{TIME_TEXT} = stripquotes($name);
			$tobj[$cnt]{TIME_NAME} = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# General column/tab
			#------------------------------------------------
			if ($type eq 'Time') {
				$section = Section($Policy, Object2Page($Policy, 'Times'), 'Times', 'Time of day');
				if ($section) {
					push @{$tobj[$cnt]{TIME_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('Time of day (hh:mm)'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					my @section;
					if (GetKey($obj, 'from_hour1')||GetKey($obj, 'from_minute1')||GetKey($obj, 'to_hour1')||GetKey($obj, 'to_minute1')) {
						$h1 = sprintf("%02d", GetKey($obj, 'from_hour1'));
						$m1 = sprintf("%02d", GetKey($obj, 'from_minute1'));
						$h2 = sprintf("%02d", GetKey($obj, 'to_hour1'));
						$m2 = sprintf("%02d", GetKey($obj, 'to_minute1'));
						push @section, {	CP2 => 1, CP4 => 1, 
											OBJ_VAL1 => html('From:'),
											OBJ_VAL2 => $h1.':'.$m1,
											OBJ_VAL3 => html('To:'),
											OBJ_VAL4 => $h2.':'.$m2 };
					}
					if (GetKey($obj, 'from_hour2')||GetKey($obj, 'from_minute2')||GetKey($obj, 'to_hour2')||GetKey($obj, 'to_minute2')) {
						$h1 = sprintf("%02d", GetKey($obj, 'from_hour2'));
						$m1 = sprintf("%02d", GetKey($obj, 'from_minute2'));
						$h2 = sprintf("%02d", GetKey($obj, 'to_hour2'));
						$m2 = sprintf("%02d", GetKey($obj, 'to_minute2'));
						push @section, {	CP2 => 1, CP4 => 1, 
											OBJ_VAL1 => html('From:'),
											OBJ_VAL2 => $h1.':'.$m1,
											OBJ_VAL3 => html('To:'),
											OBJ_VAL4 => $h2.':'.$m2 };
					}
					if (GetKey($obj, 'from_hour3')||GetKey($obj, 'from_minute3')||GetKey($obj, 'to_hour3')||GetKey($obj, 'to_minute3')) {
						$h1 = sprintf("%02d", GetKey($obj, 'from_hour3'));
						$m1 = sprintf("%02d", GetKey($obj, 'from_minute3'));
						$h2 = sprintf("%02d", GetKey($obj, 'to_hour3'));
						$m2 = sprintf("%02d", GetKey($obj, 'to_minute3'));
						push @section, {	CP2 => 1, CP4 => 1, 
											OBJ_VAL1 => html('From:'),
											OBJ_VAL2 => $h1.':'.$m1,
											OBJ_VAL3 => html('To:'),
											OBJ_VAL4 => $h2.':'.$m2 };
					}
					push @{$tobj[$cnt]{TIME_GENERAL}}, { 
						S1 => \@section,
						SF_END => 1 };
				}
	
			} elsif ($type eq 'Scheduled Event') {
				$section = Section($Policy, Object2Page($Policy, 'Times'), 'Times', 'Time of event');
				if ($section) {
					push @{$tobj[$cnt]{TIME_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('Time of Event'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
	
					$h = sprintf("%02d", GetKey($obj, 'hour'));
					$m = sprintf("%02d", GetKey($obj, 'minutes'));
					push @{$tobj[$cnt]{TIME_GENERAL}}, {
						S1 =>	[{	CP3 => 1, 
									OBJ_VAL1 => (GetKey($obj, 'days_specification') eq 'daily')?$MarkDot:$Mark,
									OBJ_VAL2 => html('At (hh:mm):'),
									OBJ_VAL3 => (GetKey($obj, 'days_specification') eq 'daily')?($h.':'.$m):'' }]
					};
		
					$var = GetKey($obj, 'every_seconds');
					$h=sprintf("%02d", int($var/3600));
					$m=sprintf("%02d", int(($var-$h*3600)/60));
					$s=sprintf("%02d", ($var-$h*3600-$m*60));
					push @{$tobj[$cnt]{TIME_GENERAL}}, {
						S1 =>	[{	CP3 => 1, 
									OBJ_VAL1 => (GetKey($obj, 'days_specification') eq 'seconds')?$MarkDot:$Mark,
									OBJ_VAL2 => html('Every:'),
									OBJ_VAL3 => (GetKey($obj, 'days_specification') eq 'seconds')?($h.'h '.$m.'m '.$s.'s'):'' }]
					};
		
					push @{$tobj[$cnt]{TIME_GENERAL}}, { SF_END => 1 };
				}
	
			} elsif ($type =~ /Group/) {
				# Group Members ...
				$section = Section($Policy, Object2Page($Policy, 'Times'), 'Times', 'Group Members');
				if ($section) {
					push @{$tobj[$cnt]{TIME_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('Members'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					my @section;
					@var = GetMembers($obj);
					if (@var) {
						foreach $var (@var) {
							%var = %{obj2html($Policy, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
							push @section, { 	C1 => 1, CP1 => 1, 
												OBJ_VAL1 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
						}
					} else {
						push @section, {	C1 =>1, 
											OBJ_VAL1 => $None };
					}
					push @{$tobj[$cnt]{TIME_GENERAL}}, { 
						S1 => \@section,
						SF_END => 1 
					}
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{TIME_GENERAL}}) { push @{$tobj[$cnt]{TIME_GENERAL}}, { VOID => 1 } }
			
			#------------------------------------------------
			# DAYS tab/column
			#------------------------------------------------
			if ($type eq 'Time') {
				$section = Section($Policy, Object2Page($Policy, 'Times'), 'Times', 'Days');
				if ($section) {
					push @{$tobj[$cnt]{TIME_DAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Days'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					$val = stripquotes(GetKey($obj, 'days_specification'));
					push @{$tobj[$cnt]{TIME_DAYS}}, { 
						TXT => 	[{	TEXT => html('Days specification'),
									CLASS => 'note_xspc' }],
						S1 => 	[{	OBJ_VAL1 => (($val eq 'none')?$MarkDot:$Mark).' None',
									OBJ_VAL2 => (($val eq 'by day in month')?$MarkDot:$Mark).' Day in month',
									OBJ_VAL3 => (($val eq 'by day in week')?$MarkDot:$Mark).' Day in week' }],
					};
										
					if ($val eq 'by day in month') {
						$var = join(', ', GetMembers(GetKey($obj, 'day_of_month')));
						$val = join(', ', GetMembers(GetKey($obj, 'month')));
						push @{$tobj[$cnt]{TIME_DAYS}}, { 
							SUBSECT_BEG => html('Days in month'), 
							S1 => 	[{	CP2 => 1,
										OBJ_VAL1 => html('Days:'),
										OBJ_VAL2 => ($var)?$var:'' },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Month:'),
										OBJ_VAL2 => ($val)?$Month{$val}:html('Any') }],
							SUBSECT_END => 1
						};
						
					} elsif ($val eq 'by day in week') {
						$var = join(', ', GetMembers(GetKey($obj, 'day_of_week')));
						push @{$tobj[$cnt]{TIME_DAYS}}, { 
							SUBSECT_BEG => html('Days in week'), 
							S1 => 	[{	CP2 => 1,
										OBJ_VAL1 => html('Days:'),
										OBJ_VAL2 => ($var)?$Month{$var}:'' }],
							SUBSECT_END => 1
						};
					}
					push @{$tobj[$cnt]{TIME_DAYS}}, { SF_END => 1 };
				}
					
			}
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{TIME_DAYS}}) { push @{$tobj[$cnt]{TIME_DAYS}}, { VOID => 1 } }
			
	
			#------------------------------------------------
			# COMMENTS tab/column of Resource
			#------------------------------------------------
			$tobj[$cnt]{TIME_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
		}
			
		$template->param(TIMES=>\@tobj);
	}
	
	##########################################################################
	# Routine: communities2html
	#
	# Description:
	# Routine to create the VPN Communities table
	##########################################################################
	sub communities2html {
		use strict;
		my (@ObjList, @tobj);
		my ($cnt, $obj, $name, $section);
		my ($topo, $type, $current_type);
		my (@var, %var, $var, $val);
		my (@tmp);
	
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
		
		#----------------------------------------------------------------
		#---                 COMMUNITIES to HTML                      ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable.':'.$name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{COMM_TYPE} = $type;
				$current_type = $type;
			}
	
			$tobj[$cnt]{COMM_TEXT} = stripquotes($name);
			$tobj[$cnt]{COMM_TYPE} = $type;
			$tobj[$cnt]{COMM_NAME} = HTMLObjectList($Policy, $obj, $name);
			
			#------------------------------------------------
			# GENERAL tab/column of Community
			#------------------------------------------------
			# General Properties
			if ($type =~ /Site to Site/) {
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'General');
				if ($section) {
					# Community Traffic Security Policy
					push @{$tobj[$cnt]{COMM_GENERAL}}, (
						{ 	SF_BEG => [{	SECTION  => html('General'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }]
						}
					);
					if ($type eq 'Site to Site - Star') {
						push @{$tobj[$cnt]{COMM_GENERAL}}, (
							{	SUBSECT_BEG => html('Enable VPN routing for satellites'),
								S1 => 	[{	OBJ_VAL1 => (GetKey($obj, 'route_through_center') eq 'true')?$MarkDot:$Mark,
											OBJ_VAL2 => html('To center only') },
										 {	OBJ_VAL1 => (GetKey($obj, 'route_through_center') eq 'to_other_satellite')?$MarkDot:$Mark,
											OBJ_VAL2 => html('To center and to other satellites through center') },
										 {	OBJ_VAL1 => (GetKey($obj, 'route_through_center') eq 'by_default')?$MarkDot:$Mark,
											OBJ_VAL2 => html('To center, or through the center to other satellites,;').
														html('to internet and other VPN targets') }],
								SUBSECT_END => 1
							}
						);
					}				
					push @{$tobj[$cnt]{COMM_GENERAL}}, (
						{	SUBSECT_BEG => html('Community Traffic Security Policy'),
							S1 => [{	OBJ_VAL1 => (GetKey($obj, 'allow_all_encrypted_traffic') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Accept all encrypted traffic') }],
							SUBSECT_END => 1,
							SF_END => 1
						}
					);
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{COMM_GENERAL}}) { push @{$tobj[$cnt]{COMM_GENERAL}}, { VOID => 1 } }
			
			#------------------------------------------------
			# GATEWAYS tab/column of Community
			#------------------------------------------------
			# Participating Gateways ...
			@var = GetMembers(GetKey($obj, 'participant_gateways'));
	
			if (($type eq 'Site to Site - Meshed') || ($type eq 'Remote Access')) {
				# Participating Gateways ...
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Participating Gateways');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'participant_gateways'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Participating Gateways'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var,
							SF_END => 1
						}
					);
				}
			} elsif ($type eq 'Site to Site - Star') {
				# Central Gateways ...
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Central Gateways');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'participant_gateways'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Central Gateways'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var },
						{ 	S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'meshed_in_center') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Mesh center gateways') }],
							SF_END => 1 }
					);
				}
				# Satellite Gateways ...
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Satellite Gateways');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'satellite_gateways'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Satellite Gateways'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var,
							SF_END => 1
						}
					);
				}
			} elsif ($type eq 'Extranet Manager') {
				# Extranet Partners ... n/a in NG AI R55 
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Extranet Partners');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'extranet_partners'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Extranet Partners'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var,
							SF_END => 1
						}
					);
				}
			}
	
			if ($type eq 'Remote Access') {
				# Participating User Groups ...
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'User Groups');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'participant_users_groups'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('User Groups'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var,
							SF_END => 1
						}
					);
				}
			}
	
			if ($type =~ /Site to Site/) {
				# Services in the clear
				$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Excluded Services');
				if ($section) {
					$var = HTMLMemberList($Policy, GetKey($obj, 'exclude_srv'), $_HREF);
					if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
					push @{$tobj[$cnt]{COMM_GATEWAYS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Excluded Services'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							OBJLIST => $var,
							SF_END => 1
						}
					);
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{COMM_GATEWAYS}}) { push @{$tobj[$cnt]{COMM_GATEWAYS}}, { VOID => 1 } }
	
			#------------------------------------------------
			# PROPS tab/column of Community
			#------------------------------------------------
			# VPN Settings
			$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'VPN Properties');
			if ($section) {
				if (($type =~ /Site to Site/) || ($type eq 'Extranet Partner')) {
					# VPN Settings phase 1
					push @{$tobj[$cnt]{COMM_PROPS}}, (
						{ 	SF_BEG => [{	SECTION  => html('VPN Properties'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }],
							SUBSECT_BEG => html('IKE (Phase 1) Properties'), 
							S1 => 	[{	CP2 => 1,
										OBJ_VAL1 => html('Perform key exchange encryption with:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_enc_alg'))) },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Perform data integrity with:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_hash_alg'))) }],
							SUBSECT_END => 1
						}
					);
	
					if ($type =~ /Site to Site/) {
						# VPN Settings phase 2
						push @{$tobj[$cnt]{COMM_PROPS}}, { 
							SUBSECT_BEG => html('IPSEC (Phase 2) Properties'), 
							S1 => 	[{	CP2 => 1,
										OBJ_VAL1 => html('Perform IPSEC data encryption with:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_enc_alg'))) },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Perform data integrity with:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_hash_alg'))) }],
							SUBSECT_END => 1
						};
					}
					push @{$tobj[$cnt]{COMM_PROPS}}, { SF_END => 1 };
				}
			}
	
			#------------------------------------------------
			# ADVANCED Properties tab/column of Community
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Communities'), 'Communities', 'Advanced Properties');
			if ($section) {
				# Advanced VPN Settings phase 1
				if (($type =~ /Site to Site/) || ($type eq 'Extranet Partner')) {
					push @{$tobj[$cnt]{COMM_PROPS}}, (
						{ 	SF_BEG => [{	SECTION  => html('Advanced Properties'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }] ,
							SUBSECT_BEG => html('IKE (Phase 1)'),
							S1 =>	[{	CP2 => 1, C2 => 2,
										OBJ_VAL1 => html('Use Diffie-Hellman group:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_dh_grp:Name'))) },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Renegotiate IKE SA every:'),
										OBJ_VAL2 => html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_rekey_time'))),
										OBJ_VAL3 => html('minutes') }]
						}
					);
		
					if ($type =~ /Site to Site/) {
						push @{$tobj[$cnt]{COMM_PROPS}}, { 
							S1 => 	[{	OBJ_VAL1 => (GetKey($obj, 'ike_p1:ike_p1_use_aggressive') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Use aggressive mode') }]
						};
					}
					push @{$tobj[$cnt]{COMM_PROPS}}, { SUBSECT_END => 1 };
	
					if ($type =~ /Site to Site/) {
						# Advanced VPN Settings phase 2
						push @{$tobj[$cnt]{COMM_PROPS}}, (
							{ SUBSECT_BEG => html('IPSEC (Phase 2)'),
							  S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'ike_p2:ike_p2_use_pfs') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Use Perfect Forward Secrecy') },
									 {	INACTIVE => (GetKey($obj, 'ike_p2:ike_p2_use_pfs') eq 'false'),
										CP3 => 1, C3 => 2,
										OBJ_VAL1 => '',
										OBJ_VAL2 => html('Use Diffie-Hellman group:'),
										OBJ_VAL3 => html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_pfs_dh_grp:Name'))) },
									 {	CP2 => 1, C1 => 2,
										OBJ_VAL1 => html('Renegotiate IPSEC SA every:'),
										OBJ_VAL2 => html(GetKey($obj, 'ike_p2:ike_p2_rekey_time')),
										OBJ_VAL3 => html('seconds') },
									 {	C2 => 3,
										OBJ_VAL1 => (GetKey($obj, 'ike_p2:ike_p2_ipcomp') eq 'DEFLATE')?$CheckTick:$Check,
										OBJ_VAL2 => html('Support Site to Site IP compression') }],
							  SUBSECT_END => 1 }
						);
					}
					# Advanced Properties NAT
					if ($type =~ /Site to Site/) {
						push @{$tobj[$cnt]{COMM_PROPS}}, (
							{ SUBSECT_BEG => html('NAT'),
							  S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'disable_NAT') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Disable NAT inside the VPN community') }],
							  SUBSECT_END => 1 }
						);
					}
					push @{$tobj[$cnt]{COMM_PROPS}}, { SF_END => 1 };
				}
			}
	
			# Check for empty cell and correct if necessary ...
			if (!defined @{$tobj[$cnt]{COMM_PROPS}}) { push @{$tobj[$cnt]{COMM_PROPS}}, { VOID => 1 } }
			
			#------------------------------------------------
			# COMMENTS tab/column of Community
			#------------------------------------------------
			$tobj[$cnt]{COMM_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
				
		}
			
		$template->param(COMMUNITIES=>\@tobj);
	}
	
	##########################################################################
	# Routine: users2html
	#
	# Description:
	# Routine to create the USERS table
	##########################################################################
	sub users2html {
		use strict;
		my ($obj, $name, $type, $current_type);
		my ($cnt, @tobj, @tint, $intcnt, $iobj);
		my ($var, %var, @var, $val, $section, $_val);
		
		my $Policy   = shift;
		my $template = shift;
		my $CPtable  = shift;
		my $obj      = shift;
		my @ObjList  = @{$obj};
	
		#----------------------------------------------------------------
		#---                     USERS to HTML                        ---
		#----------------------------------------------------------------
		$cnt = -1;
		$current_type = '';
		foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
			$cnt += 1;
			(undef, $type, $name) = split(/:/, $name);
			$obj  = GetKey($CPtable, $name);
	
			if ($type ne $current_type) {
				$tobj[$cnt]{TYPE} = 1;
				$tobj[$cnt]{OBJ_TYPE} = $type;
				$current_type = $type;
			}
			
			$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($Policy, $obj, $name);
	
			#------------------------------------------------
			# Expiration Date of User
			#------------------------------------------------
			$tobj[$cnt]{OBJ_EXPDATE} = html(stripquotes(GetKey($obj, 'expiration_date')));
			
			#------------------------------------------------
			# GROUPS tab of User
			#------------------------------------------------
			if (@{HTMLMemberList($Policy, $obj->{groups}, $_HREF)}) {
				push @{$tobj[$cnt]{OBJ_GROUPS}}, {
					OBJLIST => HTMLMemberList($Policy, $obj->{groups}, $_HREF)
				};
			}
					
			if (!defined @{$tobj[$cnt]{OBJ_GROUPS}}) { push @{$tobj[$cnt]{OBJ_GROUPS}}, { VOID => 1 } }
			
			#------------------------------------------------
			# PROPERTIES of User
			#------------------------------------------------
			$section = Section($Policy, Object2Page($Policy, 'Users'), 'Users', 'Authentication');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_PROP}}, (
					{	SF_BEG => [{	SECTION  => html('Authentication'),
										OBJ_NAME => $name,
										STATE	 => ($section eq $_OPEN) }] },
					{	S1	=>	[{	CP2 => 1,
									OBJ_VAL1 => html('Authentication Scheme:'),
									OBJ_VAL2 => $AuthScheme{stripquotes(GetKey($obj, 'auth_method'))} }],
					}
				);
	
				if (GetKey($obj, 'auth_method') eq 'radius') {
					%var = %{obj2html($Policy, GetKey($obj, 'radius_server:Table'), GetKey($obj, 'radius_server:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_PROP}}, (
						{ S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('RADIUS Server:'), 
										OBJ_VAL2 => $var }] }
					);
				} elsif (GetKey($obj, 'auth_method') eq 'tacacs') {
					%var = %{obj2html($Policy, GetKey($obj, 'tacacs_server:Table'), GetKey($obj, 'tacacs_server:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{OBJ_PROP}}, (
						{ S1 => 	[{ 	CP2 => 1, 
										OBJ_VAL1 => html('TACACS Server:'), 
										OBJ_VAL2 => $var }] }
					);
				}
							
				push @{$tobj[$cnt]{OBJ_PROP}}, { SF_END => 1 };
			}
	
			$section = Section($Policy, Object2Page($Policy, 'Users'), 'Users', 'Location');
			if ($section) {
				push @{$tobj[$cnt]{OBJ_PROP}}, (
					{	SF_BEG => [{	SECTION  => html('Location'),
									OBJ_NAME => $name,
									STATE    => ($section eq $_OPEN) }] }
			);

			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_BEG => 1 };

			$var = HTMLMemberList($Policy, GetKey($obj, 'sources'), $_HREF);
			if (!@{$var}) { $var = [{ OBJ_TEXT => '' }]	};
			push @{$tobj[$cnt]{OBJ_PROP}}, { 
				SUBSECT_BEG => html('Sources'), 
				OBJLIST => $var,
				SUBSECT_END => 1
			};

			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_NXT => 1 };

			$var = HTMLMemberList($Policy, GetKey($obj, 'destinations'), $_HREF);
			if (!@{$var}) { $var = [{ OBJ_TEXT => '' }]	};
			push @{$tobj[$cnt]{OBJ_PROP}}, { 
				SUBSECT_BEG => html('Destinations'), 
				OBJLIST => $var,
				SUBSECT_END => 1
			};

			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_END => 1 };

			push @{$tobj[$cnt]{OBJ_PROP}}, { SF_END => 1 };
		}

		$section = Section($Policy, Object2Page($Policy, 'Users'), 'Users', 'Time');
		if ($section) {
			push @{$tobj[$cnt]{OBJ_PROP}}, (
				{	SF_BEG => [{	SECTION  => html('Time'),
									OBJ_NAME => $name,
									STATE    => ($section eq $_OPEN) }] }
			);

			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_BEG => 1 };

			$val = GetKey($obj, 'days');
			push @{$tobj[$cnt]{OBJ_PROP}}, (
				{ 	SUBSECT_BEG => html('Day in week:'), 
					S1 => 	[{ 	OBJ_VAL1 => (($val & $Day{Monday})?$CheckTick:$Check),
								OBJ_VAL2 => html('Monday') },
							 {	OBJ_VAL1 => (($val & $Day{Tuesday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Tuesday') },
							 {	OBJ_VAL1 => (($val & $Day{Wednesday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Wednesday') },
							 {	OBJ_VAL1 => (($val & $Day{Thursday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Thursday') },
							 {	OBJ_VAL1 => (($val & $Day{Friday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Friday') },
							 {	OBJ_VAL1 => (($val & $Day{Saturday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Saturday') },
							 {	OBJ_VAL1 => (($val & $Day{Sunday})?$CheckTick:$Check),
							 	OBJ_VAL2 => html('Sunday') }],
					SUBSECT_END => 1
				}
			);
			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_NXT => 1 };

			push @{$tobj[$cnt]{OBJ_PROP}}, (
				{ 	SUBSECT_BEG => html('Time of day:'), 
					S1 	=>	[{	CP2 => 1,
								OBJ_VAL1 => html('From:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'fromhour'))) },
							 {	CP2 => 1,
							 	OBJ_VAL1 => html('To:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'tohour'))) }],
					SUBSECT_END => 1
				}
			);
			push @{$tobj[$cnt]{OBJ_PROP}}, { COL_END => 1 };

			push @{$tobj[$cnt]{OBJ_PROP}}, { SF_END => 1 };
		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_PROP}}) { push @{$tobj[$cnt]{OBJ_PROP}}, { VOID => 1 } }

		#------------------------------------------------
		# ADVANCED column of User
		#------------------------------------------------
		# Permission profile ...
		if ($type eq 'Administrator') {
			%var = %{obj2html($Policy, GetKey($obj, 'administrator_profile:Table'), GetKey($obj, 'administrator_profile:Name'), $_HREF)};
			push @{$tobj[$cnt]{OBJ_ADV}}, { 
				S1	=> 	[{	CP2 => 1,
							OBJ_VAL1 => html('Permissions Profile:'),
							OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT}
						}]
			}
		};
		
		$section = Section($Policy, Object2Page($Policy, 'Users'), 'Users', 'Encryption');
		if ($section) {
			push @{$tobj[$cnt]{OBJ_ADV}}, (
				{	SF_BEG => [{	SECTION  => html('Encryption'),
									OBJ_NAME => $name,
									STATE    => ($section eq $_OPEN) }]
				}
			);

			$var = GetKey($obj, 'userc:IKE');
			push @{$tobj[$cnt]{OBJ_ADV}}, (
				{	S1	=>	[{	OBJ_VAL1 => $var?$CheckTick:$Check,
								OBJ_VAL2 => html('IKE') }]
				}
			);
			if ($var) {
				@var = GetMembers(GetKey($obj, 'userc:isakmp.authmethods'));
				$_val = 0;
				foreach $val (@var) {
					if ($val eq 'pre-shared') { $_val += 1 }
					if ($val eq 'signatures') { $_val += 2 }
				}
				push @{$tobj[$cnt]{OBJ_ADV}}, (
					{	SUBSECT_BEG => html('Authentication'),
						S1	=>	[{	OBJ_VAL1 => ($_val & 1)?$CheckTick:$Check,
									OBJ_VAL2 => html('Password (Pre-shared secret)') },
								 {	OBJ_VAL1 => ($_val & 2)?$CheckTick:$Check,
									OBJ_VAL2 => html('Public Key') }],
						SUBSECT_END => 1
					}
				);
				if (GetKey('properties:firewall_properties:force_encryption_on_all_users') eq 'true') {
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{	SUBSECT_BEG => html('Encryption'),
							TXT	=>	[{	CLASS => 'note',
										TEXT  => html('The Encryption Properties defined in the;').
												 html('Remote Access VPN page of the Global Properties;').
												 html('window will be applied to this user.') }],
							SUBSECT_END => 1
						}
					);
				} else {
					push @{$tobj[$cnt]{OBJ_ADV}}, (
						{	SUBSECT_BEG => html('Encryption') },
						{	S1 => 	[{	OBJ_VAL1 => (GetKey($obj, 'userc:use_global_encryption_values') eq 'true')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Defined in the Remote Access VPN page of the;').
													html('Global Properties window') }] },
						{	S1 =>	[{	C2 => 3,
									 	OBJ_VAL1 => (GetKey($obj, 'userc:use_global_encryption_values') eq 'false')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Defined below') },
									 {	C3 => 2, CP3 => 1,
									 	OBJ_VAL2 => html('Encryption  Algorithm:'),
									 	OBJ_VAL3 => html(GetKey($obj, 'userc:isakmp.encryption')) },
									 {	OBJ_VAL2 => html('Data Integrity:'),
									 	OBJ_VAL3 => (GetKey($obj, 'userc:isakmp.data.integrity') eq 'SHA1')?$MarkDot:$Mark,
									 	OBJ_VAL4 => html('SHA1') },
									 {	C1 => 2,
									 	OBJ_VAL2 => (GetKey($obj, 'userc:isakmp.data.integrity') eq 'MD5')?$MarkDot:$Mark,
									 	OBJ_VAL3 => html('MD5') }] },
						{	SUBSECT_END => 1 }
					);
				}
			}
			push @{$tobj[$cnt]{OBJ_ADV}}, { SF_END => 1 };
		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_ADV}}) { push @{$tobj[$cnt]{OBJ_ADV}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}

	$template->param(USERS=>\@tobj);
}


##########################################################################
# Routine: usergroups2html
#
# Description:
# Routine to create the USERGROUPS table
##########################################################################
sub usergroups2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $name);
	my ($type, $current_type);

	my $Policy   = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	#----------------------------------------------------------------
	#---                   USERGROUPS to HTML                     ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($Policy, $obj, $name);
		$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($Policy, $obj, $_HREF);

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(USERGROUPS=>\@tobj);
}

sub usertemplates2html {
	print "usertemplates2html\n";
}
} #Official end of table2html (including subroutines)

1;
