package settings2html;

use strict;
use CheckPoint;
use Constants;
use Options;
use object2html;
use Strings;

require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(HTMLTargets HTMLGlobalProperties);


##########################################################################
# Routine: HTMLTargets
#
# Description:
# Write POLICY INSTALLATION TARGETS to HTML file. 
#
# Parameters:
# $rulebase	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 
##########################################################################
sub HTMLTargets {
	use strict;
	my $rb_name;
	my @targets;
	
	my $rulebase = shift;
	my $template = shift;	# TARGETS.TMPL
	
	# Determine rulebase name ...
	$rb_name = GetKey($rulebase, 'collection:Name');
	
	# Fill header with general variables of rulebase
	$template->param(	RULEBASE     => $rb_name,
						CSSPATH      => Option('CSSPath'),
				);

	if (GetKey('policies_collections', $rb_name.':all_internal_modules') eq 'true') {
		@targets = (txt2html('All internal modules', 'any_object', 'black'));
	} else {
		@targets = @{HTMLMemberList($rb_name, GetKey('policies_collections:'.$rb_name.':installable_targets'), $_HREF)};
	}

	$template->param(TARGETS => \@targets );

	return $template;
}

##########################################################################
# Routine: HTMLGlobalProperties
#
# Description:
# Write GLOBAL PROPERTIES to HTML file. 
#
# Parameters:
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 
##########################################################################
sub HTMLGlobalProperties {
	use strict;
	my ($table, @classes);
	my ($PrimaryMgmt, @SecondaryMgmt);	
	my ($obj, $rb_name, $name);
	my ($var, %var, @var, $val);
	my ($v1, $v2, $v3, $v4, $v5);
	
	my $template = shift;

	# Initialize special marker signs ...
	my $Check      = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/check.png>';
	my $CheckTick  = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/checktick.png>';
	my $Mark       = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/mark.png>';
	my $MarkDot    = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/markdot.png>';
	my $Spacer     = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/spacer.png>';
	my $None       = '<I>(none)</I>';

    # Determine Check Point table name to be searched ...	
	$table = 'network_objects';

	# Check all objects in network_objects table ...
	if (ref(GetKey($table))) {
		# table is not empty ...
		foreach $name (keys %{GetKey($table)}) {
			if (GetKey($table, $name, 'primary_management') eq 'true') {
				$PrimaryMgmt = $name
			} elsif (GetKey($table, $name, 'management') eq 'true') {
				push @SecondaryMgmt, $name
			}
		}
	}

	# Define global variables in template file ...		
	$template->param(	CSSPATH 					=> Option('CSSPath'),
						SCRIPTPATH					=> Option('ScriptPath'),
						MAINLOGO					=> Option('MainLogo'),
						SECTION_GLOBALPROPERTIES	=> 1,
						GLOBALPROPERTIES_REF		=> 'GLOBALPROPERTIES',
						PRI_MGMT     				=> $PrimaryMgmt,
						SEC_MGMT					=> join("; ", @SecondaryMgmt)
					);

	$table = GetKey('properties:firewall_properties');
		
	# ----------------------------------
	# CLASS: Firewall-1 ...
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Firewall-1';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='FW';
	my @props;
		push @props, (
			{ TXT => [{	CLASS => 'note_xspc',
					 	TEXT => html('The following properties define the position of the rules in the Rule Base:') }] }
		 );
        	
		push @props, (
			{ S1 =>	[{	CP3 => 1,
					 	OBJ_VAL1 => (GetKey($table, 'fw1enable') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept VPN-1 & FireWall-1 control connections:'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'fw1_enable_p'))} },
       				 { 	CP3 => 1, 	
        				OBJ_VAL1 => '',
        				OBJ_VAL2 => ((GetKey($table, 'raccessenable') eq 'true')?$CheckTick:$Check).
        							html(' Accept Remote Access control connections:'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'r_access_enable_p'))} },
       				 {	CP3 => 1,
        				OBJ_VAL1 => (GetKey($table, 'outgoing') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept outgoing packets originating from Gateway:'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'outgoing_p'))} },
        			 {	CP3 => 1, 	
        				OBJ_VAL1 => (GetKey($table, 'rip') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept RIP:'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'rip_p'))} },
       				 { 	CP3 => 1, 	
        				OBJ_VAL1 => (GetKey($table, 'domain_udp') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept Domain Name over UDP (Queries):'),
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'domain_udp_p'))} },
        			 {	CP3 => 1, 	
        				OBJ_VAL1 => (GetKey($table, 'rip') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept Domain Name over TCP (Zone Transfer):'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'rip_p'))} },
        			 {	CP3 => 1,
        				OBJ_VAL1 => (GetKey($table, 'icmpenable') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept ICMP requests:'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'icmpenable_p'))} },
        			 {	CP3 => 1, 	
        				OBJ_VAL1 => (GetKey($table, 'cpridenable') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html('Accept CPRID connections (SmartUpdate):'), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'cpridenable_p'))} },
        			 {	CP3 => 1, 	
        				OBJ_VAL1 => (GetKey($table, 'dagdhcpenable') eq 'true')?$CheckTick:$Check,
        				OBJ_VAL2 => html("Accept dynamic address Modules'DHCP traffic:"), 
        				OBJ_VAL3 => $Position{stripquotes(GetKey($table, 'dagdhcpenable_p'))} }] }
        );
        
		push @props, (
			{ S0 => html('Track') },
			{ S1 =>	[{	CP1 => 1,
					 	OBJ_VAL1 => (GetKey($table, 'log_implied_rules') eq 'true')?$CheckTick:$Check,
					 	OBJ_VAL2 => html("Log Implied Rules") }] }
		);
	
	$class{GP_PROPS}=\@props;

	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: Firewall-1 - Security Server ...
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Firewall-1';
	$class{GP_SUBCLASS}='Security Server';
	$class{GP_CLASS}='FW-SS';

	my @props;
		push @props, 
			{ S1 => [{ 	CP2 => 1,
						OBJ_VAL1 => html('Telnet welcome message file: '), 
        				OBJ_VAL2 => stripquotes(GetKey($table, 'telnet_msg')) },
					 { 	CP2 => 1,
        				OBJ_VAL1 => html('FTP welcome message file: '), 
        				OBJ_VAL2 => stripquotes(GetKey($table, 'ftp_msg')) },
					 { 	CP2 => 1,
        				OBJ_VAL1 => html('Rlogin welcome message file: '), 
        				OBJ_VAL2 => stripquotes(GetKey($table, 'rlogin_msg')) },
					 { 	CP2 => 1,
        				OBJ_VAL1 => html('Client authentication welcome file: '), 
        				OBJ_VAL2 => stripquotes(GetKey($table, 'clnt_auth_msg')) },
					 { 	CP2 => 1,
        				OBJ_VAL1 => html('SMTP welcome message: '), 
        				OBJ_VAL2 => stripquotes(GetKey($table, 'smtp_msg')) }] };
        
		push @props, (
			{ 	SUBSECT_BEG => html('HTTP:') },
			{	S1 => 	[{	OBJ_VAL1 => html('Note:'),
							OBJ_VAL2 => html('These settings apply only to firewalled gateways prior to NG.;').
										html('For later versions, these settings can be configured in the Node;').
										html('Properties.') }] },
			{	S1 => 	[{	CP2 => 1, CP4 => 1,
							OBJ_VAL1 => html('HTTP next proxy:'),
							OBJ_VAL2 => GetKey($table, 'http_next_proxy_host'),
							OBJ_VAL3 => html('Port:'),
							OBJ_VAL4 => GetKey($table, 'http_next_proxy_port') }] },
			{ 	SUBSECT_END => 1 }
							
		);
							
		push @props, { S0 => html('HTTP Server') };
		my @section;
        	push @section, { 	HEAD => 1,
        						OBJ_VAL1 => html('Logical Name'), 
        						OBJ_VAL2 => html('Host'),
        						OBJ_VAL3 => html('Port'),
        						OBJ_VAL4 => html('Reauthentication')};

			foreach $var (keys %{GetKey($table, 'http_servers')}) {
				if (GetKey($table, 'http_servers:'.$var.':AdminInfo:ClassName') eq 'http_server') {
    				$obj=GetKey($table, 'http_servers:'.$var);
    	        	push @section, {	OBJ_VAL1 => html($var), 
        	    						OBJ_VAL2 => html(GetKey($obj, 'host')),
            							OBJ_VAL3 => html(GetKey($obj, 'port')),
            							OBJ_VAL4 => html(GetKey($obj, 'reauth_requests'))};
				}
			}
		push @props, { T1 => \@section, C4 => 1 };
			
	
	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: Firewall-1 - VoIP ...
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Firewall-1';
	$class{GP_SUBCLASS}='VoIP';
	$class{GP_CLASS}='FW-VI';

	my @props;
		push @props, 
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'h323_log_conn') eq 'true')?$CheckTick:$Check,
					 	OBJ_VAL2 => html('Log VoIP connection') }] }; 

		push @props, (
			{ S0 => html('SIP') },
    		{ S1 =>	[{	OBJ_VAL1 => (GetKey($table, 'sip_allow_redirect') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Allow calls using a proxy or a redirect server') },
    				 {	OBJ_VAL1 => (GetKey($table, 'sip_allow_instant_messages') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Allow SIP-based Instant Messaging') },
    				 {	OBJ_VAL1 => (GetKey($table, 'sip_enforce_security_reinvite') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Allow the destination to re-invite calls') }] },
    		{ S1 =>	[{	CP2 => 1,
    			 		OBJ_VAL1 => html('Maximum invitations per call (from both directions):'),
    					OBJ_VAL2 => GetKey($table, 'sip_max_reinvite') }] }
    	);

		push @props, (
			{ S0 => html('H.323') },
    		{ S1 =>	[{	OBJ_VAL1 => (GetKey($table, 'fwh323_allow_redirect') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Allow to re-direct connections') },
    				 {	OBJ_VAL1 => (GetKey($table, 'sip_allow_instant_messages') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Prevent blank source phone numbers for gatekeeper connections') },
    				 {	OBJ_VAL1 => (GetKey($table, 'allow_h323_t120') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Enable dynamic T.120') }] }
    	);

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: Network Address Translation ...
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='NAT - Network Address Translation';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='NAT';

	my @props;
		push @props, (
			{ S0 => html('Automatic NAT rules') },
    		{ S1 =>	[{	OBJ_VAL1 => (GetKey($table, 'nat_automatic_rules_merge') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Allow bi-directional NAT') },
    				 {	OBJ_VAL1 => (GetKey($table, 'nat_dst_client_side') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Translate destination on client side') },
    				 {	OBJ_VAL1 => (GetKey($table, 'nat_automatic_arp') eq 'true')?$CheckTick:$Check,
    			 		OBJ_VAL2 => html('Automatic ARP configuration') }] }
    	);

		push @props, (
			{ S0 => html('Manual NAT rules') },
			{ S1 =>	[{ 	OBJ_VAL1 => (GetKey($table, 'nat_dst_client_side_manual') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Translate destination on client side') }] }
		);

		$v1 = stripquotes(GetKey($table, 'ip_pool_log'));
		$v2 = stripquotes(GetKey($table, 'log_ip_pool_allocation'));
		push @props, (
			{ S0 => html('IP Pool NAT') },
			{ S1 =>	[{ 	C2 => 4,
						OBJ_VAL1 => (GetKey($table, 'enable_ip_pool') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Enable IP Pool NAT for SecuRemote/SecureClient and gateway to gateway connections') }] },
			{ S1 =>	[{ 	OBJ_VAL1 => '',
					 	OBJ_VAL2 => html('Address exhaustion track:'),
						OBJ_VAL3 => (($v1 eq 'none')?$MarkDot:$Mark).html(' None'), 
						OBJ_VAL4 => (($v1 eq 'IP Exhaustion Log')?$MarkDot:$Mark).html(' Log'), 
						OBJ_VAL5 => (($v1 eq 'IP Exhaustion Alert')?$MarkDot:$Mark).html(' Alert') },
					 {	OBJ_VAL1 => '',
					 	OBJ_VAL2 => html('Address allocation and release track:'),
						OBJ_VAL3 => (($v2 eq 'none')?$MarkDot:$Mark).html(' None'), 
						OBJ_VAL4 => (($v2 eq 'IP Allocation Log')?$MarkDot:$Mark).html(' Log') }] }
		);

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: Authentication
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Authentication';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='AUTH';

	my @props;
		push @props, (
			{ S0 => html('Failed authentication attempts') },
        	{ S1 =>	[{	CP2 => 1,
        			 	OBJ_VAL1 => html('Terminate rlogin connection after'),
        				OBJ_VAL2 => html(GetKey($table, 'rlogin_max_auth_allowed')),
        				OBJ_VAL3 => html('attempts') },
        			 {	CP2 => 1,
        			  	OBJ_VAL1 => html('Terminate telnet connection after'),
        				OBJ_VAL2 => html(GetKey($table, 'telnet_max_auth_allowed')),
        				OBJ_VAL3 => html('attempts') }] },
   			{ S1 =>	[{	CP2 => 1,
   					 	OBJ_VAL1 => html('Terminate Client Authentication connection after'),
        				OBJ_VAL2 => html(GetKey($table, 'client_max_auth_allowed')),
        				OBJ_VAL3 => html('attempts') },
        			 {	CP2 => 1,
					 	OBJ_VAL1 => html('Terminate Session Authentication connection after'),
        				OBJ_VAL2 => html(GetKey($table, 'session_max_auth_allowed')),
        				OBJ_VAL3 => html('attempts') }] }
        );

		push @props, (
			{ S0 => html('Authentication of users with certificates') },
        	{ S1 =>	[{	OBJ_VAL1 => (GetKey($table, 'enforce_suffix_for_internal_users') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Authenticate internal users with this suffix only:') }] },
        	{ S1 =>	[{	CP2 => 1,
					 	OBJ_VAL1 => $Spacer,
						OBJ_VAL2 => html(stripquotes(GetKey($table, 'allowed_suffix_for_internal_users'))) }] },
        	{ S1 =>	[{	CP2 => 1,
        				OBJ_VAL1 => html('Users certificates which were initiated but not pulled, will expire after'),
        				OBJ_VAL2 => html(GetKey($table, 'auth_validity_days')),
        				OBJ_VAL3 => html('days') }] }
        );

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: VPN - Advanced
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='VPN - Advanced';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='VPN-ADV';
	
	my @props;
		push @props, (
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'enable_mep_configuration') eq 'true')?$CheckTick:$Check,
			 			OBJ_VAL2 => html('Enable Backup Gateway') },
    				 {	OBJ_VAL1 => (GetKey($table, 'vpn_peer_ls_GW') eq 'true')?$CheckTick:$Check,
			    		OBJ_VAL2 => html('Enable load distribution for Multiple Entry Points configurations (Site to Site connections)') },
    				 {	OBJ_VAL1 => (GetKey($table, 'acceptdecrypt') eq 'true')?$CheckTick:$Check,
					 	OBJ_VAL2 => html('Enable decrypt on accept for gateway to gateway traffic') }] }
		); 

		push @props, (
			{ S0 => html('CRL Grace Period') },
			{ S1 => [{	CP2 => 1,
        				OBJ_VAL1 => html('Grace period before the CRL is valid:'),
        				OBJ_VAL2 => html(GetKey($table, 'crl_start_grace')),
        				OBJ_VAL3 => html('seconds') },
        			 { 	CP2 => 1,
        			  	OBJ_VAL1 => html('Grace period after the CRL is no longer valid:'),
        				OBJ_VAL2 => html(GetKey($table, 'crl_end_grace')),
        				OBJ_VAL3 => html('seconds') },
        			 { 	CP2 => 1,
					 	OBJ_VAL1 => html('Grace period extension for SecuRemote/SecureClient:'),
        				OBJ_VAL2 => html(GetKey($table, 'sr_grace_period')),
        				OBJ_VAL3 => html('seconds') }] }
        );

		push @props, (
			{ S0 => html('Resolving mechanism') },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'resolve_interface_ranges_GW') eq 'true')?$MarkDot:$Mark,
    					OBJ_VAL2 => html("Enable VPN-1 gateways to calculate statically peer gateway's best interface") },
        			 { 	OBJ_VAL1 => (GetKey($table, 'resolve_interface_ranges_GW') eq 'true')?$Mark:$MarkDot,
    					OBJ_VAL2 => html("Enable dynamic interface resolving per gateway") }] }
    	);

		push @props, (
			{ S0 => html('IKE Denial of Service protection') },
			{ S1 => [{	CP2 => 1,
					 	OBJ_VAL1 => html('Support IKE DoS protection from identified source'),
        				OBJ_VAL2 => html($IKEDoS{GetKey($table, 'ike_dos_protection_identified_initiator')}) },
        			 { 	CP2 => 1,
					 	OBJ_VAL1 => html('Support IKE DoS protection from unidentified source'),
        				OBJ_VAL2 => html($IKEDoS{GetKey($table, 'ike_dos_protection_unidentified_initiator')}) }] }
        );

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: VPN-1 Net
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='VPN-1 Net';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='VPN-1-Net';
	
	my @props;
		push @props, { TXT => [{	CLASS => 'note_xspc',
									TEXT => html('VPN-1 Net Gateways Configuration') }] };

		push @props, (
			{ S0 => html('Security Policy') },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'vpnddcate_policy') eq 'allow_all')?$MarkDot:$Mark,
    					OBJ_VAL2 => html('Accept all connections') },
        			 { 	OBJ_VAL1 => (GetKey($table, 'vpnddcate_policy') eq 'outbound_and_encrypted')?$MarkDot:$Mark,
					 	OBJ_VAL2 => html('Accept all encrypted and all outbound connections') },
        			 { 	OBJ_VAL1 => (GetKey($table, 'vpnddcate_policy') eq 'encrypted_only')?$MarkDot:$Mark,
						OBJ_VAL2 => html('Accept encrypted connections') }, 
        			 { 	OBJ_VAL1 => (GetKey($table, 'vpnddcate_policy') eq 'block_all')?$MarkDot:$Mark,
        			 	OBJ_VAL2 => html('Block all connections') }] }
       	); 

		push @props, (
			{ S0 => html('NAT') },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'vpnddcate_nat') eq 'none')?$MarkDot:$Mark,
						OBJ_VAL2 => html('Do not hide any connections (no NAT)') }, 
        			 { 	OBJ_VAL1 => (GetKey($table, 'vpnddcate_nat') eq 'hide_clear')?$MarkDot:$Mark,
					 	OBJ_VAL2 => html('Hide only Non-Encrypted connections') }, 
        			 { 	OBJ_VAL1 => (GetKey($table, 'vpnddcate_nat') eq 'hide_all')?$MarkDot:$Mark,
					 	OBJ_VAL2 => html('Hide all connections') }] }
		); 

		push @props, (
			{ S0 => html('Tracking') },
			{ S1 => [{	CP2 => 1,
					 	OBJ_VAL1 => html('Accepted Traffic'),
        				OBJ_VAL2 => html($Track{GetKey($table, 'vpnddcate_track_accept')}) },
        			 { 	CP2 => 1,
					 	OBJ_VAL1 => html('Dropped Traffic'),
        				OBJ_VAL2 => html($Track{GetKey($table, 'vpnddcate_track_drop')}) }] }
        );

		push @props, (
			{ S0 => html('Remote Device Configuration') },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'enable_ssl_conn_to_vpn1_net') eq 'true')?$CheckTick:$Check,
    					OBJ_VAL2 => html('Enable HTTPS connections to VPN-1 Net Modules') }, 
    		 		 { 	OBJ_VAL1 => (GetKey($table, 'enable_ssh_conn_to_vpn1_net') eq 'true')?$CheckTick:$Check,
    				 	OBJ_VAL2 => html('Enable SSH connections to VPN-1 Net Modules') }] }
    	); 

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
	
	# ----------------------------------
	# CLASS: Remote Access
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access - VPN-1 SecuRemote/SecureClient';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='RA-SR';
	
	my @props;

		if (GetKey($table, 'desktop_update_frequency') ne '0' ) {
			push @props, (
				{ S0 => html('Topology Update') },
				{ S1 => [{	CP3 => 1,
							OBJ_VAL1 => $CheckTick,
        					OBJ_VAL2 => html('Update topology every'),
        					OBJ_VAL3 => html(int(GetKey($table, 'desktop_update_frequency')/3600)),
        					OBJ_VAL4 => html('Hours') }] },
				{ S1 => [{ 	OBJ_VAL1 => $Spacer,
							OBJ_VAL2 => (GetKey($table, 'desktop_silent_topo_update') eq 'true')?$MarkDot:$Mark,
							OBJ_VAL3 => html('Automatic update') },
						 {	OBJ_VAL1 => $Spacer,
							OBJ_VAL2 => (GetKey($table, 'desktop_silent_topo_update') eq 'true')?$Mark:$MarkDot,
							OBJ_VAL3 => html('Upon VPN-1 SecuRemote/SecureClient start up') }] }
			);
		} else {
			push @props, (
				{ S0 => html('Topology Update') },
				{ S1 => [{	CP3 => 1,
							OBJ_VAL1 => $Check,
        					OBJ_VAL2 => html('Update topology') }] }
        	);
        }
			

		push @props, (
			{ S0 => html('Authentication Timeout') },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_to_expire_passwords') eq 'false')?$MarkDot:$Mark,
    					OBJ_VAL2 => html('Use default value') },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_to_expire_passwords') eq 'true')?$MarkDot:$Mark,
						OBJ_VAL2 => html('Validation timeout every'),
						OBJ_VAL3 => html(GetKey($table, 'desktop_password_expiry')),
						OBJ_VAL4 => html('Minutes') }] },
			{ S1 => [{	CP3 => 1,
					 	OBJ_VAL1 => (GetKey($table, 'desktop_cache_fwz_passwords') eq 'true')?$CheckTick:$Check, 
    					OBJ_VAL2 => html('Allow caching of static passwords on client') }] }
    	);

		push @props, (
			{ S0 => html('Additional Properties') },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_keep_alive') eq 'true')?$CheckTick:$Check, 
   						OBJ_VAL2 => html('Enable Back Connections (from gateway to client)') }] },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => $Spacer,
        				OBJ_VAL2 => html('Send keep-alive packet to the Gateway every'),
        				OBJ_VAL3 => html(GetKey($table, 'desktop_keep_alive_interval')),
        				OBJ_VAL4 => html('Seconds') }] },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'encrypt_dns') eq 'true')?$CheckTick:$Check, 
    					OBJ_VAL2 => html('Encrypt DNS traffic') }] }
    	);

		push @props, (
			{ S0 => html('VPN-1 SecureClient - Logon High Availability') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_PS_HA') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Use backup Policy Servers on Logon failure (Transparent Mode only)') }] }
		);
		if (GetKey($table, 'desktop_PS_HA') eq 'true' ) {
			push @props, (
   				{ S1 => [{	OBJ_VAL1 => $Spacer,
							OBJ_VAL2 => (GetKey($table, 'desktop_PS_LB') eq 'false')?$MarkDot:$Mark,
							OBJ_VAL3 => html('Choose next Policy Server') }] },
	   			{ S1 => [{	OBJ_VAL1 => $Spacer,
							OBJ_VAL2 => (GetKey($table, 'desktop_PS_LB') eq 'true')?$MarkDot:$Mark,
							OBJ_VAL3 => html('Choose Policy Server randomly') }] }
			);
		}
 
		push @props, (
				{ S0 => html('VPN-1 SecureClient - Desktop Security Policy expiration timer') },
				{ S1 => [{	CP2 => 1, 	
								OBJ_VAL1 => html('Revert to default policy after'),
        						OBJ_VAL2 => html(GetKey($table, 'desktop_policy_expire')),
        						OBJ_VAL3 => html('Minutes') }] }
        			);

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;

	# ----------------------------------
	# CLASS: Remote Access - VPN Basic
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access';
	$class{GP_SUBCLASS}='VPN Basic';
	$class{GP_CLASS}='RA-VPNBasic';
	
	my @props;

		push @props, (
			{ S0 => html('Support authentication methods') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_ike_preshared_support') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Pre-Shared Secret (for SecuRemote/SecureClient users)') },
   					 {	CP3 => 1,
   					 	# Below setting is unknown yet..... appears always to be true !!!
						#OBJ_VAL1 => (GetKey($table, '') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL1 => $CheckTick,
						OBJ_VAL2 => html('Public Key Signatures') },
   					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_ike_hybrid_support') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Hybrid Mode (VPN-1 & Firewall-1 authentication)') }] }
		);
	
		push @props, (
			{ S0 => html('IKE over TCP') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_site_default_tcp_ike') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Gateway support IKE over TCP') }] } 
		);

		push @props, (
			{ S0 => html('IP compression') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_ike_ipcomp_support') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Enable IP compression for SecureClient') }] }
		);

		push @props, (
			{ S0 => html('Load distribution') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'vpn_peer_ls') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Enable load distribution for Multiple Entry Points configurations (Remote Access)') }] }
		);

		push @props, (
			{ S0 => html('Nokia Clients') },
   			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'ike_support_nokia_crack') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Support remote access VPN using Nokia clients') }] }
		);

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;

	# ----------------------------------
	# CLASS: Remote Access - VPN Advanced
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access';
	$class{GP_SUBCLASS}='VPN Advanced';
	$class{GP_CLASS}='RA-VPNAdv';
	
	my @props;

		push @props, (
			{ S0 => html('User Encryption Properties') },
			{ S1 =>	[{ 	CP2 => 1,
						OBJ_VAL1 => html('Encryption Algorithm:'),
        				OBJ_VAL2 => html(GetKey($table, 'isakmp.encryption')) },
					 { 	CP2 => 1,
					 	OBJ_VAL1 => html('Data Integrity:'),
        				OBJ_VAL2 => html(GetKey($table, 'isakmp.data.integrity')) }] },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'force_encryption_on_all_users') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html('Enforce Encryption Algorithm and Data Integrity on all users') },
					 {	CP3 => 1,
						OBJ_VAL1 => $Spacer, 
						OBJ_VAL2 => html('Note: This global enforcement applies to NG FP2 and higher Modules,'.';').
									html('and overrides user specific Encryption Algorithm and Data Integrity settings') }] }
		);

		push @props, (
			{ S0 => html('IKE Security assocation Properties') }
		);

		my @section;
		my @groups;
		foreach $var (GetMembers(GetKey($table, 'desktop_ike_p1_supported_dh_groups'))) {
			$val=GetKey(GetKey($var,'Table'), GetKey($var,'Name'));
			$groups[GetKey($val, 'DH_group_number')]=GetKey($var, 'Name');
		}
		push @props, (
			{ S1 =>	[{ 	OBJ_VAL1 => html('Support Diffie-Hellman groups:'),
						OBJ_VAL2 => ((defined($groups[1]))?$CheckTick:$Check).html(' Group 1 (768 bit)') },
					 { 	OBJ_VAL1 => html(''),
						OBJ_VAL2 => ((defined($groups[2]))?$CheckTick:$Check).html(' Group 2 (1024 bit)') },
					 { 	OBJ_VAL1 => html(''),
						OBJ_VAL2 => ((defined($groups[5]))?$CheckTick:$Check).html(' Group 5 (1536 bit)') },
					 { 	OBJ_VAL1 => '',
						OBJ_VAL2 => '' },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Use Diffie-Hellman group:'),
        				OBJ_VAL2 => html(stripquotes(GetKey($table, 'desktop_ike_phase1_use_DH_group:Name'))) }] }
        );
			
		push @props, (
			{ S0 => html('Resolving mechanism') },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'resolve_interface_ranges') eq 'true')?$MarkDot:$Mark, 
						OBJ_VAL2 => html('Enable SecuRemote/SecureClient to calculate statically peer;').
									html("gateway's best interface based on network topology") },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'resolve_interface_ranges') eq 'false')?$MarkDot:$Mark, 
						OBJ_VAL2 => html('Enable dynamic interface resolving by SecuRemote/SecureClient peers.;').
									html('must be defined per gateway)') }
					] }
		);

		push @props, (
			{ S0 => html('SecuRemote/SecureClient behaviour while disconnected') },
			{ TXT => [{	CLASS => 'note_xspc', 
						TEXT => html('When disconnected, traffic to the encryption domain, will be') }] },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'allow_clear_traffic_while_disconnected') eq 'false')?$MarkDot:$Mark, 
						OBJ_VAL2 => html('Dropped') },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'allow_clear_traffic_while_disconnected') eq 'true')?$MarkDot:$Mark, 
						OBJ_VAL2 => html('Sent in clear') }
					] }
		);


	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;

	# ----------------------------------
	# CLASS: Remote Access - Certificates
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access';
	$class{GP_SUBCLASS}='Certificates';
	$class{GP_CLASS}='RA-Cert';
	
	my @props;

		push @props, (
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'sr_dont_check_crl') eq 'false')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Client will verify gateway's certificate against revocation list") },
					 {	CP3 => 1,
						OBJ_VAL1 => $Spacer },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'renew_users_ica_cert') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Renew users internal CA certificates") }
					]
			}
		);

		if (GetKey($table, 'renew_users_ica_cert') eq 'true') {
			push @props, (
				{ S1 => [{	CP2 => 1, 	
							OBJ_VAL1 => html('Start renewal process'),
        					OBJ_VAL2 => html(GetKey($table, 'desktop_policy_expire')),
        					OBJ_VAL3 => html('days before expiration date') }
						]
				}
			)
		}

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;

	# ----------------------------------
	# CLASS: Remote Access - SCV
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access';
	$class{GP_SUBCLASS}='Secure Configuration Verification (SCV)';
	$class{GP_CLASS}='RA-SCV';
	
	my @props;

		push @props, (
			{ S0 => html('Gateway Secure Configuration Options') },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'scv_gw_verify_only_mode') eq 'false')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Apply Secure Configuration Verification on Simplified Mode Security Policies") }
					]
			}
		);

		push @props, (
			{ S0 => html('Upon verification failure:') },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'enforce_desktop_config') eq 'true')?$MarkDot:$Mark, 
						OBJ_VAL2 => html("Block client's connection") },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'enforce_desktop_config') eq 'false')?$MarkDot:$Mark, 
						OBJ_VAL2 => html("Accept and log client's connection") }
					]
			}
		);

		push @props, (
			{ S0 => html("Basic configuration verification on client's machine") },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_security_protect_all_ifc') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Policy is installed on all interfaces") },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_security_non_ip_protocols') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Only TCP/IP protocols are used") }
					]
			}
		);

		push @props, (
			{ S0 => html("Configuration Violation Notification on client's machine") },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_security_send_log') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Generate log") },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_security_send_warning') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Notify the user") }
					]
			}
		);


	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;

	# ----------------------------------
	# CLASS: Remote Access - SCV
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Remote Access';
	$class{GP_SUBCLASS}='Early Version Compatibility';
	$class{GP_CLASS}='RA-EVC';
	
	my @props;
		push @props, (
			{ TXT => [{	CLASS => 'note',
						TEXT =>	html(';The following settings apply only to VPN-1 SecureClient prior to NG.;').
								html('For later versions, these settings are configured in the Desktop Security policy.') }
					]}
		);
		push @props, (
			{ S1 => [{	CP2 => 1, 	
						OBJ_VAL1 => html('Required policy for all desktops:'),
       					OBJ_VAL2 => html($EarlySCV{GetKey($table, 'desktop_security_policy_code')}) }
       				]}
       	);
		push @props, (
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'desktop_security_policy_installed') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Client is enforcing required policy") }
					]}
		);

	$class{GP_PROPS}=\@props;
		
	push @classes, \%class;
				
	# ----------------------------------
	# CLASS: Smart Directory (LDAP)
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Smart Directory (LDAP)';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='LDAP';
	
	my @props;

		push @props, (
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'fwldap_UseLDAP') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Use SmartDirectory (LDAP)") }
					]}
		);
		
		if (GetKey($table, 'fwldap_UseLDAP') eq 'true') {
			push @props, (
				{ S0 => html("SmartDirectory (LDAP) Properties:") } );

			my @section;
			push @section, (
						{	CP2 => 1, 	
							OBJ_VAL1 => html('Timeout on cached users:'),
        					OBJ_VAL2 => html(GetKey($table, 'fwldap_CacheTimeout')),
        					OBJ_VAL3 => html('seconds') },
						{	CP2 => 1, 	
							OBJ_VAL1 => html('Cache size:'),
        					OBJ_VAL2 => html(GetKey($table, 'fwldap_CacheSize')),
        					OBJ_VAL3 => html('users') }
			);

			if (GetKey($table, 'fwldap_is_PasswordExpiration') eq 'true') {
				push @section,	{	CP2 => 1, 	
									OBJ_VAL1 => $CheckTick.html(' Password expires after:'),
        							OBJ_VAL2 => html(GetKey($table, 'fwldap_PasswordExpiration')),
        							OBJ_VAL3 => html('days') };
			} else {
				push @section,	{	CP2 => 1, 	
									OBJ_VAL1 => $Check.html(' Password expires after:'),
        							OBJ_VAL2 => $Spacer,
        							OBJ_VAL3 => html('days') };
			}
							
			push @props, { S1 => \@section };
			
			$var = stripquotes(GetKey($table, 'fwldap_DisplayDN'));
			push @props, (
				{ 	SECTION => html("Display User's DN at Login"),
					S1 => [{	OBJ_VAL1 => (($var eq 'no display')?$MarkDot:$Mark).html(" Don't Display").$Spacer,
								OBJ_VAL2 => (($var eq 'display')?$MarkDot:$Mark).html(" Display").$Spacer,
								OBJ_VAL3 => (($var eq 'display upon request')?$MarkDot:$Mark).html(" Display on Request") }] }
			);			

			push @props, (
				{ S0 => html("Password Strength:") },
				{ S1 => [{	CP2 => 1, 	
							OBJ_VAL1 => html('Minimal password length:'),
        					OBJ_VAL2 => html(GetKey($table, 'psswd_min_length')),
        					OBJ_VAL3 => html('characters') }] },
				{ S1 => [{	CP3 => 1,
							OBJ_VAL1 => (GetKey($table, 'psswd_min_num_of_lowercase') eq 'true')?$CheckTick:$Check, 
							OBJ_VAL2 => html("Password must include lowercase character") },
						 {	CP3 => 1,
							OBJ_VAL1 => (GetKey($table, 'psswd_min_num_of_uppercase') eq 'true')?$CheckTick:$Check, 
							OBJ_VAL2 => html("Password must include an uppercase character") },
						 {	CP3 => 1,
							OBJ_VAL1 => (GetKey($table, 'psswd_min_num_of_numbers') eq 'true')?$CheckTick:$Check, 
							OBJ_VAL2 => html("Password must include a digit") },
						 {	CP3 => 1,
							OBJ_VAL1 => (GetKey($table, 'psswd_min_num_of_symbols') eq 'true')?$CheckTick:$Check, 
							OBJ_VAL2 => html("Password must include a symbol") },
						 {	CP3 => 1,
							OBJ_VAL1 => $Spacer },
						 {	CP3 => 1,
							OBJ_VAL1 => (GetKey($table, 'fwldap_ApplyPwdRulesOnMgmt') eq 'true')?$CheckTick:$Check, 
							OBJ_VAL2 => html("Enforce rules for user management administrators") }] }
			);

		} # End IF for SmartDirectory (LDAP)

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Floodgate-1
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Floodgate-1';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='Floodgate';
	
	$obj = GetKey($table, 'floodgate_preferences');
	
	my @props;

		push @props, (
			{ S0 => html("Weight") },
			{ S1 => [{	CP2 => 1, 	
						OBJ_VAL1 => html('Maximum weight of rule:'),
        				OBJ_VAL2 => html(GetKey($obj, 'default_max_weight')) },
					 {	CP2 => 1, 	
						OBJ_VAL1 => html('Default weight of rule:'),
        				OBJ_VAL2 => html(GetKey($obj, 'default_weight')),
        				OBJ_VAL3 => html('Note: Value will be applied to;new rules only') }] },

			{ S0 => html("Rate") },
			{ S1 => [{	CP2 => 1, 	
						OBJ_VAL1 => html('Unit of measure:'),
        				OBJ_VAL2 => html($UnitsMeasure{GetKey($obj, 'rate_units')}) }] },

			{ S0 => html("Authentication timeout for QoS") },
			{ S1 => [{	CP2 => 1, 	
						OBJ_VAL1 => html('Authenticated IP expires after:'),
        				OBJ_VAL2 => html(GetKey($obj, 'auth_true_timeout')),
        				OBJ_VAL3 => html('minutes') },
					 {	CP2 => 1, 	
						OBJ_VAL1 => html('Non authenticated IP expires after:'),
        				OBJ_VAL2 => html(GetKey($obj, 'auth_false_timeout')),
        				OBJ_VAL3 => html('minutes') },
					 {	CP2 => 1, 	
						OBJ_VAL1 => html('Unanswered queried IP expires after:'),
        				OBJ_VAL2 => html(GetKey($obj, 'auth_query_timeout')),
        				OBJ_VAL3 => html('minutes') }] }
    	);
        				

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Firewall-1 GX
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='FireWall-1 GX';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='FW1-GX';
	
	my @props;

		push @props, (
			{ S0 => html("GTP Intra Tunnel Inpspection") },
			{ S1 => [{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'gtp_anti_spoofing') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Enforce GTP anti spoofing") },
					 {	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'block_gtp_in_gtp') eq 'true')?$CheckTick:$Check, 
						OBJ_VAL2 => html("Block GTP in GTP") }] }
		);

		push @props, (
			{ S0 => html("Track") },
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => ((GetKey($table, 'gtp_code_alert') eq 'true')?$CheckTick:$Check).
									html(' Produce extended logs on unmatched PDUs'),
						OBJ_VAL2 => (GetKey($table, 'gtp_code_alert') eq 'true')?
									$Position{stripquotes(GetKey($table, 'gtp_code_alert_p'))}:'' },
					 {	CP2 => 1,
						OBJ_VAL1 => html("Protocol violation track option"),
						OBJ_VAL2 => html($Track{GetKey($table, 'gtp_track')}) }] }
		);

		$var = GetKey($table, 'allow_PDU_sequence');
		$val = GetKey($table, 'PDU_sequence');

		push @props, (
			{ S0 => html("GTP PDU Integrity Tests") },
			{ TXT =>[{	CLASS => 'note_xspc',
						TEXT => html('To enhance performance, disable one or more of the extended integrity tests below') }] },
			{ S1 => [{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'check_flow_labels') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html("Verify Flow Labels") },
					 {	CP3 => 1,
						OBJ_VAL1 => ($var eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('G-PDU seq number check with a maximum deviation of '),
						OBJ_VAL3 => ($var eq 'true')?$val:'' }] }
		);

		push @props, (
			{ S0 => html("Other") },
			{ S1 => [{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'gtp_sam_close_upon_delete') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html("Send 'clean state' request on each GTP tunnel Deactivation") },
					 {	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'gtp_allow_multi_if_ggsn') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html("Allow GGSN replies from multiple interfaces") },
					 {	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'enable_reverse_connections') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html("Enable Reverse Connections") }] }
		);

		push @props, (
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => html("GTP Signaling rate limit sampling interval "),
						OBJ_VAL2 => GetKey($table, 'rate_limit_sampling_interval'),
						OBJ_VAL3 => html("seconds") },
					 {	CP2 => 1,
						OBJ_VAL1 => html("Allow one GTP Echo on each path every "),
						OBJ_VAL2 => GetKey($table, 'gtp_echo_frequency'),
						OBJ_VAL3 => html("seconds") }] }
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: UserAuthority
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='UserAuthority';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='UA';
	
	my @props;

		push @props, (
			{ S0 => html("VPN-1/FireWall-1 Trusted Domains") },
			{ TXT => [{	CLASS => 'note',
						TEXT => html(';When matching VPN-1/FireWall-1 usernames to Windows Domains usernames for;Single Sign On:') }] },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'netso_trust_win_domains') eq 'all')?$MarkDot:$Mark, 
						OBJ_VAL2 => html("Trust all Windows Domains") },
					 {	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'netso_trust_win_domains') eq 'selectively')?$MarkDot:$Mark, 
						OBJ_VAL2 => html("Trust only the following Windows Domains:") }
					]}
		);

		my @section;
		@var = GetMembers(GetKey($table, 'netso_selected_trusted_domains_list'));
		foreach $var (@var) {
			push @section, {	CP2 => 1,
								OBJ_VAL1 => $Spacer,
								OBJ_VAL2 => html($var) }
		}
		
		push @props, { S1 => \@section };
		

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Management High Availability
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Management High Availability';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='MgmtHA';
	
	my @props;

		push @props, ({ S0 => html("Management High Availability") } );
			
		push @props, (
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'auto_sync_on_install') eq 'true')?$MarkDot:$Mark,
						OBJ_VAL2 =>	html('Automatic synchronization when policy is installed') }] }
		);

		if (GetKey($table, 'auto_sync_on_install') eq 'true') {
			$var = GetKey($table, 'secondary_sync_schedule');
			%var = %{obj2html("Global", GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			$v1 = (GetKey($table, 'auto_sync_on_save') eq 'true');
			$v2 = (GetKey($table, 'auto_sync_scheduled') eq 'true');
			push @props, (
				{ S1 =>	[{	CP3 => 1,
						 	OBJ_VAL1 => $Spacer.($v1?$CheckTick:$Check),
						 	OBJ_VAL2 => html('Every time a policy is saved') },
						 {	CP3 => 1,
						 	OBJ_VAL1 => $Spacer.($v2?$CheckTick:$Check),
						 	OBJ_VAL2 => html('On scheduled event: '),
						 	OBJ_VAL3 => $v2?$var:'' }] }
			);

		}
		push @props, (
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'auto_sync_on_install') eq 'false')?$MarkDot:$Mark,
						OBJ_VAL2 =>	html('Manual synchronization only') }] },
			{ S1 =>	[{	CP2 => 1,
						OBJ_VAL1 => html('Type of notification for Management High Availability tracking '),
						OBJ_VAL2 =>	$Track{GetKey($table, 'mgmtha_alert_type')} }] }
		);
						 		
	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Connect Control
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='ConnectControl';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='CC';
	
	my @props;
		push @props, (
			{ S0 => html("Servers Availability") },
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => html("Servers availability check interval: "),
						OBJ_VAL2 => GetKey($table, 'physical_server_availability_check_interval'),
						OBJ_VAL3 => html("seconds") },
					 {	CP2 => 1,
						OBJ_VAL1 => html("Server check retries: "),
						OBJ_VAL2 => GetKey($table, 'physical_server_check_retries'),
						OBJ_VAL3 => html("times") }] }
		);
		push @props, (
			{ S0 => html("Server Persistency") },
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => html("Persistent Server timeout: "),
						OBJ_VAL2 => GetKey($table, 'logical_servers_persistent_timeout'),
						OBJ_VAL3 => html("seconds") }] }
		);
		push @props, (
			{ S0 => html("Servers Load Balancing") },
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => html("Load agents port: "),
						OBJ_VAL2 => GetKey($table, 'load_service_port') },
					 {	CP2 => 1,
						OBJ_VAL1 => html("Load measurement interval: "),
						OBJ_VAL2 => GetKey($table, 'lbalanced_load_period_wakeup_sec'),
						OBJ_VAL3 => html("seconds") }] }
		);
						 		
	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: OSE - Open Security Extension
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='OSE - Open Security Extension';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='OSE';
	
	my @props;
		$v1 = (GetKey($table, 'established_router') eq 'true');
		$v2 = (GetKey($table, 'rip_router') eq 'true');
		$v3 = (GetKey($table, 'domain_udp_router') eq 'true');
		$v4 = (GetKey($table, 'domain_tcp_router') eq 'true');
		$v5 = (GetKey($table, 'icmpenable_router') eq 'true');
		push @props, (
			{ S0 => html("OSE Access List") },
			{ S1 => [{	CP3 => 1,
						OBJ_VAL1 => $v1?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept established TCP connections:'),
						OBJ_VAL3 => $v1?$Position{stripquotes(GetKey($table, 'established_router_p'))}:'' },
					 {	CP3 => 1,
						OBJ_VAL1 => $v2?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept RIP:'),
						OBJ_VAL3 => $v2?$Position{stripquotes(GetKey($table, 'rip_router_p'))}:'' },
					 {	CP3 => 1,
						OBJ_VAL1 => $v3?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept Domain Name over UDP (Queries):'),
						OBJ_VAL3 => $v3?$Position{stripquotes(GetKey($table, 'domain_udp_router_p'))}:'' },
					 {	CP3 => 1,
						OBJ_VAL1 => $v4?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept Domain Name over TCP (Zone Transfer):'),
						OBJ_VAL3 => $v4?$Position{stripquotes(GetKey($table, 'domain_tcp_router_p'))}:'' },
					 {	CP3 => 1,
						OBJ_VAL1 => $v5?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept ICMP requests:'),
						OBJ_VAL3 => $v5?$Position{stripquotes(GetKey($table, 'icmpenable_router_p'))}:'' }] }
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: OSE - Stateful Inspection
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Stateful Inspection';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='StateInspect';
	
	my @props;
		push @props, (
			{ S0 => html("Default Session Timeouts") },
			{ S1 => [{	CP2 => 1,
						OBJ_VAL1 => html('TCP start timeout:'),
						OBJ_VAL2 => GetKey($table, 'tcpstarttimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('TCP session timeout:'),
						OBJ_VAL2 => GetKey($table, 'tcptimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('TCP end timeout:'),
						OBJ_VAL2 => GetKey($table, 'tcpendtimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('UDP virtual session timeout:'),
						OBJ_VAL2 => GetKey($table, 'udptimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('ICMP virtual session timeout:'),
						OBJ_VAL2 => GetKey($table, 'icmptimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Other IP protocols virtual session timeout:'),
						OBJ_VAL2 => GetKey($table, 'othertimeout'),
						OBJ_VAL3 => html('seconds') }] }
		);

		push @props, (
			{ S0 => html("Stateful UDP") },
			{ S1 => [{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'udpreply') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept stateful UDP replies for unknown services') }] }
		);
		if (GetKey($table, 'udpreply') eq 'true') {
			push @props, (
				{ S1 => [{	CP1 => 1,
							OBJ_VAL1 => (GetKey($table, 'udpreply_from_any_port') eq 'true')?$CheckTick:$Check,
							OBJ_VAL2 => html('Accept stateful UDP replies from any port for unknown services') }] }
			);
		}
		push @props, (
			{ S0 => html("Accept Stateful ICMP") },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'icmpreply') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Replies'),
						OBJ_VAL3 => (GetKey($table, 'icmperrors') eq 'true')?$CheckTick:$Check,
						OBJ_VAL4 => html('Errors') }] }
		);
		push @props, (
			{ S0 => html("Stateful Other IP Protocols") },
			{ S1 => [{	OBJ_VAL1 => (GetKey($table, 'otherreply') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Accept stateful other IP protocols replies for unknown services') }] }
		);
		$v1 = (GetKey($table, 'fw_allow_out_of_state_tcp') eq '0');
		$v2 = (GetKey($table, 'fw_drop_out_of_state_udp') eq 'true');
		$v3 = (GetKey($table, 'fw_drop_out_of_state_icmp') eq 'true');
		push @props, (
			{ S0 => html("Out of state packets") },
			{ S1 => [{	OBJ_VAL1 => $v1?$CheckTick:$Check,
						OBJ_VAL2 => html('Drop out of state TCP packets'),
						OBJ_VAL3 => $v1?((GetKey($table, 'fw_log_out_of_state_tcp') eq '1')?$CheckTick:$Check):'',
						OBJ_VAL4 => $v1?html('Log on drop'):'' },
					 {	OBJ_VAL1 => $v2?$CheckTick:$Check,
						OBJ_VAL2 => html('Drop out of state UDP packets'),
						OBJ_VAL3 => $v2?((GetKey($table, 'fw_log_out_of_state_udp') eq '1')?$CheckTick:$Check):'',
						OBJ_VAL4 => $v2?html('Log on drop'):'' },
					 {	OBJ_VAL1 => $v3?$CheckTick:$Check,
						OBJ_VAL2 => html('Drop out of state ICMP packets'),
						OBJ_VAL3 => $v3?((GetKey($table, 'fw_log_out_of_state_icmp') eq '1')?$CheckTick:$Check):'',
						OBJ_VAL4 => $v3?html('Log on drop'):'' }] }
		);


	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Log and Alert
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Log and Alert';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='LogAlert';
	
	my @props;
		push @props, (
			{ S0 => html("Track Options") },
			{ S1 =>	[{	CP2 => 1,
						OBJ_VAL1 => html('VPN successful key exchange:'),
						OBJ_VAL2 => $Track{GetKey($table, 'vpn_success_key_exch')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('VPN packet handling errors:'),
						OBJ_VAL2 => $Track{GetKey($table, 'vpn_packet_handle_prob')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('VPN configuration & key exchange errors:'),
						OBJ_VAL2 => $Track{GetKey($table, 'vpn_conf_n_key_exch_prob')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('IP options drop:'),
						OBJ_VAL2 => $Track{GetKey($table, 'ipoptslog')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Administrative notifications:'),
						OBJ_VAL2 => $Track{GetKey($table, 'maintenance_notification')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('SLA violation:'),
						OBJ_VAL2 => $Track{GetKey($table, 'e2e_sla_alert')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Connection matched by SAM:'),
						OBJ_VAL2 => $Track{GetKey($table, 'sam_track')} },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Dynamic object resolution failure:'),
						OBJ_VAL2 => $Track{GetKey($table, 'dynamic_objects_track')} } ]}
		);

		push @props, (
			{ S0 => html("Logging Modifiers") },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'http_log_every_connection') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Log every authenticated HTTP connection')	}] },
						
			{ S0 => html("Time Settings") },
			{ S1 =>	[{	CP2 => 1,
						OBJ_VAL1 => html('Excessive log grace period:'),
						OBJ_VAL2 => GetKey($table, 'loggrace'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('SmartView Tracker resolving timeout:'),
						OBJ_VAL2 => GetKey($table, 'pagetimeout'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Virtual Links statistics logging interval:'),
						OBJ_VAL2 => GetKey($table, 'e2e_statistics_interval'),
						OBJ_VAL3 => html('seconds') },
					 {	CP2 => 1,
						OBJ_VAL1 => html('Status fetching interval:'),
						OBJ_VAL2 => GetKey($table, 'status_mgr_interval_seconds'),
						OBJ_VAL3 => html('seconds') }] },

			{ S0 => html("Community Default Rule") },
			{ S1 =>	[{	CP2 => 1,
						OBJ_VAL1 => html('Log Traffic:'),
						OBJ_VAL2 => $Track{GetKey($table, 'log_default_rule')} }] }
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Log and Alert - Alert Commands
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Log and Alert';
	$class{GP_SUBCLASS}='Alert Commands';
	$class{GP_CLASS}='LogAlertCmds';
	
	my @props;
		push @props, (
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'alertcmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send popup alert to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'alertcmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run popup alert script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'alertcmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'mailcmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send mail alert to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'mailcmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run mail alert script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'mailcmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'snmptrapcmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send SNMP trap alert to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'snmptrapcmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run SNMP trap alert script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'snmptrapcmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralertcmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send user defined alert no.1 to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralertcmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run UserDefined script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'useralertcmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralert2cmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send user defined alert no.2 to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralert2cmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run UserDefined script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'useralert2cmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralert3cmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send user defined alert no.3 to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'useralert3cmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run UserDefined script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'useralert3cmd'))) }] },
			{ TXT =>[{	CLASS => 'note',
						TEXT => html(';') }] },
		);

		push @props, (
			{ S0 => html('Early Versions Compatibility') },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'olderalertcmd_send_to_system_status') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Send 4.x alert to SmartView Status') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => (GetKey($table, 'olderalertcmd') ne '')?$CheckTick:$Check,
						OBJ_VAL2 => html('Run 4.x alert script:'),
						OBJ_VAL3 => html(stripquotes(GetKey($table, 'olderalertcmd'))) }] }
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: OPSEC
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='OPSEC';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='OPSEC';
	
	my @props;
		push @props, (
			{ S0 => html("OPSEC Product Installation") },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'allow_remote_ra') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Allow remote registration of OPSEC products') }] },
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: SmartCenter Access
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='SmartCenter Access';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='SmartAccess';
	
	my @props;
		$var = (GetKey($table, 'admin_lock_after_bad_attempts_enable') eq 'true');
		push @props, (
			{ S0 => html("Administrators login restrictions") },
			{ TXT =>[{	CLASS => 'note_xspc',
						TEXT => html('The following settings apply only to Administrators defined through SmartDashboard.') }] },
			{ S1 =>	[{	CP3 => 1,
						OBJ_VAL1 => $var?$CheckTick:$Check,
						OBJ_VAL2 => html("Lockout Administrator's account after"),
						OBJ_VAL3 => GetKey($table, 'admin_lock_after_bad_attempts'),
						OBJ_VAL4 => html('login failures') },
					 {	CP3 => 1,
						OBJ_VAL1 => $var?$Spacer:"",
						OBJ_VAL2 => $var?(((GetKey($table, 'admin_lock_auto_release_timeout_enable') eq 'true')?$CheckTick:$Check).
									html(" Unlock Administrator's account after")):"",
						OBJ_VAL3 => $var?GetKey($table, 'admin_lock_auto_release_timeout'):"",
						OBJ_VAL4 => $var?html('minutes'):"" }] },
			{ S1 =>	[{	CP1 => 1,
						OBJ_VAL1 => (GetKey($table, 'admin_lock_send_friendly_error_msg') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html("Display a detailed message to a locked out Administrator upon denying access") }] }
		);

	$class{GP_PROPS}=\@props;

	push @classes, \%class;

	# ----------------------------------
	# CLASS: Non Unique IP Address Ranges
	# ----------------------------------
	my %class;
	$class{GP_MAINCLASS}='Non Unique IP Address Ranges';
	$class{GP_SUBCLASS}='';
	$class{GP_CLASS}='NonUniqueIP';
	
	my @props;
		push @props, (
			{ TXT =>[{	CLASS => 'note_xspc',
						TEXT => html('All addresses that are not in the list will be considered as unique addresses.') }] }
		);
		
		my @section;
		push @section, {	HEAD => 1,
			  				OBJ_VAL1 => html('First IP'),
		  					OBJ_VAL2 => html('Last IP') };
		  					
		@var = GetMembers(GetKey($table, 'private_ip_ranges'));
		foreach $var (@var) {
			push @section, {	OBJ_VAL1 => html(GetKey($var, 'ipaddr_first')),
								OBJ_VAL2 => html(GetKey($var, 'ipaddr_last')) };
		}

		push @props, { T1 => \@section, C2 => 1 };

	$class{GP_PROPS}=\@props;

	push @classes, \%class;


	# End of Global Properties. Send to templating ...	
	$template->param(GLOBALPROPERTIES => \@classes );

	return $template;
}

1;
