package object2html;

use strict;
use CheckPoint;
use Constants;
use Options;
use Strings;

require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(HTMLMemberList HTMLObjectList HTMLCompoundObject txt2html obj2html);


sub HTMLMemberList {
	# Returns a list of members of the referenced object!

	# Two forms of obj referencing. One by hash ref and one by
	# table/name combination pointing to objects_5_0.C:
	# 1. $Policy, $obj_reference, $flags
	# 2. $Policy, $table, $name, $flags
	my ($Policy, $obj);
	my ($table, $name, $class);
	my (@list, @members, $member, $flags);
	my ($key, %text);

	$Policy = shift;
	$obj    = shift;
	if (ref($obj) eq '') {
		# Object reference by table/name combination
		$table = $obj;
		$name  = shift;
		# Check for legal input; i.e. the object does exist (and is not empty) ...
		if ($name eq '' || $table eq '') {
			return \@list;
		} else {
			$obj = GetKey($table.':'.$name);
		}
	}
	$flags   = shift;
	
	if ($obj) {
		# Determine if cell should be negated ...
		if (GetKey($obj, 'op') eq '"not in"') {
			$flags = $flags | $_NEG;
		}
		
		@members = GetMembers($obj);
		# if the memberlist only holds a reference to itself, the object
		# is assumed to be the only member of the list. So it is no list
		# at all, actually!
		if ($members[0] eq $obj) {
			if (!GetKey($obj, 'compound')) {        
				# Special case: inline objects (ACTION column only?)
				foreach $key (keys %{$obj}) {
					if (ref(GetKey($obj, $key)) ne '') {
						$table=GetKey($obj, $key, 'AdminInfo:table');
						push @list, obj2html($Policy, $table, $key)
					} else {
						print "Warning: object found can not be processed!\n";
					}
				}
			}
		} else {
			# 'Normal' objectlist ...
			foreach $member (@members) {
				$table=GetKey($member, 'Table');
				$name =GetKey($member, 'Name');
				push @list, obj2html($Policy, $table, $name, $flags);
				&ObjectUsed(GetKey($table, $name), $Policy);
			}
		} 
		
		if (GetKey($obj, 'compound')) {
			foreach $key (keys %{GetKey($obj, 'compound')}) {
				push @list, HTMLCompoundObject($Policy, GetKey($obj, 'compound:'.$key), $key, $flags)
			}
		}
	}
	
	# In case there are no members, return an empty element with a space
	# this will look much better in the actual layout...
	if (@list==0) { 
		$text{OBJ_TEXT} = '&nbsp;';
		push @list, \%text;
	}
	
	return \@list;
}

sub HTMLObjectList {
	# Returns a list of html code of the object referenced
	
	# Two forms of obj referencing. One by hash ref and one by
	# table/name combination pointing to objects_5_0.C:
	# 1. $Policy, $obj_reference, $name, $flags
	# 2. $Policy, $table, $name, $flags
	my (@list, %text);
	my ($Policy, $obj);
	my ($table, $name, $flags);
	
	$Policy = shift;
	$obj     = shift;
	if (ref($obj) eq '') {
		# Object reference by table/name combination
		$table = $obj;
		$name  = shift;
		$flags = shift;
		# Check for legal input ...
		if ($name eq '' || $table eq '') {
			return \@list;
		} else {
			$obj = GetKey($table, $name);
		}
	} else {
		$table = GetKey($obj, 'AdminInfo:table');
		$name  = shift;
		$flags = shift;
	}
	
	if ($table) {
		push @list, obj2html($Policy, $table, $name, $flags);
		&ObjectUsed(GetKey($table, $name), $Policy);
	} else {
		# object does not have a table ...
		push @list, txt2html($name, GetKey($obj, 'AdminInfo:ClassName'));
	}
	
	# In case there are no members, return an empty element with a space
	# this will look much better in the actual layout...
	if (@list==0) { 
		$text{OBJ_TEXT} = '&nbsp;';
		push @list, \%text;
	}

	return \@list;
}

sub HTMLCompoundObject {
	# HTML definition for compound objects 
	# form: $Policy, $obj, $name, $flags
	use strict;
	my ($Policy, $obj, $name, $flags);
	my %html_obj;
	my ($type, $color, $table, $href);
	my ($icon);
	$Policy = shift;
	$obj     = shift;
	$name    = shift;
	$flags   = shift;
	
	if (GetKey($obj, 'AdminInfo:ClassName') eq 'rule_user_group') {
		# UserGroup@network_object
		
		# Creat usergroup part ...
		$name  = stripquotes($name);
		$name  =~ s/\@.*$//; # remove ...@network_object
		$table = 'users';
		$color = GetKey($obj, 'color') || 'black';
		%html_obj = %{obj2html($Policy, $table, $name, $flags, $color)};
		&ObjectUsed(GetKey($table, $name), $Policy);

		# Create 'at' part ...
		$name  = GetKey($obj, 'at:Name');
		$table = GetKey($obj, 'at:Table');
		$html_obj{OBJ_TEXT} .= '@'.obj2html($Policy, $table, $name, $flags)->{OBJ_TEXT};
		&ObjectUsed(GetKey($table, $name), $Policy);
		
	} elsif (GetKey($obj, 'AdminInfo:ClassName') eq 'rule_services_compound_element') {
		# Service->Resource

		# Create service part ...
		$name  = GetKey($obj, 'service:Name');
		$table = GetKey($obj, 'service:Table');
		$html_obj{OBJ_TEXT} = obj2html($Policy, $table, $name, $flags)->{OBJ_TEXT};
		&ObjectUsed(GetKey($table, $name), $Policy);

		# Create resource part ...
		$name  = GetKey($obj, 'resource:Name');
		$table = GetKey($obj, 'resource:Table');
		$html_obj{OBJ_TEXT} .= '&#8209&GT'.obj2html($Policy, $table, $name, $flags)->{OBJ_TEXT};
		$html_obj{OBJ_ICON} = obj2html($Policy, $table, $name, $flags)->{OBJ_ICON};
		&ObjectUsed(GetKey($table, $name), $Policy);

	} else {
		# Unknown compound object ...
		$table = 'unknown';
		%html_obj = %{obj2html($Policy, $table, $name, $flags)};
		&ObjectUsed(GetKey($table, $name), $Policy);
	}

	return \%html_obj;
}

#*************************************************************************
#***                         Write HTML code                           ***
#*************************************************************************

#*************************************************************************
# Routine: txt2html
#
# Description:
# Outputs a single (user-defined) object(text) in a html list
# difference with obj2html is that the object is not linked
# to any table...
#
# Parameters:
# $text		Text to be used as object
# $class	Class of the object, used for icon ref
# $color	Color of the text
# $flags	Special circumstances are combined in one binary value
#
# Returns:
# $icon		Name of icon to use

sub txt2html {
	# Outputs a single (user-defined) object(text) in a html list
	# difference with obj2html is that the object is not linked
	# to any table...

	# Form: $text, $class, $color, $flags
	use strict;
	my $text   = shift;
	my $class  = shift;
	my $color  = shift;
	my $flags  = shift;
	my ($icon, %text);

	# Determine color and type of the object ...
	$color = $CPcolors{$color || 'navy blue'};
	$icon  = stripquotes(lc(($class || 'none'))).'.png';

	if (($flags & $_NEG) != $_NEG) {
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.Option('IconPath').'/'.$icon.'>';
		$text{OBJ_TEXT} = '<A CLASS=icon-text STYLE=color:'.$color.'>'.html($text).'</A>';
	} else {
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.Option('IconPath').'/'.$icon.'><IMG CLASS=not-icon SRC='.Option('IconPath').'/'.'_negate.png>';
		$text{OBJ_TEXT} = '<A CLASS=not-text STYLE=color:'.$color.'>'.html($text).'</A>';
	}
	
	return \%text;
}

#*************************************************************************
# Routine: obj2html
#
# Description:
# Routine translates an object from the object tables
# to HTML code
#
# Parameters:
# Two forms of obj referencing. One by hash ref and one by
# table/name combination pointing to database
# 1. $Policy, $obj_reference, $flags, $color
# 2. $Policy, $table, $name, $flags, $color
#
# where:
# $Policy	Name of the rulebase
# $obj_reference	Link to the object in the database
# $table	Table of the object
# $name 	Name of the object
#
# Optional parameters:
# $flags	Special circumstances are combined in one binary value
# $color	Color to be used for the text
#
# Returns:
# \%text	Reference to the hash holding both the icon and the object
#			html code

sub obj2html {
	use strict;

	my ($Policy, $obj, $table, $name);
	my ($page, $flags, $color);
	my ($href, $class, %text, $icon, $text);

	$Policy = shift;
	$obj    = shift;
	if (ref($obj) eq '') {
		$table = $obj;
		$name  = shift;
		$obj   = GetKey($table, $name);
	} else {
		$table = GetKey($obj, 'AdminInfo:table');
		$name  = GetKey($obj, 'AdminInfo:name');
	}
	$flags = shift;
	$color = shift || 'navy blue';

	# Determine the page the object is on ...
	$page = lc(Object2Page($Policy, GetKey($table, $name, 'AdminInfo:ClassName')));

	# Are we allowed to,
	#  and can we reference to the object within html 
	#  (ergo does the $page exist)?
	$href = ( (($flags & $_HREF) == $_HREF) && ($page) );

	# Prepare text for HTML code ...
	$text  = html(stripquotes($name));

	# Determine color of object ...
	$color = (GetKey($obj, 'color') || $color);
	$color = $CPcolors{stripquotes(lc($color))};

	# Determine icon(name)s ...
	if (!$obj) {
		$icon = icon2html($table, lc($name), $flags);
	} else {
		$icon = icon2html($obj, $flags);
	}
	
	#-------------------------------------
	# Make HTML code ...
	#-------------------------------------
	# Manual NAT rules ...
	if (($name eq 'Any') && ($flags & ($_NAT_STATIC+$_NAT_HIDE))) {
		# Any object found, in this case to be display as 'Original' ...
		%text = %{txt2html('Original', 'Original', 'black')}
	} elsif ($flags & $_NAT_HIDE) {
		# Hide NAT object ...
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'><IMG CLASS=nat-icon SRC='.Option('IconPath').'/nat-hide.png>';
		$text{OBJ_TEXT} = '<A CLASS=nat-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.$text.'</A>';
	} elsif ($flags & $_NAT_STATIC) {
		# Static NAT object ...
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'><IMG CLASS=nat-icon SRC='.Option('IconPath').'/nat-stat.png>';
		$text{OBJ_TEXT} = '<A CLASS=nat-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.$text.'</A>';

	# Automatic NAT rules ...		
	} elsif ($flags & $_NAT) {
		if (GetKey($table.':'.$name.':netobj_adtr_method') eq 'adtr_static') {
			# Static NAT object ...
			$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'><IMG CLASS=nat-icon SRC='.Option('IconPath').'/nat-stat.png>';
			$text{OBJ_TEXT} = '<A CLASS=nat-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.html($text.' (Valid Address)').'</A>';
		} elsif (GetKey($table.':'.$name.':netobj_adtr_method') eq 'adtr_hide') {
			# Hide NAT object ...
			$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'><IMG CLASS=nat-icon SRC='.Option('IconPath').'/nat-hide.png>';
			$text{OBJ_TEXT} = '<A CLASS=nat-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.html($text.' (Hiding Address)').'</A>';
		}

	# NOT object (negated) ...
	} elsif ($flags & $_NEG) {
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'><IMG CLASS=not-icon SRC='.Option('IconPath').'/_negate.png>';
		$text{OBJ_TEXT} = '<A CLASS=not-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.$text.'</A>';

	# Normal object ...
	} else {
		$text{OBJ_ICON} = '<IMG CLASS=icon SRC='.$icon.'>';
		$text{OBJ_TEXT} = '<A CLASS=icon-text'.($href?' HREF='.$page.'.html#'.$name:'').' STYLE=color:'.$color.'>'.$text.'</A>';
	}

	return \%text;
}

#*************************************************************************
# Routine: icon2html
#
# Description:
# Routine to determine which icon to use for the object
#
# Parameters:
# Two forms of obj referencing. One by hash ref and one by
# table/name combination pointing to objects_5_0.C:
# 1. $Policy, $obj_reference, $name, $flags
# 2. $Policy, $table, $name, $flags
# $Policy	Name of the rulebase
# $obj_reference	Link to the object in the database
# $table	Table of the object
# $name 	Name of the object
# $flags	Special circumstances are combined in one binary value
#
# Returns:
# $icon		Name of icon to use

sub icon2html {
	my ($obj, $class, $flags);
	my ($table, $name, $icon);
	$obj   = shift;
	if (ref($obj) eq '') {
		$table = $obj;
		$name  = shift;
		$obj   = GetKey($table, $name);
	}
	$flags = shift;

	$class = GetKey($obj, 'AdminInfo:ClassName');
	$icon  = $class;

	# CheckPoint gateways/hosts/clusters ...
	if (($class eq 'gateway_ckp') ||
		($class eq 'host_ckp')    ||
		($class eq 'gateway_cluster')) {
		if (GetKey($obj, 'firewall') eq 'installed') {
			$icon .= '.fw';
		}
		if (GetKey($obj, 'VPN')) {
			$icon .= '.vpn';
		}
		if (GetKey($obj, 'management') eq 'true') {
			$icon .= '.mgt';
		}
		if (GetKey($obj, 'location') eq 'external') {
			$icon .= '.ext';
		}

	# SofaWare gateways ...
	} elsif ($class eq 'sofaware_gateway') {
		if (GetKey($obj, 'sofaware_DAG') eq 'true') {
			$icon .= '.dag';
		}
		if (GetKey($obj, 'VPN')) {
			$icon .= '.vpn';
		}
		if (GetKey($obj, 'external') eq 'true') {
			$icon .= '.ext';
		}

	# SofaWare profiles ...
	} elsif ($class eq 'sofaware_gateway_profile') {
		if (GetKey($obj, 'AdminInfo:name') eq 'Hi-Med-Low_Profile') {
			$icon .= '.hi-med-low';
		}
	} elsif ($class eq 'sofaware_profiles_security_level') {
		$icon .= '.'.GetKey($obj, 'AdminInfo:name');

	# Intranet communities (meshed or star) ...
	} elsif ($class eq 'intranet_community') {
		$icon .= '.'.GetKey($obj, 'topology');
		
	# Install On ...
	} elsif ($class eq 'install_no_gateways') {
		$icon .= '.'.GetKey($obj, 'type');
	
	# Alert column ...
	} elsif ($class eq 'alert') {
		$icon .= '.'.GetKey($obj, 'alert');
	
	# Action column
	} elsif (($table eq 'setup') && ($name eq 'block')) {
		$icon = 'drop_action';

	# VPN column in rulebase ...
	} elsif (GetKey($obj, 'AdminInfo:name') eq 'All_GwToGw') {
		$icon = 'all_gwtogw';
		
	# User Group    
	#} elsif ($class eq 'rule_user_group') {
	} elsif ($table eq 'users') {
		$icon = 'user_group';

	} 
	
	# Add IconPath and .png extension ...
	$icon = Option('IconPath').'/'.lc($icon).'.png';
	
	return $icon;
}
