package checkpoint2html;

use strict;
use CheckPoint;
use Constants;
use Options;
use Sections2html;
use object2html;
use Strings;

require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(checkpoint2html);

# Initialize special marker signs ...
my $Check      = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/check.png>';
my $CheckTick  = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/checktick.png>';
my $Mark       = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/mark.png>';
my $MarkDot    = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/markdot.png>';
my $Spacer     = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/spacer.png>';
my $None       = '<I>(none)</I>';

##########################################################################
# Routine: checkpoint2html
#
# Description:
# Routine to create the CHECKPOINT table
##########################################################################
sub checkpoint2html {
	use strict;
	my ($obj, $name, $type, $current_type);
	my ($section, %layout, $layout);
	my ($cnt, @tobj, @tint, $intcnt, $iobj);
	my ($var, %var, @var, $val, $AItype);
	
	my $Policy   = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};

	# Determine layout 
	%layout = (
		'Check Point Products'		=> 'PRODUCTS',
		'Cluster Members'			=> 'PRODUCTS',
		'Sofaware Settings'			=> 'PRODUCTS',
		'Topology'					=> 'PROP1',
		'NAT'						=> 'PROP1',
		'VPN'						=> 'PROP2',
		'Remote Access'				=> 'PROP1',
		'Authentication'			=> 'PROP1',
		'SmartDirectory'			=> 'PRODUCTS',
		'SmartView Monitor'			=> 'PRODUCTS',
		'UserAuthority Server'		=> 'PRODUCTS',
		'UserAuthority WebAccess'	=> 'PRODUCTS',
		'FireWall-1 GX'				=> 'PRODUCTS',
		'Logs and Masters'			=> 'PROP2',
		'Capacity Optimization'		=> 'PROP2',
		'Advanced'					=> 'PROP2'
	);

	#----------------------------------------------------------------
	#---                  CHECKPOINT to HTML                      ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		# Determine if type has changed ...
		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}

		# Determine if 'normal'firewall software is installed ...
		my $FW_Installed = ((GetKey($obj, 'firewall') eq 'installed') && 
							($type !~ /Embedded/) && (GetKey($obj, 'location') ne 'external'));
		# Determine if this is a 'normal' gateway with VPN installed and is locally managed ...
		my $VPN_Installed = (((GetKey($obj, 'VPN_1')    eq 'true') || (GetKey($obj, 'vpnddcate') eq 'true')) &&
							($type !~ /Embedded/) && (GetKey($obj, 'location') ne 'external'));
				
		$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
		$tobj[$cnt]{OBJ_TYPE} = $type;
		$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($Policy, $obj, $name);
		
		#------------------------------------------------
		# IP address of Check Point device
		#------------------------------------------------
		if ($type !~ /Profile/ ) {
			if (GetKey($obj, 'ipaddr') =~ /0.0.0/) {
				$tobj[$cnt]{OBJ_IPADDRESS} = 'Dynamic Address'
			} else {
				$tobj[$cnt]{OBJ_IPADDRESS} = GetKey($obj, 'ipaddr');
			}
		} else {
			# Empty field
			$tobj[$cnt]{OBJ_IPADDRESS} = '&nbsp;'
		}
		
		#------------------------------------------------
		# Properties of Checkpoint devices
		#------------------------------------------------
		# Checkpoint version and option level ...
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Check Point Products');
		$layout  = 'OBJ_'.$layout{'Check Point Products'};
		if ($section) {
			if ($type eq 'Embedded Device') {
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => 	[{	SECTION  => html('Check Point Products'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) 
									}] },
					{	S1	=>	[{	CP2 => 1,
									OBJ_VAL1 => html('Type:'),
									OBJ_VAL2 => html('Nokia IP5x')
								}] },
					{	S1	=>	[{	CP4 => 1,	
									OBJ_VAL1 => $CheckTick,
									OBJ_VAL2 => html('VPN-1 & FireWall-1 installed'),
									OBJ_VAL3 => html('Version:'), 
									OBJ_VAL4 => number(GetKey($obj, 'fwver'))
								}] },
					{	S1	=>	[{	CP2 => 1,
									OBJ_VAL1 => html('License type:'),
									OBJ_VAL2 => number(GetKey($obj, 'embedded_lictype'))
								}] },
					{	SF_END => 1 }
				);

			} elsif ($type =~ /Profile/ ) {
				# No Check Point Products
			
			} elsif (GetKey($obj, 'cp_products_installed') eq 'true') {
				my @section;
				$var = '';
				$AItype = 0;
				if (GetKey($obj, 'cpver') eq '5.0') {
					$var .= 'NG';
					$val = GetKey($obj, 'option_pack');
					if ($val ge '4') { 
						$var .= ' with Application Intelligence';
						$AItype = 1;
					} elsif ($val ne '0') {
						$var .= ' Feature Pack '.$val
					}
				} else {
					$var .= GetKey($obj, 'fwver');
				}
				push @section, { CP2 => 1,	OBJ_VAL1 => html('Version:'), 
											OBJ_VAL2 => html($var) };
				
				if ($AItype) {
					$var = (GetKey($obj, 'cp_suite_type') eq 'pro')?'Check Point Enterprise/Pro':'Check Point Express';
					push @section, { CP2 => 1,	OBJ_VAL1 => html('Type:'), 
												OBJ_VAL2 => html($var) };
				}
	
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => [{	SECTION  => html('Check Point Products'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }],
						S1	=>	\@section,
					}
				);
	
				# Check Point products installed
				my @section;
				@var = ();
				if (GetKey($obj, 'firewall') eq 'installed') 		{ push @var, 'FireWall-1' }
				if (GetKey($obj, 'VPN_1')    eq 'true')				{ push @var, 'VPN-1 Pro' }
				if (GetKey($obj, 'vpnddcate') eq 'true')			{ push @var, 'VPN-1 Net' }
				if (GetKey($obj, 'floodgate') eq 'installed')		{ push @var, 'Floodgate-1' }
				if (GetKey($obj, 'policy_server') eq 'installed')	{ push @var, 'SecureClient Policy Server' }
				if (GetKey($obj, 'SDS') eq 'installed')				{ push @var, 'SecureClient Software Distribution Server' }
				if (GetKey($obj, 'management') eq 'true') {
					if (GetKey($obj, 'primary_management') eq 'true')	{ 
						push @var, 'Primary Management Station' 
					} else { 
						push @var, 'Secondary Management Station'
					}
				}
				
				if (GetKey($obj, 'cpver') gt '4.1')					{ push @var, 'SVN Foundation' }
				if (GetKey($obj, 'log_server') eq 'true')			{ push @var, 'Log Server' }
				if (GetKey($obj, 'reporting_server') eq 'true')		{ push @var, 'SmartView Reporter' }
				if (GetKey($obj, 'real_time_monitor') eq 'true')	{ push @var, 'SmartView Monitor' }
				if (GetKey($obj, 'UA_gateway') eq 'true')			{ push @var, 'UserAuthority Server' }
				if (GetKey($obj, 'UA_WebAccess') eq 'true')			{ push @var, 'UserAuthority WebAccess' }
				if (GetKey($obj, 'Everest') eq 'true')				{ push @var, 'Firewall-1 GX' }
				if (@var) {
					foreach $var (@var) {
						push @section, { OBJ_VAL1 => $CheckTick, OBJ_VAL2 => $var };
					}
				} else {
					push @section, { OBJ_VAL1 => $None };
				}
	
				push @{$tobj[$cnt]{$layout}}, { 
					SUBSECT_BEG => 'Installed Products', 
					S1 => \@section,
					SUBSECT_END => 1,
					SF_END => 1
				};
			}
				
		}
		# Cluster members ...
		if ($type eq 'Gateway Cluster') {
			$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Cluster Members');
			$layout  = 'OBJ_'.$layout{'Cluster Members'};
			if ($section) {
				my @section;
				# Find Members ...
				@var = GetMembers(GetKey($obj, 'cluster_members'));
				if (@var) {
					foreach $var (@var) {
						%var = %{obj2html($Policy, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
						push @section, { CP1 => 1, OBJ_VAL1 => $var };
					}
				} else {
					push @section, { OBJ_VAL1 => $None };
				}
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => [{	SECTION  => html('Cluster Members'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }],
						S1	=>	\@section,
						SF_END => 1
					}
				);
			}

		}
	
		# Sofaware Settings ...
		if (GetKey($obj, 'type') eq 'sofaware_gateway') {
			$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Sofaware Settings');
			$layout  = 'OBJ_'.$layout{'Sofaware Settings'};
			if ($section) {
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => [{	SECTION  => html('Sofaware Settings'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }]
					}
				);
	
				my @section;
				push @section, { 	OBJ_VAL1 => (GetKey($obj, 'sofaware_vpn') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('VPN Enabled') };
				if (GetKey($obj, 'sofaware_vpn') eq 'true') {
					$var = (GetKey($obj, 'sofaware_mode') eq 'SiteToSite')?'Site To Site':'Remote Access';
					push @section, 	{	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => ((GetKey($obj, 'sofaware_mode') eq 'SiteToSite')?$MarkDot:$Mark).
													html(' Site to Site') },
									{	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => ((GetKey($obj, 'sofaware_mode') eq 'SR')?$MarkDot:$Mark).
													html(' Remote Access') };
				}
	
				push @{$tobj[$cnt]{$layout}}, { S1 => \@section };
	
				my @section;			
				%var = %{obj2html($Policy, GetKey($obj, 'profile:Table'), GetKey($obj, 'profile:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, {	CP2 => 1, 
									OBJ_VAL1 => html('Type:'), 
									OBJ_VAL2 => html(GetKey($obj, 'product_type:Name')) };
				if (GetKey($obj, 'external') ne 'true') {
					push @section, (
							 {	CP2 => 1, 
								OBJ_VAL1 => html('VPN-1 Edge Profile:'), 
								OBJ_VAL2 => $var },
							 {	CP2 => 1, 
								OBJ_VAL1 => html('Product Key:'), 
								OBJ_VAL2 => html(GetKey($obj, 'product_key')) },
							 {	CP2 => 1, 
								OBJ_VAL1 => html('MAC Address:'), 
								OBJ_VAL2 => html(GetKey($obj, 'MAC_address')) }
					);
				}
				
				
				push @{$tobj[$cnt]{$layout}}, ({ S1 => \@section });
	
				push @{$tobj[$cnt]{$layout}}, (
					{ S1 => [{
						OBJ_VAL1 => (GetKey($obj, 'external') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Externally Managed Gateway') }] }
				);
				
				push @{$tobj[$cnt]{$layout}}, ( { SF_END => 1 } );
			}

		}

		#------------------------------------------------
		# TOPOLOGY tab/column of Checkpoint devices
		#------------------------------------------------
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Topology');
		$layout  = 'OBJ_'.$layout{'Topology'};
		if ($section) {
			push @{$tobj[$cnt]{$layout}}, Topology($Policy, $obj, $name, $type, $section);
		}
		
		#------------------------------------------------
		# NAT tab of CheckPoint device
		#------------------------------------------------
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'NAT');
		$layout  = 'OBJ_'.$layout{'NAT'};
		if ($section) {
			push @{$tobj[$cnt]{$layout}}, NAT($Policy, $obj, $name, $type, $section);
		}

		#------------------------------------------------
		# VPN tab/column of Checkpoint devices
		#------------------------------------------------
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'VPN');
		$layout  = 'OBJ_'.$layout{'VPN'};
		if ($section) {
			push @{$tobj[$cnt]{$layout}}, VPN($Policy, $obj, $name, $type, $section);
		}
		
		#------------------------------------------------
		# Remote Access
		#------------------------------------------------
		if ($VPN_Installed && (Version($obj) ge '5.01')) {
			# CheckPoint VPN is installed and version > NG (no FP) ...
			$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Remote Access');
			$layout  = 'OBJ_'.$layout{'Remote Access'};
			if ($section) {
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => 	[{	SECTION  => html('Remote Access'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) 
									}] }
				);
	
				if (Version($obj) ge '5.03') {
    				# L2TP Support
    				push @{$tobj[$cnt]{$layout}}, (
    					{ 	SUBSECT_BEG => 	html('L2TP Support') },
    					{	S1	=>	[{	C2 => 2,
    									OBJ_VAL1 => (GetKey($obj, 'support_l2tp') eq 'true')?$CheckTick:$Check,
    									OBJ_VAL2 => html('Support L2TP (relevant only when Office Mode is active)') },
    								 {	INACTIVE => (GetKey($obj, 'support_l2tp') eq 'false'),
    									CP3 => 1,
    									OBJ_VAL1 => $Spacer,
    									OBJ_VAL2 => html('Authentication Method:'),
    									OBJ_VAL3 => (GetKey($obj, 'l2tp_auth_method') eq 'MD5')?html('MD5-Challenge'):html('Smart Card or other Certificate') },
    								 {	INACTIVE => (GetKey($obj, 'support_l2tp') eq 'false'),
    									CP3 => 1,
    									OBJ_VAL1 => $Spacer,
    									OBJ_VAL2 => html('Use this certificate:'),
    									OBJ_VAL3 => html(GetKey($obj, 'use_cert_for_l2tp')) }] },
    					{	SUBSECT_END => 1 }
    				);
				}
				
				if (Version($obj) ge '5.04') {
					# Hub Mode configuration ...
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SUBSECT_BEG => 	html('Hub Mode Configuration') },
						{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'allow_VPN_routing_from_SR') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Allow SecureClient to route traffic through this gateway') }] },
						{	SUBSECT_END => 1 }
					);
				}
	
				# NAT traversal ...
				%var = %{obj2html($Policy, GetKey($obj, 'isakmp.udpencapsulation:resource:Table'), GetKey($obj, 'isakmp.udpencapsulation:resource:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SUBSECT_BEG => 	html('NAT traversal') },
					{	S1	=>	[{	C2 => 2,
									OBJ_VAL1 => (GetKey($obj, 'isakmp.udpencapsulation:active') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Support NAT traversal mechanism (UDP encapsulation)') },
								 {	INACTIVE => (GetKey($obj, 'isakmp.udpencapsulation:active') eq 'false'),
									CP3 => 1,
									OBJ_VAL1 => $Spacer,
									OBJ_VAL2 => html('Allocated port:'),
									OBJ_VAL3 => $var }] },
					{	SUBSECT_END => 1 }
				);
	
				if (Version($obj) ge '5.04') {
					# Visitor mode configuration ...
					%var = %{obj2html($Policy, GetKey($obj, 'tcpt:resource:Table'), GetKey($obj, 'tcpt:resource:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SUBSECT_BEG => 	html('Visitor Mode configuration') },
						{	S1	=>	[{	C2 => 2,
										OBJ_VAL1 => (GetKey($obj, 'tcpt:active') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Support Visitor Mode') },
									 {	INACTIVE => (GetKey($obj, 'tcpt:active') eq 'false'),
										CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Allocated port:'),
										OBJ_VAL3 => $var },
									 {	INACTIVE => (GetKey($obj, 'tcpt:active') eq 'false'),
										CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Allocated IP Address:'),
										OBJ_VAL3 => html(stripquotes(GetKey($obj, 'tcpt:interface'))) }] },
						{	SUBSECT_END => 1 }
					);
				}
			}

			#------------------------------------------------
			# Office mode
			#------------------------------------------------
			push @{$tobj[$cnt]{$layout}}, (
				{ 	SF_BEG => 	[{	SECTION  => html('Office Mode'), 
									SUB      => 1,
									OBJ_NAME => $name, 
									STATE    => 0 
								}] }
			);
			
   			%var = %{obj2html($Policy, GetKey($obj, 'ip_assignment_group:Table'), GetKey($obj, 'ip_assignment_group:Name'), $_HREF)};
   			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @{$tobj[$cnt]{$layout}}, (
				{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'ip_assignment_offer') eq 'never')?$MarkDot:$Mark,
								OBJ_VAL2 => html('Do not offer Office Mode') },
							 {	INACTIVE => (GetKey($obj, 'ip_assignment_offer') eq 'per_group'),
							 	CP3 => 1,
							 	OBJ_VAL1 => (GetKey($obj, 'ip_assignment_offer') eq 'per_group')?$MarkDot:$Mark,
							 	OBJ_VAL2 => html('Offer Office Mode to group') },
							 {	INACTIVE => (GetKey($obj, 'ip_assignment_offer') ne 'per_group'),
							 	CP3 => 1,
							 	OBJ_VAL1 => (GetKey($obj, 'ip_assignment_offer') eq 'per_group')?$MarkDot:$Mark,
							 	OBJ_VAL2 => html('Offer Office Mode to group:'),
							 	OBJ_VAL3 => $var },
							 {	OBJ_VAL1 => (GetKey($obj, 'ip_assignment_offer') eq 'always')?$MarkDot:$Mark,
								OBJ_VAL2 => html('Allow Office Mode to all users') }] }
			);

			if (GetKey($obj, 'ip_assignment_offer') ne 'never') {
   				# Office mode Method ...
				if (GetKey($obj, 'om_method') eq 'ippool') {
					%var  = %{obj2html($Policy, GetKey($obj, 'om_ippool:Table'), GetKey($obj, 'om_ippool:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SUBSECT_BEG => 	html('Office Mode Method') },
						{	S1	=>	[{	OBJ_VAL1 => $MarkDot,
										OBJ_VAL2 => html('Manual (using IP pool)') },
									 {	C3 => 2, CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Allocate IP from network:'),
										OBJ_VAL3 => $var },
									 {	OBJ_VAL1 => $Mark,
										OBJ_VAL2 => html('Automatic (using DHCP)') }] },
						{	SUBSECT_END => 1 }
					);
				} else {
					%var  = %{obj2html($Policy, GetKey($obj, 'om_dhcp_server:Table'), GetKey($obj, 'om_dhcp_server:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SUBSECT_BEG => 	html('Office Mode Method') },
						{	S1	=>	[{	OBJ_VAL1 => $Mark,
										OBJ_VAL2 => html('Manual (using IP pool)') },
									 {	OBJ_VAL1 => $MarkDot,
										OBJ_VAL2 => html('Automatic (using DHCP)') },
									 {	CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('User specific DHCP server:'),
										OBJ_VAL3 => $var },
									 {	CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Virtual IP address for DHCP;server replies:'),
										OBJ_VAL3 => GetKey($obj, 'om_giaddr') },
									 {	CP3 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('MAC address for DHCP allocation:'),
										OBJ_VAL3 => $MacCalc{GetKey($obj, 'om_mac_calc')} }] },
						{	SUBSECT_END => 1 }
					);
				}
   			
				# IP Lease duration ...
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SUBSECT_BEG => 	html('IP Lease Duration') },
					{	S1	=>	[{	CP2 => 1,
									OBJ_VAL1 => html('IP lease duration:'), 
									OBJ_VAL2 => number(GetKey($obj, 'om_lease_time')),
									OBJ_VAL3 => html('minutes') }] },
					{	SUBSECT_END => 1 }
				);

				if (Version($obj) ge '5.04') {
					# Multiple Interfaces ...
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SUBSECT_BEG => 	html('Multiple Interfaces') },
						{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'om_enable_with_multiple_IF') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Support connectivity enhancement for gateways with;multiple external interfaces') }] },
						{	SUBSECT_END => 1 }
					);
				}

				if (Version($obj) ge '5.04') {
    				# Anti spoofing ...
    				%var  = %{obj2html($Policy, GetKey($obj, 'om_additional_ip_for_anti_spoofing:Table'), GetKey($obj, 'om_additional_ip_for_anti_spoofing:Name'), $_HREF)};
    				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    				push @{$tobj[$cnt]{$layout}}, (
    					{ 	SUBSECT_BEG => 	html('Anti-Spoofing') },
    					{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'om_perform_antispoofing') eq 'true')?$CheckTick:$Check,
    									OBJ_VAL2 => html('Perform Anti-Spoofing on Office Mode addresses') }] },
    					{	S1	=>	[{	INACTIVE => (GetKey($obj, 'om_perform_antispoofing') eq 'false'),
    								 	OBJ_VAL1 => $Spacer,
    								 	OBJ_VAL2 => html('Addtional IP Addresses for Anti-Spoofing:'),
    								 	OBJ_VAL3 => $var }] },
    					{	SUBSECT_END => 1 }
    				);
    			}

			} # end of Office Mode ...

			# Close off Office Mode ...
			push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
  			
			if (Version($obj) ge '5.03') {
    			#------------------------------------------------
    			# Clientless VPN
    			#------------------------------------------------
    			push @{$tobj[$cnt]{$layout}}, (
    				{ 	SF_BEG => 	[{	SECTION  => html('Clientless VPN'), 
    									SUB      => 1,
    									OBJ_NAME => $name, 
    									STATE    => 0 
    								}] }
    			);
    
    			push @{$tobj[$cnt]{$layout}}, (
    				{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'use_clientless_vpn') eq 'true')?$CheckTick:$Check,
    								OBJ_VAL2 => html('Support Clientless VPN') }] }
    			);
    			
				if (Version($obj) ge '5.04') {
        			if (GetKey($obj, 'use_clientless_vpn') eq 'true') {
        				push @{$tobj[$cnt]{$layout}}, (
        					{ S1	=>	[{	CP3 => 1, C2 => 3, C3 => 2,
        									OBJ_VAL1 => $Spacer,
        									OBJ_VAL2 => html('Certificate for gateway authentication:'),
        									OBJ_VAL3 => html(stripquotes(GetKey($obj, 'use_cert'))) },
        								 {	CP3 => 1, C2 => 2, C3 => 3,
        								 	OBJ_VAL1 => $Spacer,
        									OBJ_VAL2 => html('Client authentication:'),
        									OBJ_VAL3 => $VPNAsk{GetKey($obj, 'clientless_VPN_ask_user_for_certificate')} },
        								 {	CP3 => 1, C2 => 3,
        								 	OBJ_VAL1 => $Spacer,
        									OBJ_VAL2 => html('Number of concurrent servers/processes:'),
        									OBJ_VAL3 => number(GetKey($obj, 'clientless_proc_num')) }] },
        					{ S1	=>	[{	OBJ_VAL1 => $Spacer,
        									OBJ_VAL2 => (GetKey($obj, 'accept_3des_for_clientless_vpn') eq 'true')?$CheckTick:$Check,
        									OBJ_VAL3 => html('Accept only 3DES for clientless VPN connections') }] }
        				);
        			}
        		}
    			# Close off Clientless VPN ...
    			push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
    		}

			# Close off Remote Access ...
			push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };

		}

		#------------------------------------------------
		# Authentication
		#------------------------------------------------
		if ($FW_Installed) {
			# Firewall installed on object ...
			$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Authentication');
			$layout  = 'OBJ_'.$layout{'Authentication'};
			if ($section) {
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => 	[{	SECTION  => html('Authentication'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }] }
				);
				
				# Authentication Schemes ...
				$val = GetKey($obj, 'host_schemes_val');
				push @{$tobj[$cnt]{$layout}}, (
					{	SUBSECT_BEG => html('Enabled Authentication Schemes:') },
					{	COL_BEG => 1,
						S1	=>	[{	OBJ_VAL1 => ($val & $AuthCode{'VPN-1'})?$CheckTick:$Check,
									OBJ_VAL2 => html('VPN-1 & Firewall-1 Password') },
								 {	OBJ_VAL1 => ($val & $AuthCode{'SecurID'})?$CheckTick:$Check,
									OBJ_VAL2 => html('SecurID') },
								 {	OBJ_VAL1 => ($val & $AuthCode{'OS'})?$CheckTick:$Check,
									OBJ_VAL2 => html('OS Password') }] },
					{	COL_NXT => 1,
						S1	=>	[{	OBJ_VAL1 => ($val & $AuthCode{'RADIUS'})?$CheckTick:$Check,
									OBJ_VAL2 => html('RADIUS') },
								 {	OBJ_VAL1 => ($val & $AuthCode{'TACACS'})?$CheckTick:$Check,
									OBJ_VAL2 => html('TACACS') }] },
					{ 	COL_END => 1 },
					{	SUBSECT_END => 1 }
				);
	
				if (Version($obj) ge '5.0') {
    				# Authentication Settings ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Authentication Settings:') },
    					{	S1	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('User Authentication session timeout:'),
    									OBJ_VAL2 => number(GetKey($obj, 'au_timeout')),
    									OBJ_VAL3 => html('Minutes') }] },
    					{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'ca_wait_mode') eq 'true')?$CheckTick:$Check,
    									OBJ_VAL2 => html('Enable wait mode for Client Authentication') }] },
    					{	S1	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('Authentication Failure Track:'),
    									OBJ_VAL2 => html($Track{GetKey($obj, 'default_track')}) }] },
    					{	SUBSECT_END => 1 }
    				);
    			}
	
				if (Version($obj) ge '5.0') {
    				# HTTP Security Server ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('HTTP Security Server:') },
    					{	S1	=>	[{	OBJ_VAL1 => ((GetKey($obj, 'http_next_proxy_defined') eq 'true')?$CheckTick:$Check),
    									OBJ_VAL2 =>	html('Use Next Proxy:') }] },
    					{	S1	=>	[{	CP3 => 1, CP5 => 1,
    									OBJ_VAL1 => $Spacer,
    									OBJ_VAL2 => html('Host:'),
    									OBJ_VAL3 => html(stripquotes(GetKey($obj, 'http_next_proxy_host'))),
    									OBJ_VAL4 => html('Port:'),
    									OBJ_VAL5 => number(GetKey($obj, 'http_next_proxy_port')) }] },
    					{	SUBSECT_END => 1 }
    				);
    			}
	
				# End of Authentication 
				push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
			}
		}
			
		#------------------------------------------------
		# SmartDirectory (LDAP)
		#------------------------------------------------
		if ($FW_Installed && (GetKey('properties:firewall_properties:fwldap_UseLDAP') eq 'true')) {
			# Firewall installed on object ...
    		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'SmartDirectory');
    		$layout  = 'OBJ_'.$layout{'SmartDirectory'};
    		if ($section) {
    			push @{$tobj[$cnt]{$layout}}, (
    				{ 	SF_BEG => 	[{	SECTION  => html('SmartDirectory (LDAP)'), 
    									OBJ_NAME => $name, 
    									STATE    => ($section eq $_OPEN) }] }
    			);
    			
    			# Authentication Schemes ...
    			push @{$tobj[$cnt]{$layout}}, (
    				{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'display_au_list') eq 'true')?$CheckTick:$Check,
    								OBJ_VAL2 => html('Display list of distinguished names (DNs);for matching UID on login') }] },
    				{	S1	=>	[{	CP2 => 1,
    								OBJ_VAL1 => html('Timeout on LDAP requests:'),
    								OBJ_VAL2 => number(GetKey($obj, 'fwldap_RequestTimeout')),
    								OBJ_VAL3 => html('seconds') }] }
    			);
    			
    			# Account Unit Query ...
    			push @{$tobj[$cnt]{$layout}}, (
    				{	SUBSECT_BEG => html('Account Unit Query') },
    				{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'use_custom_au_list') eq 'false')?$MarkDot:$Mark,
    								OBJ_VAL2 => html('All') },
    							 {	OBJ_VAL1 => (GetKey($obj, 'use_custom_au_list') eq 'true')?$MarkDot:$Mark,
    								OBJ_VAL2 => html('Selected Account Units list') }] }
    			);
    			
    			if (GetKey($obj, 'use_custom_au_list') eq 'true') {
    				my @section;
    				my ($ref, $refobj);
    				my ($priority, $priority_index, $default_priority);
    				my ($ldap, $ldapsrv);
    
    				push @section,	{	HEAD => 1, 
    									OBJ_VAL1 => html('Account Unit'),
    									OBJ_VAL2 => html('Host'),
    									OBJ_VAL3 => html('Priority')
    				};
    
    				foreach $ref (GetMembers(GetKey($obj, 'custom_multi_server_au_list'))) {
    					%var  = %{obj2html($Policy, GetKey($ref, 'account_unit:Table'), GetKey($ref, 'account_unit:Name'), $_HREF)};
    					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    
    					$default_priority = (GetKey($ref, 'default_priority') eq 'true');
    					$priority_index   = GetKey($ref, 'priority_index');
    					$refobj  = GetKey(GetKey($ref, 'account_unit:Table').':'.GetKey($ref, 'account_unit:Name'));
    
    					foreach $ldap (GetMembers(GetKey($refobj, 'ldap_servers'))) {
    						$ldapsrv = GetKey($ldap, 'server:Name');
    						if ($default_priority) {
    							$priority = GetKey($ldap, 'priority');
    						} else {
    							$val = GetKey($ldap, 'other_priorities:'.$priority_index);
    							$priority = GetKey($ldap, 'other_priorities:'.$priority_index);
    						}
    						push @section, {
    							OBJ_VAL1 => $var,
    							OBJ_VAL2 => $ldapsrv,
    							OBJ_VAL3 => number($priority)
    						};
    						$var = '';
    					}
    				}	
    				push @{$tobj[$cnt]{$layout}}, {	C3 => 1, T1	=>	\@section }
    			}
    			push @{$tobj[$cnt]{$layout}}, {	SUBSECT_END => 1 };
    			
    			# End of SmartDirectory (LDAP)
    			push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
    		}
    	}
	
		if (Version($obj) ge '5.03') {
			#------------------------------------------------
			# SmartView Monitor
			#------------------------------------------------
			if (GetKey($obj, 'real_time_monitor') eq 'true') {
				# SmartView monitor product installed ...
				$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'SmartView Monitor');
				$layout  = 'OBJ_'.$layout{'SmartView Monitor'};
				if ($section) {
					if (GetKey($obj, 'real_time_monitor') eq 'true') {
						push @{$tobj[$cnt]{$layout}}, (
							{ 	SF_BEG => 	[{	SECTION  => html('SmartView Monitor'), 
												OBJ_NAME => $name, 
												STATE    => ($section eq $_OPEN) }] }
						);
			
						push @{$tobj[$cnt]{$layout}}, (
							{	TXT	=>	[{	CLASS => 'note_xspc',
											TEXT  => html('Select data to be stored for SmartView Monitor history reports:') }] },
							{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'enable_rtm_counters_report') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Check Point system counters (e.g. CPU usage, accepted packets)') },
										 {	OBJ_VAL1 => (GetKey($obj, 'enable_rtm_traffic_report_per_connection') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Traffic Connections') },
										 {	OBJ_VAL1 => (GetKey($obj, 'enable_rtm_traffic_report') eq 'true')?$CheckTick:$Check,
											OBJ_VAL2 => html('Traffic Throughput (Bytes per second)') }] },
							{	TXT	=>	[{	CLASS => 'note',
											TEXT  => html(';').
													 html('The history database is located on the gateway. This is a cyclic;').
													 html('database that can consume up to 20 MB.') }] },
							{	TXT	=>	[{	CLASS => 'note',
											TEXT  => html(';').
													 html(";Note: The implementation of Traffic's history reports may have;").
													 html("performance implications.") }] }
						);
			
						# End of SmartView Monitor
						push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
					}
				}
			}
		}
	
		#------------------------------------------------
		# UserAuthority Server
		#------------------------------------------------
		if ((GetKey($obj, 'UA_gateway') eq 'true') &&
			($FW_Installed || $VPN_Installed)) {
    		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'UserAuthority Server');
    		$layout  = 'OBJ_'.$layout{'UserAuthority Server'};
    		if ($section) {
    			if (GetKey($obj, 'UA_gateway') eq 'true') {
    				push @{$tobj[$cnt]{$layout}}, (
    					{ 	SF_BEG => 	[{	SECTION  => html('UserAuthority Server'), 
    										OBJ_NAME => $name, 
    										STATE    => ($section eq $_OPEN) }] }
    				);
    	
    				# Configuration Method ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Configuration Method') },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'netso:netso_auto_config') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('Automatic Configuration') },
    								 {	OBJ_VAL1 => (GetKey($obj, 'netso:netso_auto_config') eq 'false')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('Manual Configuration') }] }
    				);
    				
    				if (GetKey($obj, 'netso:netso_auto_config') eq 'false') {
    					push @{$tobj[$cnt]{$layout}}, (
    						{	S1	=>	[{	OBJ_VAL1 => $Spacer,
    										OBJ_VAL2 => html('Share identity accross the UserAuthority Server installed on:') }] }
    					);
    	
    					my @section;
    					push @section, 
    						{	OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => (GetKey($obj, 'netso:netso_vpn_redirect') eq 'true')?$CheckTick:$Check,
    							OBJ_VAL3 => html('VPN Tunnel endpoints when VPN is established') 
    						};
    						
    					$var = '';
    					if (GetKey($obj, 'netso:netso_share_WA_list')) {
    						%var = %{obj2html($Policy, GetKey($obj, 'netso:netso_share_WA_list:Table'), GetKey($obj, 'netso:netso_share_WA_list:Name'), $_HREF)};
    						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    					}
    					push @section, 
    						{	CP4 => 1, 
    							OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => (GetKey($obj, 'netso:netso_share_WA_list'))?$CheckTick:$Check,
    							OBJ_VAL3 => html('Servers that share Webaccess Authentication'),
    							OBJ_VAL4 => $var 
    						};
    	
    					$var = '';
    					if (GetKey($obj, 'netso:netso_use_UAS') eq 'true') {
    						%var = %{obj2html($Policy, GetKey($obj, 'netso:netso_UAS:Table'), GetKey($obj, 'netso:netso_UAS:Name'), $_HREF)};
    						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    					}
    					push @section, 
    						{	CP4 => 1,
    							OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => (GetKey($obj, 'netso:netso_use_UAS') eq 'true')?$CheckTick:$Check,
    							OBJ_VAL3 => html('Windows Domain Controllers'),
    							OBJ_VAL4 => $var 
    						};
    	
    					$var = '';
    					if (GetKey($obj, 'netso:netso_on_citrix_list')) {
    						%var = %{obj2html($Policy, GetKey($obj, 'netso:netso_on_citrix_list:Table'), GetKey($obj, 'netso:netso_on_citrix_list:Name'), $_HREF)};
    						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    					}
    					push @section, 
    						{	CP4 => 1,
    							OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => GetKey($obj, 'netso:netso_on_citrix_list')?$CheckTick:$Check,
    							OBJ_VAL3 => html('Citrix/Microsoft Terminal Servers'),
    							OBJ_VAL4 => $var 
    						};
    	
    					$var = '';
    					if (GetKey($obj, 'netso:netso_other_gw')) {
    						%var = %{obj2html($Policy, GetKey($obj, 'netso:netso_other_gw:Table'), GetKey($obj, 'netso:netso_other_gw:Name'), $_HREF)};
    						$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    					}
    					push @section, 
    						{	CP4 => 1,
    							OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => GetKey($obj, 'netso:netso_other_gw')?$CheckTick:$Check,
    							OBJ_VAL3 => html('VPN-1/FireWall-1 Gateways'),
    							OBJ_VAL4 => $var
    						};
    	
    					push @{$tobj[$cnt]{$layout}}, {	S1	=>	\@section }
    	
    				}
    				push @{$tobj[$cnt]{$layout}}, {	SUBSECT_END => 1 };
    											
    				# Export Policy ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Export Policy') },
    					{	TXT	=>	[{	CLASS => 'note_xspc',
    									TEXT  => html('Export the following data to externally-managed UserAuthority Servers:') }] },
    					{	COL_BEG => 1 },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'netso:netso_export_user') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('User name') },
    								 {	OBJ_VAL1 => (GetKey($obj, 'netso:netso_export_dn') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('DN') }] },
    					{	COL_NXT => 1 },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'netso:netso_export_group') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('User groups') },
    								 {	OBJ_VAL1 => (GetKey($obj, 'netso:netso_export_clientip') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('Source IP') }] },
    					{	COL_NXT => 1 },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'netso:netso_export_scheme') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL2 => html('Authentication Scheme') }] },
    					{	COL_END => 1 },
    					{	SUBSECT_END => 1 }
    				);
    					
    				# Logging Level ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Logging Level') },
    					{	S1	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('UserAuthority Logging Level:'),
    									OBJ_VAL2 => html($LogLevel{GetKey($obj, 'netso:netso_log_level')}) }] },
    					{	SUBSECT_END => 1 }
    				);
    				
    				# Close UserAuthority Server
    				push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
    			}
    		} # End of UserAuthority Server		
		}
		
		if (Version($obj) ge '5.02') {
    		#------------------------------------------------
    		# UserAuthority WebAccess
    		#------------------------------------------------
   			if (GetKey($obj, 'UA_WebAccess') eq 'true') {
	       		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'UserAuthority WebAccess');
    			$layout  = 'OBJ_'.$layout{'UserAuthority WebAccess'};
    			if ($section) {
    				push @{$tobj[$cnt]{$layout}}, (
    					{ 	SF_BEG => 	[{	SECTION  => html('UserAuthority WebAccess'), 
    										OBJ_NAME => $name, 
    										STATE    => ($section eq $_OPEN) }] }
    				);
    				
    				%var = %{obj2html($Policy, GetKey($obj, 'web_auth:user_auth_server_ip:Table'), GetKey($obj, 'web_auth:user_auth_server_ip:Name'), $_HREF)};
    				my $server  = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    				%var = %{obj2html($Policy, GetKey($obj, 'web_auth:user_auth_server_port:Table'), GetKey($obj, 'web_auth:user_auth_server_port:Name'), $_HREF)};
    				my $service = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    				push @{$tobj[$cnt]{$layout}}, (
    					{	S1	=>	[{	OBJ_VAL1 => html('UserAuthority Server:'),
    									OBJ_VAL2 => (GetKey($obj, 'web_auth:uas_is_this_machine') eq 'false')?$MarkDot:$Mark,
    									OBJ_VAL3 => html('On'),
    									OBJ_VAL4 => $server },
    								 {	C3 => 2,
    								 	OBJ_VAL1 => $Spacer,
    								 	OBJ_VAL2 => (GetKey($obj, 'uas_is_this_machine') eq 'true')?$MarkDot:$Mark,
    									OBJ_VAL3 => html('On this machine') },
    								 {	C1 => 3,
    								 	OBJ_VAL1 => html('UserAuthority Server service:'),
    								 	OBJ_VAL2 => $service },
    								 {	C1 => 3, CP2 => 1,
    								 	OBJ_VAL1 => html('Action taken for URL outside the scope of the policy:'),
    								 	OBJ_VAL2 => $WebURLAction{GetKey($obj, 'web_auth:action_for_url_external_to_all_policies')} },
    								 {	CP2 => 1, C2 => 3,
    								 	OBJ_VAL1 => html('Redirect URL to:'),
    								 	OBJ_VAL2 => html(stripquotes(GetKey($obj, 'web_auth:redirect_external_url_to')||'-')) },
    								 {	CP2 => 1, C1 => 3,
    								 	OBJ_VAL1 => html('Track:'),
    								 	OBJ_VAL2 => $Track{(GetKey($obj, 'web_auth:track'))} },
    								] }
    				);
    
		    		#------------------------------------------------
		    		# UserAuthority WebAccess - Advanced
		    		#------------------------------------------------
    				push @{$tobj[$cnt]{$layout}}, (
    					{ 	SF_BEG => 	[{	SECTION  => html('Advanced WebAccess'), 
    										SUB2     => 1,
    										OBJ_NAME => $name, 
    										STATE    => 0 }] }
    				);
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Limits and Timeouts') },
    					{	S1 	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('Maximal client request buffer size:'),
    									OBJ_VAL2 => number(GetKey($obj, 'web_auth:max_client_request_buffer')),
    									OBJ_VAL3 => html('KBytes') },
    								{	CP2 => 1,
    									OBJ_VAL1 => html('Communication to UserAuthority Server timeout:'),
    									OBJ_VAL2 => number(GetKey($obj, 'web_auth:communication_to_uas_timeout')),
    									OBJ_VAL3 => html('milliseconds') },
    								{	CP2 => 1,
    									OBJ_VAL1 => html('HTTP session timeout:'),
    									OBJ_VAL2 => number(GetKey($obj, 'web_auth:http_session_timeout')),
    									OBJ_VAL3 => html('secons') }] },
    					{	SUBSECT_END => 1 }
    				);
    
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('SSL') },
    					{	S1 	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('SSL redirection methodology:'),
    									OBJ_VAL2 => $SSLredirect{GetKey($obj, 'web_auth:ssl_redirection_methodology')} },
    								{	CP2 => 1,
    									OBJ_VAL1 => html('Redirect to:'),
    									OBJ_VAL2 => (GetKey($obj, 'web_auth:ssl_redirection_methodology') eq 'redirect_to_page')?
    												html(stripquotes(GetKey($obj, 'web_auth:ssl_redirect_page'))):'-' }] },
    					{	SUBSECT_END => 1 }
    				);
    
    				%var = %{obj2html($Policy, GetKey($obj, 'web_auth:NT_domain_controller:Table'), GetKey($obj, 'web_auth:NT_domain_controller:Name'), $_HREF)};
    				my $server  = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Windows Groups:') },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'web_auth:get_NT_groups') eq 'true')?$CheckTick:$Check,,
    									OBJ_VAL2 => html('Get Windows group data for Windows users') }] },
    					{	S1	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('Windows domain controller:'),
    									OBJ_VAL2 => (GetKey($obj, 'web_auth:get_NT_groups') eq 'true')?$server:'-' }] },
    					{	SUBSECT_END => 1 }
    				);
    
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('Case Sensitivity:') },
    					{	S1 	=>	[{	OBJ_VAL1 => (GetKey($obj, 'web_auth:URL_case_sensitive') eq 'true')?$CheckTick:$Check,,
    									OBJ_VAL2 => html('Case sensitivity naming convention for URLs') }] },
    					{	SUBSECT_END => 1 }
    				);
    									
    				# Close UserAuthority WebAccess - Advanced
    				push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
    				
    				
    				# Close UserAuthority WebAccess
    				push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
    				
    			} 
    		} # End of UserAuthority WebAccess
		}
				
		#------------------------------------------------
		# FireWall-1 GX
		#------------------------------------------------
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'FireWall-1 GX');
		$layout  = 'OBJ_'.$layout{'FireWall-1 GX'};
		if ($section) {
			push @{$tobj[$cnt]{$layout}}, Firewall1_GX($Policy, $obj, $name, $type, $section);
		}
	
		if (Version($obj) ge '5.0') {
			#------------------------------------------------
			# Logs and Masters
			#------------------------------------------------
			if (GetKey($obj, 'use_loggers_and_masters') eq 'true') {
				$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Logs and Masters');
				$layout  = 'OBJ_'.$layout{'Logs and Masters'};
				if ($section) {
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SF_BEG => 	[{	SECTION  => html('Logs and Masters'), 
											OBJ_NAME => $name, 
											STATE    => ($section eq $_OPEN) }] }
					);
		
					# Local Log Files ...
					my $obj1 = GetKey($obj, 'log_policy');
					my $unit = $DiskSpace{GetKey($obj1, 'alert_free_disk_space_metrics')};
					
					%var = %{obj2html($Policy, GetKey($obj1, 'log_switch_schedule:Table'), GetKey($obj1, 'log_switch_schedule:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{$layout}}, (
						{	SUBSECT_BEG => html('Log switch') },
						{	S1	=>	[{	CP3 => 1,
										OBJ_VAL1 => (GetKey($obj1, 'log_switch_on_file_size') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Log switch when file size is:'),
										OBJ_VAL3 => (GetKey($obj1, 'log_switch_on_file_size') eq 'true')?number(GetKey($obj1, 'log_switch_size')):'-',
										OBJ_VAL4 => html('MBytes') },
									 {	C3 => 2, CP3 => 1,
										OBJ_VAL1 => (GetKey($obj1, 'scheduled_switch') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Schedule log switch to:'),
										OBJ_VAL3 => (GetKey($obj1, 'scheduled_switch') eq 'true')?$var:'-' }] },
						{	SUBSECT_END => 1 }
					);
					
					push @{$tobj[$cnt]{$layout}}, (
						{	SUBSECT_BEG => html('Disk space management') },
						{	S1	=>	[{	CP2 => 1,
										OBJ_VAL1 => html('Measure free disk space in:'),
										OBJ_VAL2 => $unit }] },
						{	S1	=>	[{	CP3 => 1, C2 => 2,
										OBJ_VAL1 => (GetKey($obj1, 'log_delete_on_below') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Required Free Disk Space:'),
										OBJ_VAL3 => (GetKey($obj1, 'log_delete_on_below') eq 'true')?number(GetKey($obj1, 'log_delete_below_value')):'-',
										OBJ_VAL4 => html($unit) },
									 {	C1 => 5,
										OBJ_VAL1 => html('Old log files will be deleted until the required free disk space is restored.') },
									 {	CP4 => 1,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj1, 'log_keep_on_days') eq 'true')?$CheckTick:$Check,
										OBJ_VAL3 => html('Do not delete log files from the last'),
										OBJ_VAL4 => (GetKey($obj1, 'log_keep_on_days') eq 'true')?number(GetKey($obj1, 'log_keep_days_value')):'-',
										OBJ_VAL5 => html('Days') },
									 {	CP3 => 1, C2 => 2,
										OBJ_VAL1 => (GetKey($obj1, 'alert_on_disk_space') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Alert when free disk space is below:'),
										OBJ_VAL3 => (GetKey($obj1, 'alert_on_disk_space') eq 'true')?number(GetKey($obj1, 'alert_free_disk_space')):'-',
										OBJ_VAL4 => html($unit) },
									 {	CP3 => 1, C2 => 2,
										OBJ_VAL1 => (GetKey($obj1, 'stop_logging_on_free_disk_space') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Stop logging when free disk space is below:'),
										OBJ_VAL3 => (GetKey($obj1, 'stop_logging_on_free_disk_space') eq 'true')?number(GetKey($obj1, 'min_free_disk_space')):'-',
										OBJ_VAL4 => html($unit) },
									 {	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj1, 'reject_connections') eq 'true')?$CheckTick:$Check,
										OBJ_VAL3 => html('Reject all connection when logs are not saved.') }] },
						{	SUBSECT_END => 1 }
					);
		
					# Additional Logging ...
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SF_BEG => 	[{	SECTION  => html('Additional Logging'),
											SUB      => 1,
											OBJ_NAME => $name, 
											STATE    => 0 }] }
					);
		
					%var = %{obj2html($Policy, GetKey($obj1, 'log_forward_target:Table'), GetKey($obj1, 'log_forward_target:Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					%var = %{obj2html($Policy, GetKey($obj1, 'log_forward_schedule:Table'), GetKey($obj1, 'log_forward_schedule:Name'), $_HREF)};
					$val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @{$tobj[$cnt]{$layout}}, (
						{	SUBSECT_BEG => html('Log forwarding settings') },
						{	S1	=>	[{	CP3 => 1,
										OBJ_VAL1 => (GetKey($obj1, 'forward_logs') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Forward log files to Management Server:'),
										OBJ_VAL3 => (GetKey($obj1, 'forward_logs') eq 'true')?$var:'-' },
									 {	C3 => 2,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Log forwarding schedule:'),
										OBJ_VAL3 => (GetKey($obj1, 'forward_logs') eq 'true')?$val:'-' }] },
						{	S1	=>	[{	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj1, 'log_switch_before_forwarding') eq 'true')?$CheckTick:$Check,
										OBJ_VAL3 => html('Perform Log switch before log forwarding') }] },
						{	SUBSECT_END => 1 }
					);
		
					push @{$tobj[$cnt]{$layout}}, (
						{	SUBSECT_BEG => html('Advanced settings') },
						{	S1	=>	[{	C1 => 2, CP2 => 1,
										OBJ_VAL1 => html('Update Account Log every:'),
										OBJ_VAL2 => number(GetKey($obj1, 'acct_update_interval')),
										OBJ_VAL3 => html('Seconds') },
									 {	OBJ_VAL1 => (GetKey($obj1, 'etm_logging') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Turn on QoS Logging') },
									 {	OBJ_VAL1 => (GetKey($obj1, 'Citrix_ICA_application_detection') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Detect new Citrix ICA application names') }] },
						{	SUBSECT_END => 1 }
					);
		
					# Close off Additional Logging ...
					push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
		
					# Masters ...
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SF_BEG => 	[{	SECTION  => html('Masters'),
											SUB      => 1,
											OBJ_NAME => $name, 
											STATE    => 0 }] }
					);
		
					push @{$tobj[$cnt]{$layout}}, (
						{	S1	=>	[{	OBJ_VAL1 => !GetKey($obj, 'masters')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Use local definitions for Masters') },
									 {	OBJ_VAL1 => GetKey($obj, 'masters')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Define Masters') }] }
					);
		
					if (GetKey($obj, 'masters')) {
						my @section;
						foreach $val (GetMembers(GetKey($obj, 'masters'))) {
							%var = %{obj2html($Policy, GetKey($val, 'Table'), GetKey($val, 'Name'), $_HREF)};
							$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
							push @section, { OBJ_VAL1 => $var };
						}
						push @{$tobj[$cnt]{$layout}}, (
							{ 	TXT	=>	[{	CLASS => 'note_xspc',
											TEXT  => html('Use the following Management Stations for fetch and install:') }] },
							{ 	SUBSECT_BEG => chr(0) },
							{	S1	=> 	\@section },
							{	SUBSECT_END => 1 }
						);
					}
					
					# Close off Masters ...
					push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
		
					# Log Servers ...
					push @{$tobj[$cnt]{$layout}}, (
						{ 	SF_BEG => 	[{	SECTION  => html('Log Servers'),
											SUB      => 1,
											OBJ_NAME => $name, 
											STATE    => 0 }] }
					);
		
					push @{$tobj[$cnt]{$layout}}, (
						{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'define_logging_servers') eq 'false')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Use local definitions for Log Servers') },
									 {	OBJ_VAL1 => (GetKey($obj, 'define_logging_servers') eq 'true')?$MarkDot:$Mark,
										OBJ_VAL2 => html('Define Log Servers') },
									 {	OBJ_VAL1 => (GetKey($obj, 'log_servers:self_log_server') eq 'true')?$CheckTick:$Check,
										OBJ_VAL2 => html('Send Logs to this module ('.$name.')') }] }
					);
		
					if (GetKey($obj, 'define_logging_servers') eq 'true') {
						my @section;
						my %logserver;
						my $logserver;
		
						# Find all logservers (log and/or alert) ...
						foreach $val (GetMembers(GetKey($obj, 'log_servers:send_logs_to'))) {
							$logserver{GetKey($val, 'Name')}[0] = $val;
						}
						foreach $val (GetMembers(GetKey($obj, 'log_servers:send_alerts_to'))) {
							$logserver{GetKey($val, 'Name')}[1] = $val;
						}
		
						# Set headers of table with logservers ...
						push @section, {	HEAD => 1,
											OBJ_VAL1 => html('Log Server'),
											OBJ_VAL2 => html('Logs'),
											OBJ_VAL3 => html('Alerts')
						};
						# List all logservers, but eliminate duplicates ...
						foreach $val (keys(%logserver)) {
							$logserver = $logserver{$val}[0] || $logserver{$val}[1];
							%var = %{obj2html($Policy, GetKey($logserver, 'Table'), GetKey($logserver, 'Name'), $_HREF)};
							$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
							push @section, {	OBJ_VAL1 => $var,
												OBJ_VAL2 => ($logserver{$val}[0])?$CheckTick:$Check,
												OBJ_VAL3 => ($logserver{$val}[1])?$CheckTick:$Check
							};
						}
						push @{$tobj[$cnt]{$layout}}, (
							{ 	TXT	=>	[{	CLASS => 'note_xspc',
											TEXT  => html('Always send logs to:') }] },
							{ 	SUBSECT_BEG => chr(0) },
							{	C3 => 1,	T1	=> 	\@section },
							{	SUBSECT_END => 1 }
						);
		
						my @section;
						if (GetKey($obj, 'log_servers:backup_log_servers')) {
							foreach $val (GetMembers(GetKey($obj, 'log_servers:backup_log_servers'))) {
								%var = %{obj2html($Policy, GetKey($val, 'Table'), GetKey($val, 'Name'), $_HREF)};
								$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
								push @section, { OBJ_VAL1 => $var };
							}
						} else {
							push @section, { OBJ_VAL1 => $None }
						}
						push @{$tobj[$cnt]{$layout}}, (
							{ 	TXT	=>	[{	CLASS => 'note_xspc',
											TEXT  => html('When a Log Server is unreachable, send logs to:') }] },
							{ 	SUBSECT_BEG => chr(0) },
							{	S1	=> 	\@section },
							{	SUBSECT_END => 1 }
						);
						undef %logserver;
						undef $logserver;
					}
					
					# Close off Log Servers ...
					push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
		
					# Clear local variables ...
					undef $obj1;
					undef $unit;
		
					# Close off Logs and Masters ...
					push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
					
				}	# End of Logs and Masters
			}
		}
		
		#------------------------------------------------
		# Capacity Optimization
		#------------------------------------------------
		if ($FW_Installed || $VPN_Installed) {
			# Firewall or VPN-1 is installed ...
			$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Capacity Optimization');
			$layout  = 'OBJ_'.$layout{'Capacity Optimization'};
			if ($section) {
				push @{$tobj[$cnt]{$layout}}, (
					{ 	SF_BEG => 	[{	SECTION  => html('Capacity Optimization'), 
										OBJ_NAME => $name, 
										STATE    => ($section eq $_OPEN) }] }
				);
	
				# Capacity Optimization ...
				push @{$tobj[$cnt]{$layout}}, (
					{	SUBSECT_BEG => html('Capacity Optimization') },
					{	S1	=>	[{	CP2 => 1,
									OBJ_VAL1 => html('Maximum concurrent connections:'),
									OBJ_VAL2 => number(GetKey($obj, 'connections_limit')) }] },
					{	S1	=>	[{	OBJ_VAL1 => html('Calculate connections hash table size and memory pool') }] },
					{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'calculation_type') eq 'partially_automatic')?$MarkDot:$Mark,
									OBJ_VAL2 => html('Automatically') },
								 {	OBJ_VAL1 => (GetKey($obj, 'calculation_type') eq 'manual')?$MarkDot:$Mark,
									OBJ_VAL2 => html('Manually') }] },
					{	S1	=>	[{	C3 => 2, CP3 => 1, INACTIVE => (GetKey($obj, 'calculation_type') ne 'manual'),
									OBJ_VAL1 => $Spacer,
									OBJ_VAL2 => html('Connections hash table size:'),
									OBJ_VAL3 => number(GetKey($obj, 'connections_hashsize')) },
								 {	CP3 => 1, INACTIVE => (GetKey($obj, 'calculation_type') ne 'manual'),
									OBJ_VAL1 => $Spacer,
									OBJ_VAL2 => html('Memory pool size:'),
									OBJ_VAL3 => number(GetKey($obj, 'fw_hmem_size')),
									OBJ_VAL4 => html('MByte') },
								 {	CP3 => 1, INACTIVE => (GetKey($obj, 'calculation_type') ne 'manual'),
									OBJ_VAL1 => $Spacer,
									OBJ_VAL2 => html('Maximum memory pool size:'),
									OBJ_VAL3 => number(GetKey($obj, 'fw_hmem_maxsize')),
									OBJ_VAL4 => html('MByte') }] },
					{	SUBSECT_END => 1 }
				);
	
				if (Version($obj) ge '5.03') {
    				# VPN Capacity Optimization ...
    				push @{$tobj[$cnt]{$layout}}, (
    					{	SUBSECT_BEG => html('VPN Capacity Optimization') },
    					{	S1	=>	[{	CP2 => 1,
    									OBJ_VAL1 => html('Maximum concurrent IKE negotiations:'),
    									OBJ_VAL2 => number(GetKey($obj, 'max_num_negs')) },
    								 {	CP2 => 1,
    									OBJ_VAL1 => html('Maximum concurrent tunnels:'),
    									OBJ_VAL2 => number(GetKey($obj, 'max_concurrent_vpn_tunnels')) }] },
    					{	SUBSECT_END => 1 }
    				);
    			}
	
				# Close off Capacity Optimization ...
				push @{$tobj[$cnt]{$layout}}, { SF_END => 1 };
			}
		}
		
		#------------------------------------------------
		# Advanced tab of Checkpoint devices
		#------------------------------------------------
		$section = Section($Policy, Object2Page($Policy, 'CheckPoint'), 'CheckPoint', 'Advanced');
		$layout  = 'OBJ_'.$layout{'Advanced'};
		if ($section) {
			push @{$tobj[$cnt]{$layout}}, Advanced($Policy, $obj, $name, $type, $section);
		}

		# Check for empty cells and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_PRODUCTS}}) { push @{$tobj[$cnt]{OBJ_PRODUCTS}}, { VOID => 1 } }
		if (!defined @{$tobj[$cnt]{OBJ_PROP1}}) { push @{$tobj[$cnt]{OBJ_PROP1}}, { VOID => 1 } }
		if (!defined @{$tobj[$cnt]{OBJ_PROP2}}) { push @{$tobj[$cnt]{OBJ_PROP2}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}

	$template->param(CHECKPOINT=>\@tobj);
}

