package Sections2html;

use strict;
use CheckPoint;
use Constants;
use object2html;
use Options;
use Strings;

require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(Topology NAT VPN Firewall1_GX Advanced);

# Initialize special marker signs ...
my $Check      = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/check.png>';
my $CheckTick  = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/checktick.png>';
my $Mark       = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/mark.png>';
my $MarkDot    = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/markdot.png>';
my $Spacer     = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/spacer.png>';
my $None       = '<I>(none)</I>';

##########################################################################
# Routine: Topology
#
# Description:
# Routine to create the Topology section. This section is used in 
# several object types. 
#
# Parameters:
# $rb_name    Rulebase name
# $obj        Reference to object in CheckPoint database
# $name       Name of the object
# $type       Type of the object as determined in Constants.pm
# $state      State of the section (open or closed)
#
# Returns:
# @section    An array with the section to be imported directly
#             in the templates.pm library
##########################################################################
sub Topology {
	my $rb_name = shift;
	my $obj     = shift;
	my $name    = shift;
	my $type    = shift;
	my $state   = shift;
	
	my (@section, @section1, @section2);
	my (%list, %var);
	my (@var, $var, $val);
	
	#------------------------------------------------
	# Interfaces ...
	#------------------------------------------------
	@var = GetMembers(GetKey($obj, 'interfaces'));
	if (@var) {
		if ($type =~ /Node|Host/) {
			# Without AntiSpoofing information ...
			push @section1, {	
				HEAD => 1, 
				OBJ_VAL1 => html('Name'), 
				OBJ_VAL2 => html('IP address'), 
			};
			foreach $var (@var) {
				$list{GetKey($var, 'officialname')}=$var;
			}					
			foreach $var (sort keys %list) {
				$var = $list{$var};
				push @section1, {
					OBJ_VAL1 => html(stripquotes(GetKey($var, 'officialname'))), 
					OBJ_VAL2 => GetKey($var, 'ipaddr').'/'.$Subnet{GetKey($var, 'netmask')}
				}
			}
		} else {
			# With AntiSpoofing information ...
			push @section1, {	
				HEAD => 1, 
				OBJ_VAL1 => html('Name'), 
				OBJ_VAL2 => html('IP address'), 
				OBJ_VAL3 => html('Allowed IP addresses')
			};
		
			foreach $var (@var) {
				$list{GetKey($var, 'officialname')}=$var;
			}					
			foreach $var (sort keys %list) {
				# Collect AntiSpoofing information ...
				$var = $list{$var};
				%var = {};
				$val = '';
				if (($type eq 'Cluster Member') && (GetKey($var, 'shared') eq 'true')) {
					$val=html('(cluster interface) ');
				} elsif (GetKey($var, 'netaccess:leads_to_internet') eq 'true') {
					%var = %{txt2html('Internet', 'any_object', 'black')};
				} elsif (GetKey($var, 'netaccess:access') eq 'this') {
					%var = %{txt2html('This Net', 'network', 'black')};
				} elsif (GetKey($var, 'netaccess:access') eq 'specific') {
					%var = %{obj2html($rb_name, GetKey($var, 'netaccess:allowed:Table'), GetKey($var, 'netaccess:allowed:Name'), $_HREF)};
				} elsif (GetKey($var, 'netaccess:access') eq 'undefined') {
					$val = html('(not defined) ');
				}
				if (!$val) { $val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }
				
				push @section1, {
					OBJ_VAL1 => html(stripquotes(GetKey($var, 'officialname'))), 
					OBJ_VAL2 => GetKey($var, 'ipaddr').'/'.$Subnet{GetKey($var, 'netmask')},
					OBJ_VAL3 => $val
				}
			}
		}
			
	}

	#------------------------------------------------
	# VPN Domain ...
	#------------------------------------------------
	if ((GetKey($obj, 'VPN_1') eq 'true') || (GetKey($obj, 'third_party_encryption') eq 'true')) {
		if ($type =~ /Host/) {
			push @section2, {	OBJ_VAL1 => $CheckTick,
								OBJ_VAL2 => html('This Host') };
		} else {
			%var = %{obj2html($rb_name, GetKey($obj, 'manual_encdomain:Table'), GetKey($obj, 'manual_encdomain:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section2,	{	OBJ_VAL1 => (GetKey($obj, 'encdomain') eq 'addresses_behind_gw')?$MarkDot:$Mark,
								OBJ_VAL2 => html('All IP addresses behind interfaces') },
							{	OBJ_VAL1 => (GetKey($obj, 'encdomain') eq 'manual')?$MarkDot:$Mark,
								OBJ_VAL2 => html('Manually defined: ').
											((GetKey($obj, 'encdomain') eq 'manual')?$var:$Spacer) };
		}
		
	} elsif ((GetKey($obj, 'type') eq 'sofaware_gateway') &
			 (GetKey($obj, 'sofaware_vpn') eq 'true') & 
			 (GetKey($obj, 'sofaware_mode') eq 'SiteToSite')) {
		%var = %{obj2html($rb_name, GetKey($obj, 'manual_encdomain:Table'), GetKey($obj, 'manual_encdomain:Name'), $_HREF)};
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		push @section2,	{	OBJ_VAL1 => (GetKey($obj, 'encdomain') eq 'addresses_behind_gw')?$MarkDot:$Mark,
							OBJ_VAL2 => html('This gateway') },
						{	OBJ_VAL1 => (GetKey($obj, 'encdomain') eq 'manual')?$MarkDot:$Mark,
							OBJ_VAL2 => html('Manually defined: ').
										((GetKey($obj, 'encdomain') eq 'manual')?$var:$Spacer) };
	}

	#------------------------------------------------
	# Combine the two subsections ...
	#------------------------------------------------
	if (@section1 || @section2) {
		push @section, {
			SF_BEG => [{	SECTION  => html('Topology'),
							OBJ_NAME => $name,
							STATE	 => ($state eq $_OPEN) }]
		};
		if (@section1) {
			push @section, { T1 => \@section1, C3 => 1 };
		};
		if (@section2) {
			push @section, { 
				SUBSECT_BEG => html('VPN Domain'), 
				S1 => \@section2,
				SUBSECT_END => 1 
			};
		}
		push @section, { SF_END => 1 };
	}

	return @section;
}

##########################################################################
# Routine: NAT
#
# Description:
# Routine to create the NAT section. This section is used in 
# several object types. 
#
# Parameters:
# $rb_name    Rulebase name
# $obj        Reference to object in CheckPoint database
# $name       Name of the object
# $type       Type of the object as determined in Constants.pm
# $state      State of the section (open or closed)
#
# Returns:
# @section    An array with the section to be imported directly
#             in the templates.pm library
##########################################################################
sub NAT {
	my $rb_name = shift;
	my $obj     = shift;
	my $name    = shift;
	my $type    = shift;
	my $state   = shift;
	
	my $section;
	my (@section, @section1, @section2);
	my (%list, %var);
	my (@var, $var, $val, @communities);
	my ($gateway, $CP_vpn, $sofaware_vpn, $external_vpn);

	#------------------------------------------------
	# NAT configuration
	#------------------------------------------------
	if (GetKey($obj, 'add_adtr_rule') eq 'true') {
		push @section, {
			SF_BEG => [{	SECTION  => html('NAT'),
							OBJ_NAME => $name,
							STATE	 => ($state eq $_OPEN) }]
		};

		if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
			push @section, (
				{ S1 => 	[{ 	CP2 => 1, 
								OBJ_VAL1 => html('Translation method:'), 
								OBJ_VAL2 => html('Static') },
							 { 	CP2 => 1, 
								OBJ_VAL1 => html('Translate to IP address:'), 
								OBJ_VAL2 => GetKey($obj, 'valid_ipaddr') }] }
			);
		} else {
			push @section, (
				{ S1 => 	[{ 	CP2 => 1, 
								OBJ_VAL1 => html('Translation method:'), 
								OBJ_VAL2 => html('Hide') }] },
				{ S1 => 	[{ 	OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') eq '0.0.0.0')?$MarkDot:$Mark, 
								OBJ_VAL2 => html('Hide behind Gateway'),
								OBJ_VAL3 => $Spacer },
							 { 	CP3 => 1,
								OBJ_VAL1 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?$MarkDot:$Mark, 
								OBJ_VAL2 => html('Hide behind IP address'),
								OBJ_VAL3 => (GetKey($obj, 'valid_ipaddr') ne '0.0.0.0')?html(GetKey($obj, 'valid_ipaddr')):'-' }] }
			);
		}
		%var = %{obj2html($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		push @section, (
			{ S1 => 	[{ 	CP2 => 1, 
							OBJ_VAL1 => html('Install on Gateway:'), 
							OBJ_VAL2 => $var||'-' }] }
		);
		push @section, { SF_END => 1 };
	}        
	return @section;
}


##########################################################################
# Routine: VPN
#
# Description:
# Routine to create the VPN and VPN Advanced section. This section is used in 
# several object types. 
#
# Parameters:
# $rb_name    Rulebase name
# $obj        Reference to object in CheckPoint database
# $name       Name of the object
# $type       Type of the object as determined in Constants.pm
# $state      State of the section (open or closed)
#
# Returns:
# @section    An array with the section to be imported directly
#             in the templates.pm library
##########################################################################
sub VPN {
	my $rb_name = shift;
	my $obj     = shift;
	my $name    = shift;
	my $type    = shift;
	my $state   = shift;
	
	my $section;
	my (@section, @section1, @section2);
	my (%list, %var);
	my (@var, $var, $val, @communities);
	my ($gateway, $CP_vpn, $sofaware_vpn, $external_vpn);

	# Check if VPN is enabled ...
	if ((GetKey($obj, 'VPN_1') ne 'true') && 
		(GetKey($obj, 'sofaware_vpn') ne 'true') && 
		(GetKey($obj, 'third_party_encryption') ne 'true') && 
		(GetKey($obj, 'management') ne 'true')) {
		return @section;
	}

	# Define characteristics ...
	$gateway      = (GetKey($obj, 'type') =~ /gateway/);
	$CP_vpn       = ((GetKey($obj, 'VPN_1') eq 'true') || (GetKey($obj, 'vpnddcate') eq 'true'));
	$sofaware_vpn = (GetKey($obj, 'sofaware_vpn') eq 'true');
	$external_vpn = ((GetKey($obj, 'third_party_encryption') eq 'true') ||
					 ($CP_vpn && (GetKey($obj, 'location') eq 'external')) );
	
	
	#------------------------------------------------
	# VPN tab ...
	#------------------------------------------------
	push @section, {
		SF_BEG => [{	SECTION  => html('VPN'),
						OBJ_NAME => $name,
						STATE	 => ($state eq $_OPEN) }]
	};

	#------------------------------------------------
	# Determine communities for this gateway ...
	#------------------------------------------------
	foreach $var (keys %{GetKey('communities')}) {
		# Check community for membership
		foreach $val (GetMembers(GetKey('communities:'.$var.':participant_gateways'))) {
			if (GetKey($val, 'Name') eq $name) {
				push @communities, $var;
			}
		}
		foreach $val (GetMembers(GetKey('communities:'.$var.':satellite_gateways'))) {
			if (GetKey($val, 'Name') eq $name) {
				push @communities, $var;
			}
		}
	}

	my @section1;
	if (@communities) {
		# Create HTML list of objects ...
		foreach $var (@communities) {
			%var = %{obj2html($rb_name, 'communities', $var, $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section1, { OBJ_VAL1 => $var };
		}
	} else {
		push @section1, { OBJ_VAL1 => $None }
	}
			
	push @section, (
		{ TXT =>	[{	CLASS => 'note_xspc',
						TEXT  => html('Member of the following VPN Communities:') }] },
		{ SUBSECT_BEG => chr(0) },
		{ S1 => \@section1 },
		{ SUBSECT_END => 1 }
	);
	
	#------------------------------------------------
	# Matching criteria ... (certificates)
	#------------------------------------------------
	if ($external_vpn) {
		push @section, (
			{ SF_BEG => [{	SECTION  => html('Certificate Matching Criteria'),
							SUB2     => 1,
							OBJ_NAME => $name,
							STATE	 => 0 }] }
		);
		
		if (GetKey($obj, 'isakmp.allowed_ca')) {
			%var = %{obj2html($rb_name, GetKey($obj, 'isakmp.allowed_ca:Table'), GetKey($obj, 'isakmp.allowed_ca:Name'), $_HREF)};
		} else {
			%var = %{txt2html('Any', 'any_object', 'black')};
		}
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		
		push @section, (
			{ 	TXT	=>	[{	CLASS => 'note_xspc',
							TEXT  => html('Gateway must present a certificate issued by CA:') }],
				S1	=>	[{	OBJ_VAL1 => $Spacer,
							OBJ_VAL2 => $var }]
			}
		);
		# Matching criteria for certificate ...
		foreach $var (GetMembers(GetKey($obj, 'isakmp.matchpeer'))) {
			$var{$var} = 1;
		}
		
		push @section, (
			{	TXT	=>	[{	CLASS => 'note_xspc',
							TEXT  => html('The certificate should match any of the following:') }],
				S1	=>	[{	CP3 => 1,
							OBJ_VAL1 => $var{'DN'}?$CheckTick:$Check,
							OBJ_VAL2 => html('DN'),
							OBJ_VAL3 => html(stripquotes(GetKey($obj, 'isakmp.dn'))) },
						 {	C2 => 2,
							OBJ_VAL1 => $var{'"DNS/IP"'}?$CheckTick:$Check,
							OBJ_VAL2 => html('IP Address') },
						 {	CP3 => 1,
							OBJ_VAL1 => $var{'Email'}?$CheckTick:$Check,
							OBJ_VAL2 => html('e-mail'),
							OBJ_VAL3 => html(stripquotes(GetKey($obj, 'isakmp.email'))) }]
			}
		);

		push @section, { SF_END => 1 };							
	}

	#------------------------------------------------
	# Traditional mode configuration ...
	#------------------------------------------------
	if ($CP_vpn || $external_vpn) {
		push @section, (
			{ SF_BEG => [{	SECTION  => html('Traditional mode configuration'),
							SUB2     => 1,
							OBJ_NAME => $name,
							STATE	 => 0 }] }
		);
		
		# Key exchange encryption methods supported ...
		foreach $var (GetMembers(GetKey($obj, 'isakmp.encmethods'))) {
			$var{$var} = 1;
		}
		push @section, (
			{ COL_BEG => 1 },
			{ SUBSECT_BEG => html('Key exchange:') },
			{ S1 => [{	INACTIVE => (Version($obj) ge '5.0'),
						OBJ_VAL1 => ($var{'AES-256'})?$CheckTick:$Check,
						OBJ_VAL2 => html('AES-256') },
					 {	OBJ_VAL1 => ($var{'3DES'})?$CheckTick:$Check,
						OBJ_VAL2 => html('3DES') },
					 {	OBJ_VAL1 => ($var{'DES'})?$CheckTick:$Check,
						OBJ_VAL2 => html('DES') },
					 {	OBJ_VAL1 => ($var{'CAST'})?$CheckTick:$Check,
						OBJ_VAL2 => html('CAST') }] },
			{ SUBSECT_END => 1 }
		);
	
		# Data integrity methods supported ...
		foreach $var (GetMembers(GetKey($obj, 'isakmp.hashmethods'))) {
			$var{$var} = 1;
		}
		push @section, (
			{ COL_NXT => 1 },
			{ SUBSECT_BEG => html('Data integrity:') },
			{ S1 => [{	OBJ_VAL1 => ($var{'MD5'})?$CheckTick:$Check,
						OBJ_VAL2 => html('MD5') },
					 {	OBJ_VAL1 => ($var{'SHA1'})?$CheckTick:$Check,
						OBJ_VAL2 => html('SHA1') }] },
			{ SUBSECT_END => 1 },
			{ COL_END => 1 }
		);
		
		# Authentication methods ...
		foreach $var (GetMembers(GetKey($obj, 'isakmp.authmethods'))) {
			$var{$var} = 1;
		}
		push @section, (
			{ SUBSECT_BEG => html('Support authentication methods:') },
			{ S1 => [{	OBJ_VAL1 => ($var{'pre-shared'})?$CheckTick:$Check,
						OBJ_VAL2 => html('Pre-Shared Secret') },
					 {	OBJ_VAL1 => ($var{'signatures'})?$CheckTick:$Check,
						OBJ_VAL2 => html('Public Key Signatures') }] }
		);

		if (($var{'signatures'}) && !$external_vpn) {
			foreach $var (GetMembers(GetKey($obj, 'isakmp.allowed_cert'))) {
				$var{$var} = 1;
			}
    		push @section, (
    			{ 	SUBSECT_BEG => html('Allowed certificates') },
    			{ 	TXT =>	[{	CLASS => 'note_xspc',
    							TEXT  => html('When negotiating with a locally managed peer gateway:') }],
    				S1 =>	[{	OBJ_VAL1 => $var{'Any'}?$MarkDot:$Mark,
    							OBJ_VAL2 => html('The gateway can use any of its certificates') },
    						 {	OBJ_VAL1 => $var{'defaultCert'}?$MarkDot:$Mark,
    						 	OBJ_VAL2 => html('The gateway must use a certificate issued by this;Certificate Authority') }]
    			}
    		);
    		if ($var{'defaultCert'}) {
    			%var = %{obj2html($rb_name, GetKey($obj, 'isakmp.allowed_ca:Table'), GetKey($obj, 'isakmp.allowed_ca:Name'), $_HREF)};
    			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    			push @section, 	{	
    				S1 => [{	OBJ_VAL1 => $Spacer,
    							OBJ_VAL2 => $var }] 
    			};
    		}
    		push @section, { SUBSECT_END => 1 };
		}
	
		push @section, (
			{ SUBSECT_END => 1 },
			{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'exportable') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Exportable for SecuRemote/SecureClient') }] }
		);
		
		#------------------------------------------------
		# Traditional Mode Advanced ...
		#------------------------------------------------
		push @section, (
			{ SF_BEG => [{	SECTION  => html('Traditional mode Advanced'),
							SUB2     => 1,
							OBJ_NAME => $name,
							STATE	 => 0 }] }
		);
		
		if (Version($obj) ge '5.0') {
    		# Diffie-Hellman groups:
    		foreach $var (GetMembers(GetKey($obj, 'isakmp.phase1_DH_groups'))) {
    			$var{GetKey($var, 'Name')} = 1;
    		}
    		push @section, (
    			{	TXT =>	[{	CLASS => 'note_xspc',
    							TEXT  => html('Support Diffie-Hellman groups for IKE (phase 1) Security associations') }],
    				S1 =>	[{	OBJ_VAL1 => $var{'"Group 1 (768 bit)"'}?$CheckTick:$Check,
    							OBJ_VAL2 => html('Group 1 (768 bit)') },
    						 {	OBJ_VAL1 => $var{'"Group 2 (1024 bit)"'}?$CheckTick:$Check,
    							OBJ_VAL2 => html('Group 2 (1024 bit)') },
    						 {	OBJ_VAL1 => $var{'"Group 5 (1536 bit)"'}?$CheckTick:$Check,
    							OBJ_VAL2 => html('Group 5 (1536 bit)') }]
    			}
    		);
    	}
    	
		if (Version($obj) ge '5.0') {
    		# Rekeying Parameters
    		push @section, (
    			{	SUBSECT_BEG => html('Rekeying Parameters'),
    				S1 =>	[{	CP2 => 1, C1 => 2,
    							OBJ_VAL1 => html('Renegotiate IKE (phase 1) Security associations every'),
    							OBJ_VAL2 => GetKey($obj, 'isakmp.phase1_rekeying_time'),
    							OBJ_VAL3 => html('Minutes') },
    						 {	CP2 => 1, C1 => 2,
    							OBJ_VAL1 => html('Renegotiate IPSEC (phase 2) Security associations every'),
    							OBJ_VAL2 => GetKey($obj, 'isakmp.phase2_rekeying_time'),
    							OBJ_VAL3 => html('Seconds') },
    						 {	INACTIVE => (GetKey($obj, 'isakmp.phase2_use_rekeying_kbytes') eq 'false'),
    						 	CP3 => 1,
    						 	OBJ_VAL1 => $CheckTick,
    							OBJ_VAL2 => html('Renegotiate IPSEC (phase 2) Security associations every'),
    							OBJ_VAL3 => GetKey($obj, 'isakmp.phase2_rekeying_kbytes'),
    							OBJ_VAL4 => html('KBytes') },
    						 {	INACTIVE => (GetKey($obj, 'isakmp.phase2_use_rekeying_kbytes') eq 'true'),
    						 	OBJ_VAL1 => $Check,
    							OBJ_VAL2 => html('Renegotiate IPSEC (phase 2) Security associations every') }],
    				SUBSECT_END => 1
    			}
    		);
    	}
		
		# Misc.
		push @section, (
			{	SUBSECT_BEG => html('Misc'),
				S1 =>	[{	OBJ_VAL1 => (GetKey($obj, 'ISAKMP_aggressive_support') eq 'true')?$CheckTick:$Check,
							OBJ_VAL2 => html('Support agressive mode') }]
			}
		);
		push @section, { SF_END => 1 };

		# End of Traditional Mode Advanced ...				
		push @section, { SF_END => 1 };
	}

	#------------------------------------------------
	# Backup gateways ...
	#------------------------------------------------
	if ($gateway && $CP_vpn) {
		if (GetKey('properties:firewall_properties:enable_mep_configuration') eq 'true') {
    		$var = '';
    		if (GetKey($obj, 'backup_gw') eq 'true') {
    			%var = %{obj2html($rb_name, GetKey($obj, 'backup_gateway:Table'), GetKey($obj, 'backup_gateway:Name'), $_HREF)};
    			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
    		}
    		push @section, (
    			{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'backup_gw') eq 'true')?$CheckTick:$Check,
    						OBJ_VAL2 => html('Use Backup Gateways:'),
    						OBJ_VAL3 => $var }] }
    		);
		}
	}


	#------------------------------------------------
	# VPN Advanced ...
	#------------------------------------------------
	if (!$sofaware_vpn) {
		push @section, (
			{ SF_BEG => [{	SECTION  => html('VPN Advanced'),
							SUB      => 1,
							OBJ_NAME => $name,
							STATE	 => 0 }] },
			{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'ISAKMP_subnet_support') eq 'true')?$CheckTick:$Check,
						OBJ_VAL2 => html('Support key exchange for subnets') }] }
		);

		if (($gateway && !($external_vpn)) && (Version($obj) ge '5.03')) {
			push @section, (
				{ SUBSECT_BEG => html('Restart Options') },
				{ S1 => [{	OBJ_VAL1 => (GetKey($obj, 'ike_support_crash_recovery_sr') eq 'true')?$CheckTick:$Check,
							OBJ_VAL2 => html('Perform an organized shutdown of tunnels;upon gateway restart') }] },
				{ SUBSECT_END => 1 },
			);
		}
		push @section, { SF_END => 1 };
	}


	push @section , { SF_END => 1 };
	
	return @section;
	
} # End of sub VPN


##########################################################################
# Routine: Firewall1_GX
#
# Description:
# Routine to create the Firewall-1 GX section. This section is used in 
# several object types. 
#
# Parameters:
# $rb_name    Rulebase name
# $obj        Reference to object in CheckPoint database
# $name       Name of the object
# $type       Type of the object as determined in Constants.pm
# $state      State of the section (open or closed)
#
# Returns:
# @section    An array with the section to be imported directly
#             in the templates.pm library
##########################################################################
sub Firewall1_GX {
	my $rb_name = shift;
	my $obj     = shift;
	my $name    = shift;
	my $type    = shift;
	my $state   = shift;
	
	my (@section);
	
	if (GetKey($obj, 'enforce_gtp_rate_limit') eq 'true') {
		push @section, (
			{ 	SF_BEG => [{	SECTION  => html('FireWall-1 GX'), 
								OBJ_NAME => $name, 
								STATE    => ($state eq $_OPEN) }],
				S1 =>	[{	CP3 => 1,
							OBJ_VAL1 => $CheckTick,
							OBJ_VAL2 => html('Enforce GTP Signal packet rate limit'),
							OBJ_VAL3 =>	GetKey($obj, 'gtp_rate_limit'),
							OBJ_VAL4 =>	html('PDU/sec') }],
				SF_END => 1
			}
		);
	}
	return @section;
	
} # End of sub Firewall-1_GX


##########################################################################
# Routine: Advanced
#
# Description:
# Routine to create the Advanced section. This section is used in 
# several object types. In most cases only SNMP is involved, sometimes
# more.
#
# Parameters:
# $rb_name    Rulebase name
# $obj        Reference to object in CheckPoint database
# $name       Name of the object
# $type       Type of the object as determined in Constants.pm
# $state      State of the section (open or closed)
#
# Returns:
# @section    An array with the section to be imported directly
#             in the templates.pm library
##########################################################################
sub Advanced {
	my $rb_name = shift;
	my $obj     = shift;
	my $name    = shift;
	my $type    = shift;
	my $state   = shift;
	
	my (@section, @section1);
	my (%var, $var, $obj1);
	
	my $advanced  = (
		($type eq 'Gateway Cluster') || ($type eq 'Gateway') ||
		(($type eq 'Check Point Host') && (GetKey($obj, 'primary_management') eq 'true'))
	);
	my $snmp_only = (
		($type eq 'Gateway Node') || ($type eq 'Host Node') || ($type eq 'Check Point Host') || ($type =~ /Router/)
	);
			
	if ($advanced || $snmp_only) {
		# SNMP ...
		if ((GetKey($obj, 'read_community') ne '') || (GetKey($obj, 'write_community') ne '')) {
			push @section1, (
				{	S1 =>	[{	CP2 => 1,
								OBJ_VAL1 => html('sysName:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'sysName'))) },
							 {	CP2 => 1,
								OBJ_VAL1 => html('sysLocation:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'sysLocation'))) },
							 {	CP2 => 1,
								OBJ_VAL1 => html('sysContact:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'sysContact'))) },
							 {	CP2 => 1,
								OBJ_VAL1 => html('sysDescription:'),
								OBJ_VAL2 => html(stripquotes(GetKey($obj, 'sysDescr'))) }]
				}
			);
		}
	}
	
	if ($advanced) {
		if (Version($obj) ge '5.0') {
    		# SMTP ...
    		push @section1, (
    			{ 	SF_BEG => [{	SECTION  => html('SMTP'), 
    								SUB      => 1,
    								OBJ_NAME => $name, 
    								STATE    => 0 }] },
    			{	S1 =>	[{	CP2 => 1,
    							OBJ_VAL1 => html('Connection timeout:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:timeout'),
    							OBJ_VAL3 => html('Sec') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Dequeuer scan period:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:scan_period'),
    							OBJ_VAL3 => html('Sec') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Mail resend period:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:resend_period'),
    							OBJ_VAL3 => html('Sec') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Mail abandon time:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:abandon_time'),
    							OBJ_VAL3 => html('Sec') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Maximum number of recipients:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:maxrecipients') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Maximum generated concurrent connections:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:max_conns') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Maximum generated concurrent connections per site:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:max_conns_per_site') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html("Don't accept mail larger than:"),
    							OBJ_VAL2 => GetKey($obj, 'smtp:max_mail_size'),
    							OBJ_VAL3 => html('Kb') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Maximum number of mail messages in spool:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:spool_limit') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Check if spool goes over/under the limit every:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:spool_limit_scan_period'),
    							OBJ_VAL3 => html('Sec') }] },
    			{	S1	=>	[{	CP2 => 1,
    							OBJ_VAL1 => html('Spool Directory:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:rundir') },
    						 {	CP2 => 1,
    							OBJ_VAL1 => html('Postmaster Name:'),
    							OBJ_VAL2 => GetKey($obj, 'smtp:postmaster') }] },
    			{	SF_END => 1 }
    		);			
		}
		
		# SAM ...
		push @section1, (
			{ 	SF_BEG => [{	SECTION  => html('SAM'), 
								SUB      => 1,
								OBJ_NAME => $name, 
								STATE    => 0 }] },
			{	TXT	=>	[{	CLASS => 'note',
							TEXT  => html('Using the SAM protocol, configure the manner in which dynamic;').
									 html('restrictions are added to the VPN-1 & FW-1 Rule Base.') }] },
			{ 	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'sam_allow_remote_request') eq 'true')?$CheckTick:$Check,
							OBJ_VAL2 => html("Forward SAM clients' request to other SAM servers") }] }
		);

		my $early = GetKey($obj, 'sam_proxy_backwards_compatibility_security_method');
		push @section1, (
			{	SUBSECT_BEG	=> html('When connecting to this SAM server') },
			{ 	S1	=>	[{	OBJ_VAL1 => $early?$CheckTick:$Check,
							OBJ_VAL2 => html("Use early versions compatibility mode") }] }
		);
		if ($early) {
			push @section1, (
				{	S1	=>	[{	OBJ_VAL1 => $Spacer,
						 		OBJ_VAL2 => ($early eq 'opsec')?$MarkDot:$Mark,
						 		OBJ_VAL3 => html('Clear (opsec)') },
							 {	OBJ_VAL1 => $Spacer,
						 		OBJ_VAL2 => ($early eq 'auth_opsec')?$MarkDot:$Mark,
						 		OBJ_VAL3 => html('OPSEC Authentication (auth_opsec)') },
							 {	OBJ_VAL1 => $Spacer,
						 		OBJ_VAL2 => ($early eq 'ssl_opsec')?$MarkDot:$Mark,
						 		OBJ_VAL3 => html('OPSEC SSL (ssl_opsec)') },
							 {	OBJ_VAL1 => $Spacer,
						 		OBJ_VAL2 => ($early eq 'ssl_clear_opsec')?$MarkDot:$Mark,
						 		OBJ_VAL3 => html('OPSEC SSL Clear (ssl_clear_opsec)') },
							 ] }
			);
		}
		push @section1, (
			{ 	S1	=>	[{	CP3 => 2,
							OBJ_VAL1 => (GetKey($obj, 'sam_enable_purge_history_file') eq 'true')?$CheckTick:$Check,
							OBJ_VAL2 => html('Purge SAM file when it reaches:'),
							OBJ_VAL3 => (GetKey($obj, 'sam_enable_purge_history_file') eq 'true')?GetKey($obj, 'sam_purge_file_start_size'):'-',
							OBJ_VAL4 => html('KBytes') }] }
		);

		push @section1, { SUBSECT_END => 1 };

		# Close of SAM
		push @section1, { SF_END => 1 };

		if (Version($obj) ge '5.0') {
			# Connection Persistence ...
			push @section1, (
				{ 	SF_BEG => [{	SECTION  => html('Connection Persistence'), 
									SUB      => 1,
									OBJ_NAME => $name, 
									STATE    => 0 }] },
				{	S1	=>	[{	OBJ_VAL1 => html('Handling established connections when installing a new policy:') }] },
				{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'fw_keep_old_conns') eq 'true')?$MarkDot:$Mark,
								OBJ_VAL2 => html('Keep all connections') },
							 {	OBJ_VAL1 => (GetKey($obj, 'save_data_conns') eq 'true')?$MarkDot:$Mark,
								OBJ_VAL2 => html('Keep data connections') },
							 {	OBJ_VAL1 => ((GetKey($obj, 'fw_keep_old_conns') eq 'false') &&
											 (GetKey($obj, 'save_data_conns')   eq 'false'))?$MarkDot:$Mark,
								OBJ_VAL2 => html('Rematch connections') }] },
				{	SF_END => 1 }
			);
		}
		
		# Permissions to Install ...
		push @section1, (
			{ 	SF_BEG => [{	SECTION  => html('Permissions to Install'), 
								SUB      => 1,
								OBJ_NAME => $name, 
								STATE    => 0 }] },
			{	TXT	=>	[{	CLASS => 'note',
							TEXT  => html('Only the selected groups are allowed to;install policies on this object.') }] }
		);
		
		$var = HTMLMemberList($rb_name, GetKey($obj, 'AdminInfo:object_permissions:manage'), $_HREF);
		if (!@{$var}) { $var = [{ OBJ_TEXT => $None }]	};
		push @section1, { OBJLIST => $var };
		push @section1, { SF_END => 1 };

		if (Version($obj) ge '5.0') {
			# SYNDefender >= NG ...
			push @section1, (
				{ 	SF_BEG => [{	SECTION  => html('SYNDefender'), 
									SUB      => 1,
									OBJ_NAME => $name, 
									STATE    => 0 }] },
				{	TXT	=>	[{	CLASS => 'note_bold',
								TEXT  => html("Note: All modules' SYNDefender settings might've been;overriden by the SmartDefense configuration!") }] }
			);

			if (Version($obj) ge '5.03') {
				# SYNDefender >= NG FP3 ...
				push @section1, (
					{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'asm_synatk') eq 'true')?$CheckTick:$Check,
									OBJ_VAL2 => html('Activate SYN Attack protection') }] }
				);
				if (GetKey($obj, 'asm_synatk') eq 'true') {
					push @section1, (
						{	S1=>	[{	CP3 => 1, C2 => 2,
										OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => html('Timeout for SYN attack identification:'),
										OBJ_VAL3 => GetKey($obj, 'asm_synatk_timeout'),
										OBJ_VAL4 => html('seconds') },
									 {	OBJ_VAL1 => $Spacer,
										OBJ_VAL2 => (GetKey($obj, 'asm_synatk_external_only') eq 'true')?$CheckTick:$Check,
										OBJ_VAL3 => html('Protect external interface only') }] }
					);
					push @section1, (
						{	SUBSECT_BEG => html('SYN Attack Detection') },
						{	S1	=>	[{	CP2 => 1,
										OBJ_VAL1 => html('Switch to SYN Relay Defense upon;detection of at least'),
										OBJ_VAL2 => GetKey($obj, 'asm_synatk_threshold'),
										OBJ_VAL3 => html('SYN packets per timeout') }] },
						{	SUBSECT_END => 1 },
						{	SUBSECT_BEG => html('Logging Options') },
						{	S1	=>	[{	CP2 => 1,
										OBJ_VAL1 => html('Track:'),
										OBJ_VAL2 => $Track{GetKey($obj, 'asm_synatk_log')} },
									 {	CP2 => 1,
										OBJ_VAL1 => html('Track level:'),
										OBJ_VAL2 => $SYNTrack{GetKey($obj, 'asm_synatk_log_level')} }] },
						{	SUBSECT_END => 1 }
					);
				}
				
			} else {
				# SYNDefender < NG FP3
				push @section1, (
					{	SUBSECT_BEG => html('SYN attacks defense method:') },
					{	S1	=>	[{	OBJ_VAL1 => (GetKey($obj, 'fwsynatk_method') eq '0')?$MarkDot:$Mark,
									OBJ_VAL2 => html('None') },
								 {	OBJ_VAL1 => (GetKey($obj, 'fwsynatk_method') eq '1')?$MarkDot:$Mark,
									OBJ_VAL2 => html('SYN relay') },
								 {	OBJ_VAL1 => (GetKey($obj, 'fwsynatk_method') eq '3')?$MarkDot:$Mark,
									OBJ_VAL2 => html('Passive SYN gateway') }] }
				);
								
				if (GetKey($obj, 'fwsynatk_method')) {
					push @section1, (
						{	S1	=>	[{	CP2 =>1,
										OBJ_VAL1 => html('Timeout for SYN attack identification:'),
										OBJ_VAL2 => number(GetKey($obj, 'fwsynatk_timeout')),
										OBJ_VAL3 => html('seconds') },
									 {	CP2 => 1, INACTIVE => (GetKey($obj, 'fwsynatk_method') lt '3'),
										OBJ_VAL1 => html('Maximum protected sessions:'),
										OBJ_VAL2 => number(GetKey($obj, 'fwsynatk_max')) }] },
						{ 	S1	=>	[{	OBJ_VAL1 => GetKey($obj, 'fwsynatk_warning')?$CheckTick:$Check,
										OBJ_VAL2 => html('Display warning messages') }] }
					);
				}
				push @section1, { SUBSECT_END => 1 };
			}

			push @section1, { SF_END => 1 };
		}

	} # End of if ($advanced)
	
	# Complete section Advanced ...
	if (@section1) {
		if ($advanced) {
			push @section, (
				{ 	SF_BEG => 	[{	SECTION  => html('Advanced'), 
									OBJ_NAME => $name, 
									STATE    => ($state eq $_OPEN) }] }
			);
		} else {
			push @section, (
				{ 	SF_BEG => 	[{	SECTION  => html('SNMP'), 
									OBJ_NAME => $name, 
									STATE    => ($state eq $_OPEN) }] }
			);
		}			
		push @section, @section1;
		push @section, { SF_END => 1 };
	}
	
	return @section;
	
} # end of sub Advanced

1;
