package OutputHTML;

use strict;
use warnings;

our (@ISA, @EXPORT);

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(OutputHTMLPages);


# Constant definitions ...
use Constants;

# Special purpose packages which store their own data ...
use CheckPoint;
use Options;

# HTML Output specific library ...
use tables2html;
use settings2html;

# General purpose libraries used ...
use Template;
use Strings;

# Define global variable (to this package) ...
my $SCR_NAME;		# Source code program name
my $VERSION;		# Source code version number


##########################################################################
# Routine: OutputHTMLPages
#
# Description:
# Write Objects to HTML files. Relating to the page definitions
# in CPpages.def, the different pages are generated
#
# Parameters:
# (none)
#
# Returns: nothing, html pages are written to file
sub OutputHTMLPages {
	use strict;
	my ($htmlpage, $filename, $file, @files);
	my ($Policy, $Policy_dir);
	my (@pages, $page, @tables, $table);
	my (%NrTables, %NrObjects);

	$SCR_NAME=shift;
	$VERSION=shift;

	# Create directory for HTML (will return error if already exists, but this is ignored)
	$Policy_dir = Option('HTMLPath'); 
	mkdir $Policy_dir;
	
	#--------------------------------------------------------------------
	# Create GLOBAL PROPERTIES html code for this management server ...
	#--------------------------------------------------------------------
	print "Processing Global Properties\n";

	$htmlpage = Template->new(	filename=>Option('TmplPath').'/globalproperties.tmpl',
								cache => 1);
    		
	# Remove old file ...
	unlink Option('HTMLPath').'/_globalproperties.html';

	# Open the output file ...
	open (FH, ">",Option('HTMLPath').'/_globalproperties.html')
		or die "Cannot open the temp OutputFile ".Option('HTMLPath')."/_globalproperties.html!\n\n";
   
	$htmlpage = HTMLGlobalProperties($htmlpage);

	# Write filled template file to output ...
	if ($htmlpage ne '') {
		print FH $htmlpage->output;
	}
	# Close the html page ...
	close (FH);

	#--------------------------------------------------------------------
	# Create Policy dependent html pages ...
	#--------------------------------------------------------------------
	foreach $Policy (keys %{GetKey('policies_collections')}) {
		# Check if this rulebase should be converted to HTML ...
		if (Rulebase($Policy)) {
    		print "Processing Policy Package: ".$Policy."\n";
    
    		# Create directory for rulebase (will return error if already exists, but this is ignored)
    		$Policy_dir = Option('HTMLPath')."/".stripquotes($Policy); 
    		mkdir $Policy_dir;
    
    		#--------------------------------------------------------------------
    		# Remove all *.html files from the HTML directory ...
    		#--------------------------------------------------------------------
    		@files = glob($Policy_dir.'/*.html');
    		unlink @files;
    		@files = glob($Policy_dir.'/*.tmpl');
    		unlink @files;
    		
    		#--------------------------------------------------------------------
    		# Copy Global Properties html code if requested ...
    		#--------------------------------------------------------------------
    		if (Option($Policy, 'GlobalProps')) {
        		open (FH, ">", $Policy_dir.'/globalproperties.html')
        			or die "Cannot open the output file $Policy_dir/globalproperties.html!\n\n";

    			open (INFILE, Option('HTMLPath').'/_globalproperties.html');

    			while (<INFILE>) {
    				print FH strip($_)."\n" if /\S/
    			}
        		close INFILE;
    			close FH;
    		}
    		
    		#--------------------------------------------------------------------
    		# Create HEADER html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(	filename=>Option('TmplPath').'/header.tmpl',
    									cache => 1);
    
    		# Open the file ...
    		open (FH, ">",$Policy_dir.'/header.html')
    			or die "Cannot open the the OutputFile $Policy_dir/header.html!\n\n";
    
    		$htmlpage = HTMLHeader($Policy, $htmlpage);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    
    		#--------------------------------------------------------------------
    		# Create FOOTER html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(	filename=>Option('TmplPath').'/footer.tmpl',
    									cache => 1);
    
    		# Open the file ...
    		open (FH, ">",$Policy_dir.'/footer.html')
    			or die "Cannot open the the OutputFile $Policy_dir/footer.html!\n\n";
    
    		$htmlpage = HTMLFooter($Policy, $htmlpage);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    
    		#--------------------------------------------------------------------
    		# Create POLICY INSTALLATION TARGETS html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(	filename=>Option('TmplPath').'/targets.tmpl',
    									cache => 1);
    		
    		# Open the file ...
    		open (FH, ">",$Policy_dir.'/targets.html')
    			or die "Cannot open the OutputFile $Policy_dir/targets.html!\n\n";
    
    		$htmlpage = HTMLTargets($Policy, $htmlpage);

    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);

       		#--------------------------------------------------------------------
    		# Create template files for the PAGES to generate ...
    		#--------------------------------------------------------------------
    		@pages=Page($Policy, 'PAGES');
    		foreach $page (@pages) {
    			open (FH, ">", $Policy_dir.'/'.lc($page).'.tmpl')
    				or die "Cannot open the output-template file $Policy_dir/$page.tmpl!\n\n";
    
    			@tables=Page($Policy, $page, 'TABLES');
    			foreach $table (@tables) {
    				print FH '<TMPL_INCLUDE "'.Option('TmplPath').'/'.lc($table).'.tmpl">'."\n";
    			}
    			
    			close FH;
    		}		
    
    		#--------------------------------------------------------------------
    		# Create html code for all PAGES ...
    		#--------------------------------------------------------------------
    		@pages=Page($Policy, 'PAGES');
    		foreach $page (@pages) {
    			# Assume page is empty ...
    			$NrTables{$page}=0;
    			
    			# Determine template file to use ...
    			$htmlpage = Template->new(	filename=>$Policy_dir.'/'.lc($page).'.tmpl',
    										cache => 1,
    										max_includes => 100);
    			
    			@tables = Page($Policy, $page, 'TABLES');
    			foreach $table (@tables) {
    				$NrObjects{$table} = table2html($Policy, $htmlpage, $table);
    				
    				# SPECIAL CASE: the NAT table can hold automatic NAT translations ... force it to be displayed!
    				if ($table eq 'NAT') { $NrObjects{$table} += 1000 }
    				
    				# Increase nr. of tables if appropiate ...
    				if ($NrObjects{$table}) {$NrTables{$page} += 1 }
    			}

				if ($NrTables{$page}) {
        			# Open the file ...
        			open (FH, ">",$Policy_dir.'/_'.lc($page).'.html')
        				or die "Cannot open the the OutputFile $Policy_dir/_".lc($page).".html!\n\n";
        
        			# Write filled template file to output ...
        			if ($htmlpage ne '') {
        				print FH $htmlpage->output;
        			}
        			# Close the html page ...
        			close FH;
        		}
    
    		} # END foreach $page

    		#--------------------------------------------------------------------
    		# Create MENU code for this rulebase ...
    		# (this code can only be generated after the pages have been completed)
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(	filename=>Option('TmplPath').'/menu.tmpl',
    									cache => 1);
    
    		# Open the file ...
    		open (FH, ">",$Policy_dir.'/menu.html')
    			or die "Cannot open the the OutputFile $Policy_dir/menu.html!\n\n";
    
    		$htmlpage = HTMLMenu($Policy, $htmlpage, \%NrTables, \%NrObjects);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    		
    		#--------------------------------------------------------------------
    		# Create final html pages from the section files ...
    		#--------------------------------------------------------------------
    		@pages=Page($Policy, 'PAGES');
    		
    		foreach $page (@pages) {
				if ($NrTables{$page}) {
        			open (FH, ">", $Policy_dir.'/'.lc($page).'.html')
        				or die "Cannot open the output file $page.html!\n\n";
    
    				open (INFILE, $Policy_dir.'/header.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $Policy_dir.'/menu.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $Policy_dir.'/_'.lc($page).'.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $Policy_dir.'/footer.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
       			
        			close FH;
        		}
    		}		

    		#--------------------------------------------------------------------
    		# Clean up temp files ...
    		#--------------------------------------------------------------------
    		@files = glob($Policy_dir.'/*.tmpl');
    		unlink @files;
    		@files = glob($Policy_dir.'/_*.html');
    		unlink @files;
    		unlink $Policy_dir.'/header.html', $Policy_dir.'/footer.html', $Policy_dir.'/menu.html';
		}    		
	} # END foreach $Policy
	
	#--------------------------------------------------------------------
	# Clean up Global Properties file ...
	#--------------------------------------------------------------------
	unlink Option('HTMLPath').'/_globalproperties.html';
}


##########################################################################
# Routine: HTMLHeader
#
# Description:
# Write HEADER to HTML file. The output file is included 
# in the actual output later on.
#
# Parameters:
# $Policy	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 

sub HTMLHeader {
	use strict;
	
	my $Policy   = shift;
	my $template = shift;	# HEADER.TMPL
	
	# Fill header with general variables of rulebase
	$template->param(	RULEBASE     => $Policy,
						LASTMODIFIED => stripquotes(GetKey('policies_collections:'.$Policy.':AdminInfo:LastModified:Time')),
						CSSPATH      => Option('CSSPath'),
						SCRIPTPATH   => Option('ScriptPath'),
						MAINLOGO     => Option('MainLogo'),
						LOGO         => Option($Policy, 'Logo'),
						
						GLOBPROP       => Option($Policy, 'GlobalProps'),
						GLOBPROP_PAGE  => 'globalproperties.html',
						GLOBPROP_TEXT  => 'Global Properties',
						TARGET       => 1,
						TARGET_PAGE  => 'targets.html',
						TARGET_TEXT  => 'Installation Targets'
				);

	return $template;
}


##########################################################################
# Routine: HTMLFooter
#
# Description:
# Write FOOTER to HTML file. 
#
# Parameters:
# $Policy	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 

sub HTMLFooter {
	use strict;
	my $Policy   = shift;
	my $template = shift;	# FOOTER.TMPL
	
	# Fill header with general variables of rulebase
	my $now = gmtime();

	$template->param(	CreatedDate  => $now,
						Script       => $SCR_NAME,
						Version      => $VERSION
				);
				
	return $template;
}


##########################################################################
# Routine: HTMLMenu
#
# Description:
# Write Menu to HTML file. This file is later added to the actual output
#
# Parameters:
# $Policy	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
# \%ActiveTables  Reference to hash which holds if tables
#                 are empty or not
#
# Returns: 
# $template	Filled template 

sub HTMLMenu {
	use strict;
	my (@pages, $page);
	my (@tables, $table);
	my (%style, %menu, %submenu);
	my (@styles, @menus);
	my (%NrTables, %NrObjects, $cnt);
	
	my $Policy = shift;
	my $template = shift;	# MENU.TMPL
	my $ref1   = shift;
	my $ref2   = shift;
	
	%NrTables  = %{$ref1};
	%NrObjects = %{$ref2};

	# Set path to scripts in menu template ...	
	$template->param( SCRIPTPATH   => Option('ScriptPath') );

	# Discover pages to be displayed ....
	@pages=Page($Policy, 'PAGES');

	# Define styles for the menus ...
	$cnt=25;
	foreach $page (@pages) {
		if ($NrTables{$page}) {
    		my %style;
    		%style = ( 	MENU_TXT	 => $page,
    					MENU_LEFT	 => $cnt,
    					SUBMENU_LEFT => $cnt+5
    				);
    		push @styles, \%style;
    		$cnt += 100;
		}
	}
	$template->param(STYLES=>\@styles);

	foreach $page (@pages) {
		if ($NrTables{$page}) {
    		my %menu;
    		%menu = (	PAGE_TXT	=> $page,
    					PAGE_FILE 	=> lc($page)
    				);
    		@tables=Page($Policy, $page, 'TABLES');
    		foreach $table (@tables) {
    			if ($NrObjects{$table}) {
        			my %submenu;
        			%submenu = ( SECTION_TXT => $table,
        						 SECTION_FILE => lc($page)
        					);
        			push @{$menu{SUBMENUS}}, \%submenu;
    			}
    		}
			push @menus, \%menu;
    	}
	}
	$template->param(MENUS=>\@menus);
	
	return $template;
}

1
