package ExportObjects;
# Still to do:
# - how to write lists / reference objects / anonymous objects to flat file
# - how to determine from which rulebase to export the data (only in case for rulebase files themselves...)?
# - how to use template files?

use strict;
use warnings;

our (@ISA, @EXPORT);

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(ExportObjects);

# Special purpose packages which store their own data ...
use CheckPoint;
use Constants;
use Options;
use Strings;

##########################################################################
# Define global (to this package) variables
my $SCR_NAME;		# Source code program name
my $VERSION;		# Source code version number
my %Definition;		# Hash to store the options from the definition file



##########################################################################
# Export Objects according to template files
##########################################################################
# Routine: ExportObjects
#
# Description:
# Exports objects and its parameters to a flatfile defined by a separate
# definition file
#
# Parameters:
# Defintion file name
#
# Returns: nothing, output is written to the outputfile specified
sub ExportObjects {
	use strict;
	my ($Policy, $dir, $DEFfile);
	my ($section, $table, $name, $object, $AllObjects);
	my ($class, $header, $field, $sep, $lsep, $OutputFile);
	my (@name, @class, @fields, @headers);
	my (@val, $val, $first, $rulebase, $number);

	$SCR_NAME = shift;
	$VERSION  = shift;

	# Create directory for exportfile (will return error if already exists, but this is ignored)
	$dir = Option('EXPORTPATH'); 
	mkdir $dir;

	#--------------------------------------------------------------------
	# Create export files for each policy ...
	#--------------------------------------------------------------------
	foreach $Policy (keys %{GetKey('policies_collections')}) {
		# Check if this rulebase should be exported ...
		if (Rulebase($Policy)) {
    		print "Exporting objects from policy package: ".$Policy."\n";

    		# Create directory for policy package (will return error if already exists, but this is ignored)
    		$dir = Option('EXPORTPATH')."/".stripquotes($Policy); 
    		mkdir $dir;
    
			# Read export definition file for this policy package ...
			$DEFfile = Option($Policy, 'EXPORTDEF');
			if ($DEFfile) { 
				&ReadDEFfile($DEFfile);
			} else {
				# No definition file for this policy package; skipping ...
				print ">>> No definition file defined for this policy package. Skipping ...\n";
				next
			}

			# Determine if all objects need to be listed ... 
			$AllObjects = Option($Policy, 'ALLOBJECTS');

			# Process all rules first to determine which objects are used
			# in this policy (security rulebase, NAT rulebase, desktop 
			# security policy).
			if (!$AllObjects) {
				# 1. security rulebase
				my ($rulebase, $rule);
				
				foreach $rulebase (GetMembers(GetKey($HTMLTable{SECURITY}{TABLE}))) {
					if (GetKey($rulebase, 'collection:Name') eq $Policy) {
						# Found current policy ...
						foreach $rule (GetMembers(GetKey($rulebase, $HTMLTable{SECURITY}{LIST}))) {
							ObjectUsed($rule, $Policy);
						}
					}
				}
				# 2. manual nat rulebase
				foreach $rulebase (GetMembers(GetKey($HTMLTable{NAT}{TABLE}))) {
					if (GetKey($rulebase, 'collection:Name') eq $Policy) {
						# Found current policy ...
						foreach $rule (GetMembers(GetKey($rulebase, $HTMLTable{NAT}{LIST}))) {
							ObjectUsed($rule, $Policy);
						}
					}
				}
				# 3. desktop security rulebase
				foreach $rulebase (GetMembers(GetKey($HTMLTable{DESKTOP}{TABLE}))) {
					if (GetKey($rulebase, 'collection:Name') eq $Policy) {
						# Found current policy ...
						foreach $rule (GetMembers(GetKey($rulebase, $HTMLTable{DESKTOP}{LIST}))) {
							ObjectUsed($rule, $Policy);
						}
					}
				}
			}
			
        	# Go through every table to be exported ...
        	foreach $section (keys %Definition) {
        
        		# Get the definition for the next table to export ...
        		$table=$Definition{$section}{TABLE};
        		$sep=stripquotes($Definition{$section}{SEPARATOR}) || ',';
        		$lsep=stripquotes($Definition{$section}{'LIST-SEPARATOR'}) || ':';
        		$OutputFile=$Definition{$section}{OUTPUTFILE};
				if (! $OutputFile) {
					$OutputFile = Option('EXPORTPATH')."/".$Policy."/".$section.".exp";
				} elsif ($OutputFile !~ /[\\\/]/ ) {
					$OutputFile = Option('EXPORTPATH')."/".$Policy."/".$OutputFile;
				}
					
        		@name=@{$Definition{$section}{NAME}};
        		@class=@{$Definition{$section}{CLASS}};
        		@headers=@{$Definition{$section}{HEADERS}};
        		@fields=@{$Definition{$section}{FIELDS}};

        		# Open the output file ...
        		unlink $OutputFile;
        		open (FH, ">$OutputFile")
        			or die "Cannot open the OutputFile $OutputFile ...\n\n";
        			
        		# Check if table specified does exist ...
        		if (GetKey($table)) {
        			
        			# Write header line to output file ...
        			if (!@headers) {
        				#print FH $section;
						$first=0;
        				foreach $header (@fields) {
        					print FH (($first++)?$sep:"").$header;
        				}
        				print FH "\n";
        			} else {
        				#print FH $section;
						$first=0;
        				foreach $header (@headers) {
        					print FH (($first++)?$sep:"").$header;
        				}
        				print FH "\n";
        			}
        
        			if ($table =~ /rule-base/) {
        				# Handle any rule-base differently from a list of objects ...	
        				@val=GetMembers(GetKey($table));
        				foreach $val (@val) {
        					if (GetKey($val, 'collection:Name') eq $Policy) {
        						# Found the correct policy ...
        						$number=0;
        						foreach $name (@name) {
        							foreach $object (GetMembers(GetKey($val, $name))) {
       									if (GetKey($object, 'AdminInfo:ClassName') !~ /_header_/ ) { 
       										++$number;
       									}
        								# Print the requested fields ...
        								$first=0;
        								foreach $field (@fields) {
        									if (GetKey($object, 'AdminInfo:ClassName') =~ /_header_/ ) {
        										# If header rule, print no rule number ...
	       										print FH (($first++)?$sep:"").object2ascii($object, $field, $lsep);
    	    								} else {
	       										print FH (($first++)?$sep:"").object2ascii($object, $field, $lsep, $number);
    	    								}
        								}
       									print FH "\n";
        							}
        						}
        					}
        				}
        			} else {
						if (ref(GetKey($table))) {
							# table is not empty, so write lines with objects to output file ...
							
       						$number=0;
            				foreach $name (keys %{GetKey($table)}) {
            					# Check if all objects must be written else if object is used in this policy ...
								if ($AllObjects || (GetKey($table, $name, 'is_used', $Policy) eq 'true')) {
									# Check if object matches all filters ...
            						$class=GetKey($table, $name, 'AdminInfo:ClassName');
            						if (InArray($name, \@name) && InArray($class, \@class)) {
           								# Found object to export !!!
           								++$number;
        								$first=0;
   										foreach $field (@fields) {
       										print FH (($first++)?$sep:"").object2ascii($table, $name, $field, $lsep, $number);
           								}
           								print FH "\n";
            						}
            					}
            				} # End of: foreach $name
            			}
            		} #  Endif of: list or table of objects ...
        		}
        		close FH;
        	}
        }

   	}
} # End sub



##########################################################################
# Routine: ReadDEFfile
#
# Description:
# Reads the definition file and store the found options in the hash %Option
#
# Parameters:
# $DEFfile	the name of the definitions file
#
# Returns:
# nothing; data is stored in the global hash %Definition
sub ReadDEFfile {
	use strict;
	my $DEFfile = shift;
	my ($section, $key, $val);
	my (@class, @fields, @headers);

	if (! -e $DEFfile) {
		# No definition file for this policy package; skipping ...
		print ">>> Definition file $DEFfile not found for this policy package. Skipping ...\n";
		return 1;
	}
	
	open (INFILE, $DEFfile);
	
	while (<INFILE>) {
		chomp;
		if (/^\[(.*)\]/) {
			# Save value of section ...
			my $s=$1;
			
			# New section; first check out and correct the previous section ...
			if ($section) {
				if (!@{$Definition{$section}{CLASS}}) {
					push @{$Definition{$section}{CLASS}}, '.*';
				}
				if (!@{$Definition{$section}{NAME}}) {
					push @{$Definition{$section}{NAME}}, '.*';
				}
			}

			$section=strip($s);

			# Initialize variables for this section ...
			$Definition{$section}{TABLE}='';
			$Definition{$section}{SEPARATOR}='';
			$Definition{$section}{OUTPUTFILE}='';
			@{$Definition{$section}{NAME}}=();
			@{$Definition{$section}{CLASS}}=();
			@{$Definition{$section}{HEADERS}}=();
			@{$Definition{$section}{FIELDS}}=();
			
		} elsif (/^\s*[;!#]/) {
			# Commentline, skip
		} elsif (/^\s*(\S+)\s*=(.*)$/) {
			# Define a new set of variables ...
			$key=uc(strip($1)); 
			$val=strip($2); 
			
			if ($key eq 'NAME') {
				push @{$Definition{$section}{NAME}}, strip($val);
			} elsif ($key eq 'CLASS') {
				push @{$Definition{$section}{CLASS}}, strip($val);
			} elsif ($key eq 'HEADERS') {
				@headers=split(/[\s,]/, $val);
				@headers=grep(!/^\s*$/, @headers);
				push @{$Definition{$section}{HEADERS}}, @headers;
			} elsif ($key eq 'FIELDS') {
				@fields=split(/[\s,]/, $val);
				@fields=grep(!/^\s*$/, @fields);
				push @{$Definition{$section}{FIELDS}}, @fields;
			} elsif (($key eq 'SEPARATOR') ||
					 ($key eq 'LIST-SEPARATOR')) {
				if (uc($val) eq 'TAB')   {$val = "\t" }
				if (uc($val) eq 'SPACE') {$val = " " }
				if (uc($val) eq 'COMMA') {$val = "," }
				$Definition{$section}{$key}=$val;
			} else {
				$Definition{$section}{$key}=$val;
			}
		}
	}
	if ($section) {
		if (!@{$Definition{$section}{CLASS}}) {
			push @{$Definition{$section}{CLASS}}, '.*';
		}
		if (!@{$Definition{$section}{NAME}}) {
			push @{$Definition{$section}{NAME}}, '.*';
		}
	}

	close (INFILE);
}


##########################################################################
# Routine: object2ascii
#
# Description:
# Outputs the field of the object in ASCII format
#
# Parameters:
# $table	The table of the object
# $name		The name of the object to process
# $field	The field to process
# $lsep		The list-separator (if needed)
# $number	The object or rule number (optional)
#   or
# $object	The reference to the object to process
# $field	The field to process
# $lsep		The list-separator (if needed)
# $number	The object or rule number (optional)
#
# Returns:
# $val		Textbased output of field
sub object2ascii {
	use strict;
	my ($object, $table, $name, $field, $lsep, $number);
	my (@val, $val, $v);
	my (@members);
	
	$object  = shift;
	if (!ref($object)) {
		$table = $object;
		$name = shift;
		$object = GetKey($table, $name);
	}
	$field  = shift;
	$lsep   = shift;
	$number = shift;
	
	# First check for RESERVED words ...
	if (lc($field) eq 'name') {
		# Name of object; only to be used in regular tables.
		# This field is not supplied if the table exported is a rulebase. 
		$val = $name || "";

	} elsif (lc($field) eq 'number') {
		# Object- or rule number ...
		$val = $number || "";
		
	} elsif (lc($field) eq 'members') {
		# List of members ...
		foreach $val (GetMembers(GetKey($object))) {
			push @val, GetKey($val, 'Name');
		}
		$val=join ($lsep, @val);
	} else {
		$val = GetKey($object, $field);
	}

	
	# Check for more complex data structures ...
	if (ref($val) eq 'ARRAY') {
		# List of objects, use that as the value ...
		# PPW Members of an array can also be referenced objects... TO BE FIXED ...
		# PPW - need example to code the exception !!!
		$val=join($lsep, @{$val});
		
	} elsif (ref($val)) {
		# Structure is a HASH of objects ...
		@members = GetMembers($val);

		# Check out what it can be ...
		if (GetKey($val, 'Name')) {
			# Hash does have a 'Name' field! Use that ...
			$val=GetKey($val, 'Name');
		} elsif (GetKey($val, 'name')) {
			# Hash does have a 'name' field! Use that ...
			# Note: this is old syntax of CheckPoint...
			$val=GetKey($val, 'name');
			
		} elsif (GetKey($object, $field.':AdminInfo:ClassName')) {
			# Field is a CheckPoint object with properties and references ...
			if ($members[0] ne $val) {
				# Check if ther are regular members ...
				foreach $v (@members) {
					push @val, GetKey($v, 'Name');
				}
				$val=join($lsep, @val);
			}
			
			if (GetKey($object, $field.':AdminInfo:ClassName') =~ /translate(.*)/) {
				# Manual NAT translation ...
				$v = $1;
				
				if ($val eq 'Any') {
					$val = "Original";
				} else {
					# NAT object ...
					$val = $val."(".substr($v,1).")";
				}

			} elsif (GetKey($object, $field.':compound')) {
				# Compound element in rulebase ...
				foreach $v (keys %{GetKey($object, $field.':compound')}) { 
					push @val, stripquotes($v);
				}
				$val=join($lsep, @val);
			}

			
		} elsif ($members[0] eq $val) {
			# Special case:
			# HASH has no members whatsoever. Find the key/name of the object and use this as value ...
			#  ('Action' field in rulebases)
			foreach $v (keys %{$val}) {
				push @val, $v;
			}
			$val=join($lsep, @val);

		} else {
			# Field is only a referencing object (no own properties) ...
			foreach $v (@members) {
				push @val, GetKey($v, 'Name');
			}
			$val=join($lsep, @val);
		}			

			
	}
	

	# Determine if cell should be NEGATED ...
	if (GetKey($object, $field.':op') eq '"not in"') {
		$val="NOT{".stripquotes($val)."}";
	}

	return stripquotes($val);
}

sub InArray {
	my $member;
	my $string=shift;
	my $array=shift;

	foreach $member (@{$array}) {
		if ($string =~ m/($member)/) {
			return 1;
		}
	}
	return 0;
}
	
1;
