#!/usr/bin/perl

#############################################################################
# CPUsers.pl: Converts the export file created with 'fw dbexport' to a
#             CheckPoint compatible database file (users.C).
#
# Note: this script is unsupported by Checkpoint or any representatives.
#
# Copyright (C) 2004 Peter-Paul Worm
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#############################################################################
use strict;

my $SCR_NAME 	= 'CPUsers';
my $VERSION  	= '1.0';

# Constant definitions ...
my $_usersfile='users.exp';
my $_groupfile='groups.exp';
my $_outputfile='users.C';
my $LF="\n";

##########################################################################
# Data format of fields in export file
##########################################################################
# These fields are reported in 'fw dbexport' of the userdatabase. 
# This table defines the format they should be stored in the database.
# This data should also be in the file 'classes.C', but we have not used 
#  this file as of yet.	Might be a future improvement.
##########################################################################
my %usertmpl = (
	AdminInfo => {
		ClassName => 'string',
		table => 'string',
		name => 'string'
	},
	color => 'string',
	groups => 'objectlist:users',
	destinations => 'objectlist:network_objects',
	sources => 'objectlist:network_objects',
	auth_method => 'string',
	fromhour => 'time',
	tohour => 'time',
	expiration_date => 'date',
	days => 'binarylist:MON:TUE:WED:THU:FRI:SAT:SUN',
	accept_track => 'string',
	internal_password => 'string',
	SKEY_seed => 'string',
	SKEY_number => 'string',
	SKEY_passwd => 'string',
	SKEY_gateway => 'string',
	SKEY_method => 'string',
	comments => 'string',
	radius_server => 'object:servers',
	tacacs_server => 'object:servers',
	vlan_auth_list => 'list',
	userc => {
		'expire' => 'time',
		'isakmp.transform' => 'string',
		'isakmp.data.integrity' => 'string',
		'isakmp.encryption' => 'string',
		'isakmp.methods' => 'list',
		'isakmp.encmethods' => 'list',
		'isakmp.hashmethods' => 'list',
		'isakmp.authmethods' => 'list',
		'isakmp.shared.secret' => 'string'
	},
	administrator => 'boolean',
	administrator_profile => 'object',
	is_administrator_group => 'boolean',
	type => 'string'
);	

my %grouptmpl = (
	AdminInfo => {
		ClassName => 'string',
		table => 'string',
		name => 'string'
	},
	ReferenceObject => 'objectlist:users',
	color => 'string',
	groups => 'objectlist:users',
	comments => 'string',
	is_administrator_group => 'boolean',
	type => 'string'
);	


##########################################################################
# Variable declarations ...

my $usersfile;		# Holds the file(name) with exported users
my $groupfile;      # Holds the file(name) with exported groups
my $outputfile;		# Holds the user database filename to write the structure to (including path)
my $users;			# Holds the user structure in the database

##########################################################################
# Read commandline to determine in- and output file

($usersfile, $groupfile, $outputfile) = ReadCommandLine();

##########################################################################
# Initialize user hash and read input files

$users->{version}='5.0';

if ($usersfile) { ReadExportFile($usersfile, $users, \%usertmpl,  'user') }
if ($groupfile) { ReadExportFile($groupfile, $users, \%grouptmpl, 'usrgroup') }

##########################################################################
# Find all groupmembers and store them in the database
my $user;
my $group;

foreach $user (keys %{$users}) {
	if (ref($users->{$user}) eq 'HASH') {
		if ($users->{$user}{type} eq 'user') {
			if (ref($users->{$user}{groups}) eq 'ARRAY') {
				foreach $group (@{$users->{$user}{groups}}) {
					push @{$users->{$group}{ReferenceObject}}, $user;
					if (!$users->{$group}{type}) {
						AddUserValue(\%grouptmpl, $users->{$group}, 'type', 'usrgroup');
					}
				}
			}
		}
	}
}

##########################################################################
# Add AdminInfo to all objects
my $user;

foreach $user (keys %{$users}) {
	if (ref($users->{$user}) eq 'HASH') {
		AddUserValue(\%usertmpl, $users->{$user}, 'table', 'users');
		if ($users->{$user}{type} eq 'user') {
			AddUserValue(\%usertmpl, $users->{$user}, 'ClassName', 'user');
		} elsif ($users->{$user}{is_administrator_group} eq 'true') {
			AddUserValue(\%grouptmpl, $users->{$user}, 'ClassName', 'administrator_group');
		} elsif ($users->{$user}{type} eq 'usrgroup') {
			AddUserValue(\%grouptmpl, $users->{$user}, 'ClassName', 'user_group');
		}
	}
}
		

##########################################################################
# Write all users to the outputfile
my $user;		# Holds the name of the current user
my $t=0;		# Holds the current depth (number of tabs)

open (OUT, ">", $outputfile)
	or die "CPUsers.pl could not perform required conversion.\n\nReason: Cannot open the outputfile $outputfile !\n\n";

# Write first lines to output file ...
print OUT "(".$LF;
print OUT "\t"x++$t.":version (5.0)".$LF;

foreach $user (sort(keys %{$users})) {
	if (ref($users->{$user}) eq 'HASH') {
	
		print OUT "\t"x$t++.": (".fixstr($user).$LF;

		if ($users->{$user}{type} eq 'user') {
			PrintUser(\%usertmpl, $users->{$user}, $t);
		} elsif ($users->{$user}{type} eq 'usrgroup') {
			PrintUser(\%grouptmpl, $users->{$user}, $t);
		}
		print OUT "\t"x--$t.")".$LF;
	}
}

# Finish output file ...
print OUT ")".$LF;

close (OUT);


##########################################################################
#                       END OF MAIN ROUTINE                              #
##########################################################################


##########################################################################
# Routine: ReadCommandLine
#
# Description:
# Scans the commandline for in- and output file names
#
# Parameters:
# (none)
#
# Returns:
# $usersfile	- Export filename with users created by CheckPoints' export function
# $groupfile    - Export filename with groups created by 'fwm dbexport -g'
# $outputfile	- CheckPoint database file name
sub ReadCommandLine {
	use strict;
	my ($i, $arg);
	my ($usersfile, $groupfile, $outputfile);
	
	my $args=$#ARGV;
	
	$i=0;
	while ($i<=$args) {
		$arg="\L$ARGV[$i]";
		if ($arg eq "-u") {
			$i++;
			$usersfile=$ARGV[$i];
		} elsif ($arg eq "-g") {
			$i++;
			$groupfile=$ARGV[$i];
		} elsif ($arg eq "-o") {
			$i++;
			$outputfile=$ARGV[$i];
		} elsif (!$usersfile) {
			$usersfile=$ARGV[$i];
		} elsif (!$groupfile) {
			$groupfile=$ARGV[$i];
		} elsif (!$outputfile) {
			$outputfile=$ARGV[$i];
		}
		$i++;
	}
	
	if (!$usersfile) {
		die "CPUsers.pl could not perform required conversion.\n\nReason: No inputfile specified\n\n";
	} else {
		if (!$groupfile) {
			$groupfile='';
		} elsif (!$outputfile) {
			$outputfile = $_outputfile;
		}
	}

	return $usersfile, $groupfile, $outputfile;
}
	
##########################################################################
# Read the inputfile and store the users in the database
sub ReadExportFile {
	my $file  = shift;
	my $users = shift;
	my $tmpl  = shift;
	my $type  = shift;

	my @headers;		# Stores the headers of the fields in the export file
	my %struct;			# Stores the fields in the header of the export file
	my ($line, $i, $user);
	my (@fields, $field, @list);
	
	open (IN, "<", $file)
		or die "CPUsers.pl could not perform required conversion.\n\nReason: Cannot open the inputfile $file !\n\n";
	
	# First line of file holds headers ...	
	$line = <IN>;
	@headers = split(/;/, $line);
	
	for ($i=0; $i<@headers; $i++) {
		$headers[$i]=strip($headers[$i]);
		$struct{$headers[$i]}=$i;
	}
	
	while (<IN>) {
		@fields = split(/;/, $_);
		$user = $fields[$struct{name}];
		# Create hash for user ...
		$users->{$user}={};
			
		# Process User ...
		for ($i=0; $i<@headers-1; $i++) {
			$field=$headers[$i];
			# Check if value does exist, otherwise skip ...
			if (strip($fields[$i])) {
				# Check if value is a list of values ...
				if ($fields[$i] =~ /\s*{(.*)}\s*/) {
					@list = split(/,/, $1);
					AddUserValue($tmpl, $users->{$user}, $field, \@list);
				} else {
					AddUserValue($tmpl, $users->{$user}, $field, strip($fields[$i]));
				}
			} else {
				AddUserValue($tmpl, $users->{$user}, $field, '');
			}

		}
		# Set type of object ...
		AddUserValue($tmpl, $users->{$user}, 'type', $type);
	}
	close (IN);
}

##########################################################################
# Add a value to a user in the user hash structure
#
# $tmpl		Template for user structure
# $user 	Reference to the user object itself
# $field	Field name
# $value	Value of the field to fill
sub AddUserValue {
	my $tmpl  = shift;
	my $user  = shift;
	my $field = shift;
	my $value = shift;
	my $key;
	my $child;

	foreach $key (keys %{$tmpl}) {

		if (ref($tmpl->{$key}) eq 'HASH') {
			# Ignore field values which are defined as a hash itself ...
			# Example: userc
			if (!defined($user->{$key})) {
				$user->{$key}={};
			}
			AddUserValue($tmpl->{$key}, $user->{$key}, $field, $value);
			
		} elsif ($key eq $field) {
			# Found location of field, now add value to hash structure ...
			if ($tmpl->{$field} =~ /list/) {
				if (ref($value) eq 'ARRAY') {
					push @{$user->{$field}}, @{$value};
				} else {
					$user->{$field}=$value;
				}
			} else {
				$user->{$field}=$value;
			}
		}
	}
};

##########################################################################
# Print all fields of an object
#
# $tmpl		Template for user structure
# $user 	Reference to the user object itself
# $t		Number of tabs in output file before text
sub PrintUser {
	my $tmpl = shift;
	my $user = shift;
	my $t = shift;

	my $field;
	my $member;
	my @list;
	my ($key, $key1, $val, $p);
	
	foreach $field (keys %{$user}) {
		if (ref($user->{$field}) eq 'HASH') {
			if (keys %{$user->{$field}} != 0) {
				print OUT "\t"x$t++.":".$field." (".$LF;
				PrintUser($tmpl->{$field}, $user->{$field}, $t);
				print OUT "\t"x--$t.")".$LF;
			}
			
		} elsif ($tmpl->{$field} =~ /object:(.*)/) {
			if ($user->{$field}) {
				print OUT "\t"x$t++.":".$field." (ReferenceObject".$LF;
				if ($user->{$field} eq 'Any') {
					print OUT "\t"x$t.":Name (Any)".$LF;
					print OUT "\t"x$t.":Table (globals)".$LF;
				} else {
					print OUT "\t"x$t.":Name (".$user->{$field}.")".$LF;
					print OUT "\t"x$t.":Table (".$1.")".$LF;
				}
				print OUT "\t"x--$t.")".$LF;
			} else {
				print OUT "\t"x$t.":".$field." ()".$LF;
			}
			
		} elsif ($tmpl->{$field} =~ /objectlist:(.*)/) {
			if ($user->{$field}) {
				if ($field ne 'ReferenceObject') { print OUT "\t"x$t++.":".$field." (".$LF }
				
				foreach $member (@{$user->{$field}}) {
					if ($member eq 'Any') {
						print OUT "\t"x$t++.": (ReferenceObject".$LF;
						print OUT "\t"x$t.":Name (Any)".$LF;
						print OUT "\t"x$t.":Table (globals)".$LF;
						print OUT "\t"x--$t.")".$LF;
					} else {
						print OUT "\t"x$t++.": (ReferenceObject".$LF;
						print OUT "\t"x$t.":Name (".$member.")".$LF;
						print OUT "\t"x$t.":Table (".$1.")".$LF;
						print OUT "\t"x--$t.")".$LF;
					}
				}
				if ($field ne 'ReferenceObject') { print OUT "\t"x--$t.")".$LF }
				
			} else {
				print OUT "\t"x$t.":".$field." ()".$LF;
			}
		
		} elsif ($tmpl->{$field} eq 'list') {
			if ($user->{$field}) {
				if (ref($user->{$field}) eq 'ARRAY') {
					@list = @{$user->{$field}};
					if (@list) {
						print OUT "\t"x$t++.":".$field." (".$LF;
						foreach $member (@list) {
							print OUT "\t"x$t.": (".$member.")".$LF;
						}
						print OUT "\t"x--$t.")".$LF;
					} else {
						print OUT "\t"x$t.":".$field." ()".$LF;
					}
				} else {
					print OUT "\t"x$t.":".$field." ()".$LF;
				}
			}	

		} elsif (($tmpl->{$field} eq 'string') ||
				 ($tmpl->{$field} eq 'time')   ||
				 ($tmpl->{$field} eq 'date'))  {
			print OUT "\t"x$t.":".$field." (".fixstr($user->{$field}).")".$LF;

		} elsif ($tmpl->{$field} eq 'boolean') {
			if (!$user->{$field}) {	$user->{$field}='false' }
			print OUT "\t"x$t.":".$field." (".lc($user->{$field}).")".$LF;

		} elsif ($tmpl->{$field} =~ /binarylist:(.*)/) {
			@list=split(':', $1);
			
			if ($user->{$field}) {
				$val=0; 
				if (ref($user->{$field}) eq 'ARRAY') {
					foreach $key (@{$user->{$field}}) {
						for ($p=0; $p<@list; $p++) {
							if ($list[$p] eq $key) {
								$val += 2**($p);
							}
						}
					}
				}
				print OUT "\t"x$t.":".$field." (".$val.")".$LF;
			}
		}
	}
}

##########################################################################
# Strip whitespaces of string
sub strip {
	my $string = shift;
	if ($string) {
		$string =~ s/^\s+//;
		$string =~ s/\s+$//;
	}
	return $string;
}

##########################################################################
# Fix a sring for output to file
sub fixstr {
	my $string = shift;
	if ($string =~ /"/) {
		# quoted string ...
	} elsif ($string =~ /[{}:# ]/) {
		$string = '"'.$string.'"';
	}
	return $string;
}

__END__

=head1 Name

B<CPUsers.pl> - Perl program to convert the export file created with 'fw dbexport' to a CheckPoint compatible database file (users.C).

=head1 Description

When creating the program for CPRules, the lack of user data became apparent. The only way to get the users from a CheckPoint Firewall-1 installation is to run a I<'fwm dbexport'> to export the users stored in the I<fwauth.NDB> databasefile. This results in a comma delimited file which is impractical to use with the CPRules program.

In version R55 of the GUI we suddenly discovered a file called B<users.C> which did (also) hold all user data, but in a 'normal' database format. This might well be the way CheckPoint is going, but unfortunately for now we don't have that luxury. As CPRules is written to read the CheckPoint databases we decided to write a script to convert the output of the dbexport to a normal CheckPoint database format.

=head1 Export files

There are two options one can use when exporting the users on a CheckPoint management server:

To export the users: 

  fwm dbexport -f <output file name>
  
To export the usergroups:

  fwm dbexport -g -f <output file name>

Both resulting files should be supplied to this program at the same time and will be converted to one output file. The program will put the users in the right groups for you.

=head1 Commandline options

The commandline options are only used to define all the files used. The official way is to use 'switches' before the filenames to identify the use:

perl CPUsers.pl -u <export file of users> [-g <export file of groups>] [-o <output file name>]

These switches can be used in random order. 

The short version of the commandline is to use no switches, but supply just the filenames. In this case the order is important:

perl CPUsers.pl <export file of users> [<export file of groups>] [<output file name>]

As you might notice in the above examples, the <export file of users> is mandatory, but the <export file of groups> not. Nor is the <output file name>. In case the groupsfile is missing, the software will create the groups from the userfile. In this situation some information will be missing, as there are no comments nor group membership of groups. 

If the outputfile is missing, the default filename B<'users.C'> will be used.

=head1 Version and Bug reports

This script is the first version of CPUsers and hopefully the last.

However bug reports and requests for modifications can be send to Peter-Paul.Worm@wormnet.nl

=head1 Author

Peter-Paul Worm (Peter-Paul.Worm@wormnet.nl)
