#!/usr/bin/perl

#############################################################################
# CPrules.pl: Analyze and print a report of Firewall-1 rules and objects
#
# Note: this script is unsupported by Checkpoint or representatives.
#
# Copyright (C) 2003 Peter-Paul Worm
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#############################################################################
use strict;

my $SCR_NAME 	= 'CPRules';
my $VERSION  	= '1.08';
my $INIFile  	= 'CPrules.ini';
my $PageFile	= 'CPpages.def';

my $lib_dir;
my $working_dir;
BEGIN {
        if ($0 =~ /(.*)\\(\S*)$/) {
                $working_dir=$1;
                $lib_dir=$working_dir.'\Lib'
        } elsif ($0 =~ /(.*)\/(\S*)$/) {
                $working_dir=$1;
                $lib_dir=$working_dir.'/Lib'
        } else {
                $lib_dir='Lib'
        }
}
use lib $lib_dir; 

# Object Export Output specific library ...
use ExportObjects;
use OutputHTML;

# Special purpose packages which store their own data ...
use CheckPoint;
use Options;


##########################################################################
#                            MAIN ROUTINE                                #
##########################################################################
MAIN: {

    my $ReturnCode;	# Temp variable to store returncodes
    
    # Read options from INI - and Page file ...
    &ReadOptions($INIFile, $PageFile, $working_dir);
    
    # Read Objects ...
    print "Reading objects ...\n";
    $ReturnCode = &GetObjects(Option('FW1Objects'));
    if ($ReturnCode==-1) {
    	print "FATAL: Could not open CheckPoint's objects file!!!\n";
    	print "--> ".Option('FW1Objects')."\n";
    	print "\n";
    	print "Unable to continue ... exiting\n";
    	exit;
    }
    
    # Read Rulebases ...
    print "Reading rulebases ...\n";
    $ReturnCode = &GetObjects(Option('FW1Rules'));
    if ($ReturnCode==-1) {
    	print "WARNING: Could not open CheckPoint's rulebases file!!!\n";
    	print "--> ".Option('FW1Rules')."\n";
    }
    
    # Read Desktop Security Rulebases ...
    &GetObjects(Option('FW1SLPRules'));
    
    # Read Users ...
    if (!Option('FW1Users')) {
    	if (Option('EXP_Users')) {
    		# Create users.C from export file(s) ...
    		print "Converting exported user file(s) to database ...\n";
    		system 'perl CPusers.pl '.Option('EXP_Users').' '.Option('EXP_Groups').' users.C';
    		if ($? == 0) {
    			print "Reading users ...\n";
    			&GetObjects('users.C', 'users');
    		} else {
    			print "Errorcode: $?\n";
    			print "WARNING: The users database could not be read, removing users from output ...\n\n";
    		}
    	}
    } else {
    	print "Reading users ...\n";
    	$ReturnCode = &GetObjects(Option('FW1Users'), 'users');
    	if ($ReturnCode==-1) {
    		print "WARNING: Could not open CheckPoint's users file!!!\n";
    		print "--> ".Option('FW1Users')."\n";
    	}
    }
    
    
    ##########################################################################
    #                          OUTPUT SECTION                                #
    ##########################################################################
    if (Option('Format') eq 'HTML') {
    	# Output HTML pages ...
    	print "Generating HTML pages ...\n";
    	&OutputHTMLPages($SCR_NAME, $VERSION);
    } else {
    	# Export to flatfile ...
    	print "Generating Export file(s) ...\n";
    	&ExportObjects($SCR_NAME, $VERSION);
    }

} # End of MAIN:

##########################################################################
#                       END OF MAIN PROGRAM                              #
##########################################################################



__END__

=head1 Name

B<CPRules.pl> - Perl program to convert a Check Point FW-1 configuration to HTML or to export this information to an ASCII file.

=head1 Version

This is version 1.08 of CPRules. This is a special version prior to the NGX release where the 'export' option has been implemented. This feature was requested by several people. It enables the option to export the data and convert it to a different format.

This is the last version that will be released for CheckPoints Next Generation software only. The next step is to review all code for NGX.

=head1 Description

As the management of a FW is a security-sensitive subject, one can not allow others to access the management servers. With 'others' being non-administrators of the firewall, like internal or external customers. In many cases it is even prohibited by the security policy of your company and sometimes the mgmt LAN is even physically disconnected!

However in many cases it is very convenient to have a copy of the configuration available for viewing by a select group of 'others'. This can be for a helpdesk for trouble shooting purposes (not needing to call you every five minutes), a customer demanding to have an insight in their FW configuration or just for backup purposes.

On the other hand, it's not wise to leave this kind of configurations lying around for everybody to see (you're auditors would have a field day). So you might not want everything being published, but only a subset of the configuration (f.i. only the FW rulebase itself). Such a subset would enable the customer (being either internal or external) to request meaningfull changes without having to bother the administrators first.

To be able to do all this, we need a tool to convert the Check Point files to a readable format. This tool would need to be configurable to allow administrators to define what to publish and how. CPRULES is designed to accomodate in just that wish.

There has been repeated requests to be able to export the data to ASCII format, so it could be imported in a database or converted to different output formats. It can also open the way to import the objects (or a selection) into a different program, although other tools are also available for this.

Searching the internet the only tool available to convert Check Point configurations is a perl script called FW1RULES, written and maintained by Volker Tanger <volker.tanger@wyae.de>. Admittedly this program did serve very well and is widely used. This program is originally written to cope with Check Point FW-1 version 4.1 code and has later been adapted to support NG configurations as well. It drastically needed a rewrite of the code due to additions and changes made in the past. That is basically where CPRULES has taken of.

=head1 Configurable Options

There are two ways to configure the program. The first and easiest way is through the configuration file called B<CPRules.ini>. The second option is to use the commandline parameters. The advantage of the .ini file is that many parameters can be set per rulebase. This is not possible through the commandline switches.

The software will read the CPRules.ini file first and adds the commandline switches later on. This enables you to use a default setup and overwrite it at will from the commandline. If you are handling more management stations, this would enable you to call the program for every management server in turn, only differing the location of the Check Point files.

Default behaviour is to process all policy packages found in the objects_5_0.C file. In some cases you just want to get the results for one policy/rulebase or a subset of them. Both is possible through the (include and exclude) commands in the CPRules.ini file. Through the commandline it is only possible to process one (-b switch) or all policies.

=head2 CPRules.ini

As discussed above, the CPRules.ini is the preferred way to configure the behaviour of the program. The CPRules.ini is searched for in the current directory and if not found in the working directory of CPRules, i.e. the folder where the script itself is located. 

The configuration options are divided in several sections: I<CPRules file locations>, I<Check Point file locations>, I<HTML file locations> and I<Policy specific settings>. The next three corresponding paragraphs will outline the commands.

The program has two main functions, creating HTML files or exporting objects to ASCII files. The parameters for both can be included in the same .ini file, although the program will only process one at the time. 

=head3 CPRules file locations

The first section of the CPRules.ini file holds all the path and file names needed by the software itself:

B<TmplPath> - The program uses templates to build the HTML pages. Templating allows you to make changes to the output without messing with the program. For more details about the syntax, see the documentation of the 'I<template.pm>' file. This parameter defines the location of the template files. The program searches for the folder 'Templates' in both the current- and the working directory. 

  TmplPath=./tmpl

B<HTMLPath> - The output of the program are HTML pages per rulebase. This will result in subfolders in the HTML folder per policy (f.i. ./html/MyRulebase). This parameter defines the location of the HTML files.

  HTMLPath=./html

B<ExportPath> - This parameter defines the location of the exported files. As with the HTML files this will result in subfolders per policy (f.i. ./export/MyRulebase).

  ExportPath=./export

=head3 Check Point file locations

The third section of the CPRules.ini file holds the location of the Check Point files needed. How the files get there is completely up to you, as long as they are accessible. B<Note:> Check Point files are case-sensitive, so be careful not to misspell the filenames.

B<FW1Rules> - Check Point stores all the rulebases in one file, called 'rulebases_5_0.fws'. This is the only rulebase file needed. This parameter defines the location and name of this file relative to the work folder.

  FW1Rules=./rulebases_5_0.fws

B<FW1SLPRules> - Check Point stores the desktop security rulebase in a database file called 'slprulebases_5_0.fws' (Secure LAN Policy).  This parameter defines the location and name of this file relative to the work folder.

  FW1SLPRules = ./slprulebases_5_0.fws

B<FW1Objects> - Check Point stores all the objects, services, etc in one database file called 'objects_5_0.C'.  This parameter defines the location and name of this file relative to the work folder.

  FW1Objects=./objects_5_0.C

B<FW1Users> - The users of Check Point are currently not stored in a 'normal' database file. However in the GUI of R55 there are indications this might change. The file used is called 'users.C' and has the same structure as the 'objects_5_0.C' file. This parameter defines the location and name of this file relative to the work folder. If this file is not present yet, one has to rely on the exported users and groups. See below I<EXP_Users> and I<EXP_Groups>.

B<EXP_Users> - Till now the users of Check Point are stored in the file 'fwauth.NDB', which is not readable. The users can be exported from this database by using the command:

  fwm dbexport -f <filename>

This parameter defines the location and name of this exported users file relative to the work folder. The program will use I<CPUsers.pl> to create the 'users.C' file from the export file(s).

B<EXP_Groups> - As with the users, the groups of Check Point are stored in the file 'fwauth.NDB', which is not readable. The groups can be exported from this database by using the command:

  fwm dbexport -g -f <filename>

This parameter defines the location and name of the exported groups file relative to the work folder. The program will use I<CPUsers.pl> to create the 'users.C' file from the export file(s). If the groups file is not included, vital information will be missing, as comments, groups in groups information and administrator groups.

=head3 HTML file locations

The second section of the CPRules.ini file holds all the path and file names needed to view the produced webpages correctly. Note that these paths are hardcoded in the HTML code produced! So you'd better check the files are there when viewing the results of the software.

B<IconPath> - The objects all have different icons per type or class. These icons have been extracted from the Check Point files and stored in a separate folder. This folder is needed for correct displaying of the HTML pages, so the path is relative to the HTML directory. The default value assumes the 'icons'-folder is a subfolder of the HTML folder. This path will be hardcoded in the HTML pages, so be sure the icons are actual there!

  IconPath=../icons

B<CSSPath> - The location of the stylesheets used. There is currently only one stylesheet used, called I<cprules.css>.

  CSSPath=../css

B<ScriptPath> - The HTML output uses three java scripts: I<menu.js>, I<back2top.js> and I<visible.js>. This parameter defines the location of those scripts relative to the HTML files.

  ScriptPath=../scripts

B<MainLogo> - This defines the location of the main logo of the program. This could be the logo of the ISP providing the service, or a self-designed CPRules logo. Note that this will be hardcoded in the HTML files, so be sure it can be found during viewing.

=head3 Policy specific settings

Default behaviour of the program is to process all policies found in the objects_5_0.C file. The I<Include> and I<Exclude> commands can be used to alter this behaviour. You should never have to combine both statements, but you are allowed to.

B<Exclude> - Do NOT generate HTML for this policy. If the first (include/exclude) command found is an Exclude command, all other policies are INcluded automatically.

B<Include> - Do generate HTML for this policy. If the first (include/exclude) command found is an Include command all other policies are EXcluded automatically.

B<Note:> If a policy is both included and excluded (???), the first entry will prevail.

I<Example:>

  Include=MyCompany
  Exclude=Standard

The policy specific section holds additional commands to alter the output of the program. As the program creates the output files per policy, these settings must be adaptable per policy. To accomplish this, section headers with the name of the policy are used to define the setting for this specific policy.

To make life easier, a B<[Default]> section can (and will) be used to hold all 'default' settings. If a specific setting is not specified in a policy section, the corresponding setting in the [Default] section will be used. So in many practical situations, there will be no other policy sections at all!

B<Logo> - The HTML pages always have a logo in the left top of the page (see I<MainLogo>). The Logo parameter is used to define the image in the right top of the pages. This is designed to be the logo of the customer which rulebase is shown here. Note that this will be hardcoded in the HTML files, so be sure it can be found during viewing.

  Logo=./html/MyCompany.jpg

B<ExportDef> - When exporting objects (using the '-e' switch on the commandline), there has to be a definition file which defines what and how to export the objects. As with all parameters this can be defined per policy. This parameter defines the definition file. for more information on the definition file, see the corresponding chapter.

B<AllObjects> - By default (0) the HTML pages will only list those objects actually used in the policy. This parameter (if set to '1') will change this behaviour and will list all objects available. That might come in handy during testing or when documenting the configuration.

  AllObjects=0

B<GlobalProps> - By default (1) the Global Properties will be included in the output files. To suppress the Global Properties, set the value to '0'.

  GlobalProps=1

I<Example:>

  [Default]
  Logo=./html/MyCompany.jpg
  AllObjects=0
  GlobalProps=0
  ExportDef=./CPExport.def
  
  [CompanyB]
  Logo=./html/CompanyB/OtherLogo.gif
  AllObjects=1
  GlobalProps=1

=head2 Commandline switches

All the default values that can be set in the CPRules.ini can also be set using a commandline switch. For a more detailed description see that section. There is always a verbose version and a short version available and the switches are case-insensitive.

=head3 Exporting objects to ASCII

To export objects to ASCII files instead of creating HTML files, always use the 'export' switch:

  --Export
  -e

To define the location of the exported files (default: ./export):

  --ExportPath <path>
  -ep <path>
  
To define the definition file of the exported objects:

  --ExportDef <path and filename>
  -ed <path and filename>

=head3 Creating HTML output

To set the path to the template folder (default: ./tmpl):

  --TmplPath <path>
  -t <path>

To set the path to the HTML folder (default: ./html):

  --HTMLPath <path>
  -h <path>

To set the path to the folder with icons relative to the HTML folder (default: ../icons):

  --IconPath <path>
  -i <path>

To set the path to the folder which holds the css stylesheets (default: ../css):

  --CSSPath <path>
  -c <path>

To set the path to the folder which holds the java-scripts used (default: ../scripts):

  --ScriptPath <path>
  -s <path>

To define the logo in the upper left top of the HTML pages (default: (none)):

  --MainLogo <path and filename>
  -m <path and filename>
  
=head3 CheckPoint file locations

To define the location of the Check Point rulebase file (default: ./rulebases_5_0.fws):

  --FW1Rules <path and filename>
  -r <path and filename>

To define the location of the Check Point Desktop Security rulebase file (default: ./slprulebases_5_0.fws):

  --FW1SLPRules <path and filename>
  -slp <path and filename>

To define the location of the Check Point object database (default: ./objects_5_0.C):

  --FW1Objects <path and filename>
  -o <path and filename>

To define the location of the Check Point users database (if available):

  --FW1Users <path and filename>
  -u <path and filename>

To define the location of the exported users file:

  --EXP_Users <path and filename>
  -eu <path and filename>

To define the location of the exported groups file:

  --EXP_Groups <path and filename>
  -eg <path and filename>

=head3 Policy specific switches

To specify one specific policy only (default: all policies). The '--Rulebae' and '-b' switches are supported for backwards compatibility only:

  --Policy <policy name>
  -p <policy name>
  
  --Rulebase <rulebase name>
  -b <rulebase name>

To define the logo in the upper right top of the HTML pages (default: (none)):

  --Logo <path and filename>
  -l <path and filename>

To define if all objects should be listed by default, the following switch should be used. This switch turns the setting ON (it's OFF by default):

  --AllObjects

To define if the Global Properties should be excluded from the output, the following switch should be used. This switch turns the setting OFF (it's ON by default):

  --NoGlobalProps

I<Examples:>

The typical installation of the CPRules will often be static. The icons, templates and html files are in a fixed location, so only the location of the Check Point files needs to be defined. All other parameters are defined in the .ini file. Such a setup would result in a commandline as shown below (commandline is displayed over more lines for readability):

  cprules --FW1Objects /data/fwmgmt2/objects_5_0.C 
          --FW1Rules /data/fwmgmt2/rulebases_5_0.fws
          --FW1Users /data/fwmgmt2/users.C
          --AllObjects

To run the program for one rulebase only with the corresponding logo:

  cprules --Rulebase MyCompany --Logo ./MyCompany.gif

To export ALL objects as defined in the export definition file specified:

  cprules --Export --ExportDef ./defs/MyExport.def --AllObjects

=head2 CPExport.def

The objects in the CheckPoint database can be exported to ASCII using the '-e' switch. This mode requires the definition of how to export which objects. The default file used for this is I<CPExport.def>. A different file (per policy) can be used by using the I<ExportDef> parameter in the CPrules.ini file or the I<--ExportDef> commandline switch.

The export-definition file is B<NOT> a template file. This might become an addition in the future, but till now the objecs can only be exported as a (comma/tab/...-) delimited file. This will be sufficient for exporting objects to another format, but creating advanced reports is not possible yet.

The I<CPExport.def> consists of several sections, each defining a different table to be exported and how this should be exported. The possible parameters are described below.

B<Note:> The files defined are overwritten and not appended, so make sure a different outputfile name is defined per section!

B<table> - The I<table> parameter defines the CheckPoint table to be used. Most tables come from objects_5_0.C but they are not restricted to that file (the rule-bases are typically extracted from the rulebases_5_0.fws file).

I<Example:>

  table=network_objects

B<name> - The I<name> parameter defines the name of the objects to be exported. This parameter is mandatory when exporting B<rulebases> to distinguish between the different type of rulebases. for the normal security rulebase this would be 'rule', for a NAT rulebase it would be 'rule_adtr'. The latter does export the MANUAL NAT rules only, not the automatic NAT rules as these are properties of the objects themselves!

I<Example:>

  name=rule

When exporting B<objects>, the parameter holds a wildcard for the name of the objects. So as with the class parameter, the value specified is treated as a regular expression. This parameter can be repeated several times to include more names.

I<Example:>

  name=^CUS_
  name=FW

The first name statement will include all objects starting with 'CUS_'. The second statement will include all objects which holds the string 'FW' in the name somewhere.

B<class> - The I<class> parameter is probably the hardest one of all parameters. Every object does have a class defined (:AdminInfo:ClassName), if it is a 'normal' object or a rule in a rulebase. The I<class> parameter defines which type of objects will be included in the export file. Several classes can be combined in one table by repeating the I<class> parameter. To make it easier, this parameter is interpreted as a regular expression. That means the program searches for a match in the actual classname of the object. if no class is specified, all objects will be included.

I<Example:>

  class=udp_service
  class=^tcp

The first class statement includes all objects which have 'udp_service' in the classname. The second line includes all classnames that starts with 'tcp'.

B<headers> - The I<headers> parameter is optional and defines the column headers of the objects exported. The list is (white)space delimited, so unfortunately the headers themselves can not include spaces. When the I<class> parameter is ommited, the fieldnames will be used as the header text.

I<Example:>

  headers=Name IP_address SubnetMask NAT_address GroupMembers Comments

B<fields> - The I<fields> parameter defines the fields of the objects to export. The list is (white)space delimited. The names of the fields can be found in the CheckPoint database files, see a few examples below. There are however a few reserved words: 

=over 4

=item * members

This defines the members of an object (group). Members of an object should be taken loosely, but the most common situation is to use this option to get all the members of a group. One could argue that the CheckPoint name for this is "ReferenceObject" (which can also be used), but I<members> is for convenience of the user.

=item * name

When exporting a table of objects this is simply the name of the object.
When exporting a rulebase this will return the name of the rule. This is an option that can only be used with NGX and later

=item * number

When exporting a table of objects this will simply count the number of objects exported.
When exporting a rulebase this will return the rulenumber. In this case header rules will be skipped (no number is exported).

=back

I<Example:>

  fields=name ipaddr netmask valid_ipaddr type members comments

Note: In case a field holds more objects (f.i. members of a group), all objects will be listed separated by the I<list-separator> parameter.

Note: if a field can not be found as attribute of an object, it is left empty.

B<separator> - The I<separator> is to define which separator is to be used in the exported file. Any character(s) can be used, but there are three reserved words: COMMA, TAB and SPACE. These will be translated to their respective character codes. Default value is a COMMA (',').

I<Example:>

  separator=TAB

B<list-separator> - The I<list-separator> parameter is used to defined which character is used to separate several objects within one field (see fields description). As with the I<separator> parameter three reserved words can be used: COMMA, TAB and SPACE. Default value is a colon (':').

I<Example:>

  list-separator=:

B<outputfile> - The I<outputfile> parameter defines the name of the output file and the location of this file. if no path is specified the normal exportpath will be appended (by default './export/<Policy name>/'). if no outputfile is specified the <name of the section>.exp is used instead.

I<Example:> 

  outputfile=tcp_resources.csv


Finally an I<example> of an entire table definition is presented. This example exports all objects in the database, comma delimited:

  [All network_objects]
  table=network_objects
  class=
  headers=Class Type Location IP Subnetmask
  fields=AdminInfo:ClassName type location ipaddr netmask
  separator=,
  outputfile=all_network_objects.csv


=head1 Advanced configuration options

This chapter describes how the output of CPRules is setup and how it can be altered. This section is for advanced users. Normal operation of the software would not need any alterations to these settings. 

The default setup creates all HTML pages with all (used) object types converted to HTML. This might well not be what you want. The page, table and section definitions enables you to change the layout to your taste. 

B<Tip:> Make a backup of the original files before changing the files described here.

B<Tip:> Run a report and check it out before you start modifying the layout definitions. 

=head2 Page layout

Which tables are presented on which HTML pages and which section is presented in which table, is defined in the file called: B<CPPages.def>. This configuration file enables you to determine what to present to your customers. Although it is possible to NOT show an entire table or section, it is not possible to alter the content of the specific section or table through this file.

In CPPages.def the page the entire setup can be defined per rulebase by creating a separate section per rulebase. However there is also a section called B<[Default]> which will be used if there is no section defined for a rulebase. This enables you to specify a different layout for every rulebase!

The layout of a rulebase consists of pages, tables and sections. Of these three only the page names are free to choose! The tables are defined in the I<Constants.pm> file (described in the next paragraph), but the sectionnames are even hardcoded in the source files. Trying to change them might have interesting results :-).

Every page definition starts with a B<Page=> command. Every table on this page is listed beneath the page definitione and denoted by one or more B<Table=> commands. Every table holds one or more B<[Section name]> definitions. For the pages and tables this is straightforward. To show the structure an example without sections is shown below.

I<Example:>

  [MyRulebase]
  Page = Rulebase
  Table = Rules
  Table = NATRules

  Page=MyPage
  Table=Hosts
  Table=Services

  Page=My2ndPage
  Table=Resources
  Table=OPSEC

Now many tables consists of several sections. Every section can be SKIPped (not displayed), OPEN or CLOSED. The latter two means that the section is folded open or closed. Clicking on the section name will revert this. By default all sections are CLOSED, but this can be altered. First impressions show that larger databases display much better with all sections CLOSED, but smaller databases might well turn out better with OPENed sections.

The below (partial) example shows one page with three tables in which the Topology section is OPENed, but the other sections are CLOSED.

I<Example:>

  [MyRulebase]
  Page = Gateways
  
  Table = CheckPoint
  Check Point Products = CLOSED
  Cluster Members      = CLOSED
  Sofaware Settings    = CLOSED
  Topology             = OPEN
  ...
  
  Table = Gateways
  Topology      = CLOSED
  VPN           = CLOSED
  Web Server    = CLOSED
  FireWall-1 GX = CLOSED
  Advanced      = CLOSED
  
  Table = OSEDevices
  Topology      = CLOSED
  SNMP	        = CLOSED
  Setup         = CLOSED


=head2 Table definitions

I<B<Warning:> Changing table definitions is not straight-forward. You need to have thorough understanding of the Check Point database structure! However it provides an manageable way (for the author) to cope with future changes/additions Check Point comes up with.>

The table definitions are in a package called B<Constants.pm>. This Perl package holds all Check Point related constants used in CPRules.

The hash used for the table definitions is %HTMLTable and it is a bit complex. The purpose is to define which objects are to be displayed in which table AND to define the order in which they are presented within the table. A restriction is that all objects within one HTML table must be member of the same Check Point table. This table is reflected in the value of B<TABLE>. I<At this time it is not possible to combine objects from different Check Point tables into one HTML table.> 

All Check Point objects do have a 'class' (in the database this is defined in 'AdminInfo:ClassName'). Objects of the same class B<have> to be grouped together in one table although they can have different types. It is fine to combine different classes within one table. The best table to show this grouping is the I<Services> table:

  SERVICES => {
  TABLE => 'services',
    CLASS => {
      'tcp_service'          => '1:TCP',
      'compound_tcp_service' => '2:Compound TCP',
      'tcp_citrix_service'   => '3:Citrix TCP',
      'udp_service'          => '4:UDP',
      'rpc_service'          => '5:RPC',
      'icmp_service'         => '6:ICMP',
      'other_service'        => '7:Other',
      'dcerpc_service'       => '8:DCE-RPC',
      'gtp_service'          => '9:GTP',
      'gtp_v1_service'       => '9:GTP'
    }
  },

The SERVICES table above is getting the objects from the Check Point table 'services'. The numbers followed by a colon (':')in the text define the order in which they will be presented in the HTML page. For instance the services with type 'tcp' are presented before the services with type 'udp'. The remainder of the text is passed to the routine that creates the HTML pages and is currently used as header text for the service subsections (see the examples).

If the CLASS of an object is not sufficient to determine the different members, it is possible to define other keys to select on. An example to explain this is seen in the gateway table definition:

  GATEWAYS =>	{
    TABLE => 'network_objects',
    CLASS => {
      'gateway_plain' => {
      	'VPN'                     => { ''    => '1:Gateway Node' },
      	'VPN:AdminInfo:ClassName' => { 'vpn' => '2:Interoperable Device' }
      }
    }
  },

The class 'gateway_plain' holds two different types of gateways. To distinguish between them we need to look at the 'VPN-ClassName' of the objects. See above how that is defined in the hash %Table. Note the 'VPN:AdminInfo:ClassName' to denote the path to the key used.

There is one final possibility. The rules in a rulebase are no separate objects, but are part of a list of rules. This case is reflected in the %Table by not defining a TYPE, but by defining a LIST:

  RULES => {
    TABLE => 'rule-base',
    LIST  => 'rule'
  },

For more specific details check the code, to see what is passed exactly to the corresponding routines and how that can be handled.

=head1 Templating

Templating is used to differentiate between the codewriter and the layout people and have the best of both worlds. Implementing this is easier said than done. In this case we needed to have enough options for the codewriter to determine the layout of the tables and sections in the tables, but without needing to write actual HTML code. To fix this dilemma, we have included a template file called I<section.tmpl> which defines the different formats information can be displayed (see below). This file is used in several other files as INCLUDE file. Now the codewriter has/can have a number of pre-defined layouts and the layout people can define the actual HTML code and influence the actual result.

This is B<not> a codewriters guidebook. For examples of the described options one better examine the source code and the template files. The templating engine is discussed in a separate document (I<template.pm>) and that is the place to learn about data structures used.

=head2 Layout commands

The I<section.tmpl> file is used to add a section to a table. The include statement will normally be included in a loop, to allow for more than one section to be added (e.g. all nodes). 

A section is always delimited by a BEGin and END statement. Not closing of a section might result in "interesting" layouts :-). The following section delimiters are currently defined:

  SF_BEG ... SF_END
  SUBSECT_BEG ... SUBSECT_END
  COL_BEG ... COL_NXT ... COL_END

=head3 S0 - Section Header

The section S0 generates a text with a horizontal line behind it. This can be used to indicate a new 'chapter' or to clearly distinguish between two sections. This feature is being superseded by the SF_BEG and SF_END structure, but is and will be supported for backwards compatibility (it is still begin used in the Global Properties, but this will be changed).

=head3 SF_BEG ... SF_END

The SF stands for 'B<S>ection B<F>olded'. It defines a section that can be folded closed or opened (see the Page Layout section). Clicking on the header of the section closes or opens the section. 

=head3 SUBSECT_BEG ... SUBSECT_END

This option generates a (sub)section in the table by using the html FIELDSET commands. This results in the browser to draw a box around the included text, indicating that this information is related. 

Previously there was also a SECTION command that could be used if the content was all in one section. This still works, but is only supported for backwards compatibility. It is not used anymore in the source code of CPrules.

=head3 COL_BEG ... COL_NXT ... COL_END

To place two or more sections next to each other, columns can be used. The COL_BEG starts the columns, the COL_NXT initiates a new column and the COL_END closes the columns off.

=head2 Formatting commands

The layout of a section is based on a few input parameters supplied in the source code. The first option defines the general format and the suboptions define the specific field layouts.

=head3 TXT - Text Lines

In some case you just need to include a simple text message or note in a section. This option will just do that for you. The format of the text is determined by the CLASS suboption, which relates directly to classes defined in the I<cprules.css> file. Therefore, this should be meaningfull names. Currently we have defined two classes:

=over 1

=item *
note - defines a normal note or instruction in the section text

=item *
note_bold - defines an important note in the text; designed to draw the attention of the reader.

=item *
note_xspc - used for a single line of a note, but now has extra spacing above and beneath the text

=item *
note_xspc_bold - used for a single line of important note, with extra spacing above and beneaht the text. Designed to draw the attention of the reader.

=back

=head3 S1 - Section

This section is probably the most important section. Every line in this section consists of a maximum of five (5) fields or columns. As this section is actually a table, the fields are nicely lined up beneath each other.

The standard options in the S1 section:

=over 1

=item *
C1 .. C5 - Defines the number of actual columns used for this field. Note if this option is used the total number of fields is reduced. However it gives the option to use two or more columns for one field. 

=item *
CP1 .. CP5 - Defines which field is to be depicted as a (Check Point) value, in which case the class 'cpvalue' is used.

=item *
OBJ_VAL1 .. OBJ_VAL5 - Defines the actual text/string to be displayed. The field value.

=back

Advanced option in the S1 section:

=over 1

=item *
INACTIVE - Defines the current line NOT to be displayed. This might sound like a weird option, but it enables the codewrite to make decisions to show certain lines without using endless if - endif constructions.

=back

=head3 T1 - Table with headers

The T1 section generates an actual table with lines and a header line. 

=over 1

=item *
C1 .. C5 - Defines the number of columns used for this table. Set one of the values to '1' to define the number of columns!

=item *
HEAD - Defines the current line to be a header line. Does not neccessarily need to be the first line defined, but it will appear at the top of the table.

=item *
OBJ_VAL1 .. OBJ_VAL5 - Defines the actual text/strings to be displayed in the table. The field values.

=back

=head3 OBJLIST - List of Check Point objects

This is a special section, as it uses the same syntax as the routine I<HTMLMemberList>. This routine is used to get the HTML code for all the members of an object (f.i. the Participating Gateways in a VPN community). Nothing more nothing less.

=head3 VOID - Empty field

This special section is used to include a '&nbsp;' in a tablecell. In case a tablecell is empty, the browser will not display the borders either. This section just add a space to the cell, so the browser won't see it as an empty cell.

=head1 Bug reports

The version history is supplied in a different history file together with all bugfixes. This file can be viewed on the webpage "http://www.wormnet.nl/cprules/history/History of CPrules.doc"

Bug reports and requests for modifications can be send to Peter-Paul.Worm@wormnet.nl

=head1 Author

Peter-Paul Worm (Peter-Paul.Worm@wormnet.nl)

