package tables2html;

use strict;
use CheckPoint;
use Constants;
use Options;
use object2html;
use Strings;

require Exporter;
our (@ISA, @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(table2html);

# Initialize special marker signs ...
my $Check      = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/check.png>';
my $CheckTick  = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/checktick.png>';
my $Mark       = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/mark.png>';
my $MarkDot    = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/markdot.png>';
my $Spacer     = '<IMG STYLE="vertical-align: middle;" SRC='.Option('IconPath').'/spacer.png>';
my $None       = '<I>(none)</I>';

#*************************************************************************
#***                        Tables to HTML                             ***
#*************************************************************************
# Routine: table2html
#
# Description:
# Overall routine to generate the different tables to HTML code
#
# Parameters:
# $rulebase	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
# $table	Name of table to be generated into template
#
# Returns:
# NrObjects The number of objects in the table
#
# $template	Filled template with the requested sections

sub table2html {
	use strict;
	my (@ObjList, $rb_name, $AllObjects);
	my ($obj, $table, $name, $type);
	my ($CPtable, $CPtype);
	my ($key, $var);
	my $sub2html;
	
	my $rulebase = shift;
	my $template = shift;
	my $table    = shift;
	my $TABLE    = uc($table);

	# Determine rulebase name ...
	$rb_name = GetKey($rulebase, 'collection:Name');

	# Determine if all objects need to be listed ... 
	$AllObjects = Option($rb_name, 'ALLOBJECTS');

	# Determine Check Point table name to be searched ...	
	$CPtable = $Table{$TABLE}{TABLE};

	if ($Table{uc($table)}{LIST}) {
		# Create list of TYPE object ...
		$type = $Table{$TABLE}{LIST};
		@ObjList = GetMembers(GetKey($rulebase, $type));
	} else {
		# Create list of objects ...
		if (ref(GetKey($CPtable))) {
			# table is not empty ...
			foreach $name (keys %{GetKey($CPtable)}) {
				if ($AllObjects || (GetKey($CPtable, $name, 'is_used', $rb_name) eq 'true')) {
					$CPtype = lc(GetKey($CPtable, $name, 'type'));
					$type   = $Table{$TABLE}{TYPE}{$CPtype};
					if (ref($type)) {
						foreach $key (keys %{$type}) {
							$var  = lc(stripquotes(GetKey($CPtable, $name, $key)));
							$type = $Table{$TABLE}{TYPE}{$CPtype}{$key}{$var};
						}
					}
					if ($type) { push @ObjList, $type.':'.$name }
				}
			}
		}
	}

	# Create table if there are objects to be displayed ;
	# The exception is the NAT rulebase, as there does not
	# have to be actual NAT rules. They can be automatic.
	if ((@ObjList)||($TABLE eq 'NATRULES')) {
		# Define global variables in template file ...		
		$template->param(	'SECTION_'.$table	=> 1,
							$table.'_REF'		=> $table
						);
		
		# Call the corresponding subroutine ...	
		no strict 'refs';
		$sub2html = lc($table).'2html';
		if (defined(&$sub2html)) { &$sub2html($rulebase, $template, $CPtable, \@ObjList) }
	}

	# Return the number of objects in the list
	return @ObjList;
}


##########################################################################
# Routine: rules2html
#
# Description:
# Routine to create the normal rules section
##########################################################################
sub rules2html {
	use strict;
	my (@rules, $rule);
	my ($rnr, $rulenr);
	my (@trules, @install);
	my ($rb_name, $rb_simplified);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	@rules       = @{$obj};

	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	$rb_simplified = (GetKey($rulebase, "use_VPN_communities") eq 'true');
	
	# RULEBASE properties to HTML ...
	$template->param( RUL_SIMPLIFIED	=> $rb_simplified );
	
	#----------------------------------------------------------------
	#---                      RULES to HTML                       ---
	#----------------------------------------------------------------
	$rulenr=0;
	$rnr=-1;
	foreach $rule (@rules) {
		if (GetKey($rule, 'AdminInfo:ClassName') eq 'security_header_rule') {
			$rnr += 1;
			# Comment line in rulebase
			$trules[$rnr]{RUL_HEADER}	= 1;
			$trules[$rnr]{RUL_HDR_TEXT}	= stripquotes(GetKey($rule, 'header_text'));

		} elsif (GetKey($rule, "AdminInfo:ClassName" eq 'security_rule')) {
			# Actual rule
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_SIMPLIFIED} = $rb_simplified;
			$trules[$rnr]{RUL_HEADER}	= 0;
			$trules[$rnr]{RUL_NO}		= $rulenr;
			$trules[$rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');

			$trules[$rnr]{RUL_SOURCE}	= HTMLMemberList($rb_name, GetKey($rule, 'src'), $_HREF);
			$trules[$rnr]{RUL_DEST}		= HTMLMemberList($rb_name, GetKey($rule, 'dst'), $_HREF);
			$trules[$rnr]{RUL_VPN}    	= HTMLMemberList($rb_name, GetKey($rule, 'through'), $_HREF);
			$trules[$rnr]{RUL_SERVICE}	= HTMLMemberList($rb_name, GetKey($rule, 'services'), $_HREF);
			$trules[$rnr]{RUL_ACTION}	= HTMLMemberList($rb_name, GetKey($rule, 'action'), $_HREF);
			$trules[$rnr]{RUL_TRACK}	= HTMLMemberList($rb_name, GetKey($rule, 'track'), $_HREF);;
			$trules[$rnr]{RUL_WHEN}		= HTMLMemberList($rb_name, GetKey($rule, 'time'), $_HREF);;
			@install=GetMembers(GetKey($rule, 'install'));
			if (GetKey($install[0], 'Name') eq 'Any') {
				$trules[$rnr]{RUL_INSTALL}	= [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{RUL_INSTALL}	= HTMLMemberList($rb_name, GetKey($rule, 'install'), $_HREF);
			}

			$trules[$rnr]{RUL_COMMENTS}	= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
		}
	}
	$template->param(RULES=>\@trules);
	
	return $template;
}


##########################################################################
# Routine: natrules2html
#
# Description:
# Routine to create the NAT rules section
##########################################################################
sub natrules2html {
	use strict;
	my (@rules, $rule);
	my ($rnr, $rulenr);
	my (%fwtargets, $fw);
	my ($rb_name, $rb_simplified, $AllObjects);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	@rules       = @{$obj};

	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");

	# Determine if all objects need to be listed ... 
	$AllObjects = Option($rb_name, 'ALLOBJECTS');

	# Determine installation targets ...
	if (lc(GetKey("policies_collections:".$rb_name.":all_internal_modules")) eq 'true') {
		$fwtargets{All}=1;
	} else {
		$fwtargets{All}=0;
		foreach $fw (GetMembers(GetKey("policies_collections:".$rb_name.":installable_targets"))) { 
			$fwtargets{GetKey($fw, "Name")}=1
		}
	}

	#----------------------------------------------------------------
	#---                    NATRULES to HTML                      ---
	#----------------------------------------------------------------
	$rulenr=0;

	# Insert NAT rules BEFORE
	my (@trules, @install);

	$rnr=-1;
	foreach $rule (@rules) {
		if (GetKey($rule, 'rule_block_number') eq '0') {
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_NO}		= $rulenr;
			$trules[$rnr]{RUL_DISABLED}	= (GetKey($rule, 'disabled') eq 'true');
	
			$trules[$rnr]{ORG_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr'), $_HREF);
			$trules[$rnr]{ORG_DESTINATION}	= HTMLMemberList($rb_name, GetKey($rule, 'dst_adtr'), $_HREF);
			$trules[$rnr]{ORG_SERVICE}		= HTMLMemberList($rb_name, GetKey($rule, 'services_adtr'), $_HREF);
			if (GetKey($rule.':services_adtr', 'adtr_method') eq 'adtr_method_static') {
				$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_STATIC);
			} else {
				$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_HIDE);
			}
			$trules[$rnr]{TR_DESTINATION}	= HTMLMemberList($rb_name, GetKey($rule, 'dst_adtr_translated'), $_HREF+$_NAT_STATIC);
			$trules[$rnr]{TR_SERVICE}		= HTMLMemberList($rb_name, GetKey($rule, 'services_adtr_translated'), $_HREF+$_NAT_STATIC);
			@install=GetMembers(GetKey($rule, 'install'));
			if (GetKey($install[0], 'Name') eq 'Any') {
				$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{INSTALL}	    = HTMLMemberList($rb_name, GetKey($rule, 'install'), $_HREF);
			}
			$trules[$rnr]{RUL_COMMENTS}		= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
		}
	}
	$template->param(NAT_RULES_FIRST=>\@trules);
		
	# Insert automatic NAT rules ...
	# Auto NAT rule order:
	# 1. first hosts(H) than networks(N)
	# 2. first static(S) than hide(H)
	# 3. alphabetical

	# Initialize tables for this section ...
	my @trules; 
	my (@HSlist, @HHlist, @NSlist, @NHlist);
	my ($obj, $table, $name);

	# Dynamic NAT rules can be stored in ALL network objects ...
	$table='network_objects';

	if (ref(GetKey($table))) {
		foreach $name (keys %{GetKey($table)}) {
			$obj=GetKey($table.':'.$name);
			if (GetKey($obj, 'add_adtr_rule') eq 'true') {
				if (	$AllObjects || (GetKey($obj, 'the_firewalling_obj:Name') eq 'All') ||
						$fwtargets{All} || $fwtargets{GetKey($obj, 'the_firewalling_obj:Name')} ) {
					if (GetKey($obj, 'netmask') ne '') {  # Network object
						if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_hide') {
							push @NHlist, $name;
						} else {
							push @NSlist, $name;
						}
					} else {                # Host object
						if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_hide') {
							push @HHlist, $name;
						} else {
							push @HSlist, $name;
						}
					}
				}
			}
		}
	}
	
	# Build automatic NAT rules
	$rnr=-1;
	foreach $name (sort(@HSlist), sort(@HHlist), sort(@NSlist), sort(@NHlist)) {
		# Host NAT ...
		$obj=GetKey($table, $name);

		if (GetKey($obj, 'netmask')) {
			# Network to Network: no NAT
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_NO}		= $rulenr;

			$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($rb_name, $table, $name, $_HREF);
			$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($rb_name, $table, $name, $_HREF);
			$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($rb_name, 'globals', 'Any');
			$trules[$rnr]{TR_SOURCE}		= [txt2html('Original', 'Original', 'black')];
			$trules[$rnr]{TR_DESTINATION}	= [txt2html('Original', 'Original', 'black')];
			$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
			if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
				$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{INSTALL}	    = HTMLObjectList($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
			}
			$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);
		}
		
		# From inside to outside NAT ...
		$rnr += 1;
		$rulenr += 1;
		$trules[$rnr]{RUL_NO} = $rulenr;

		$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($rb_name, $table, $name, $_HREF);
		$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($rb_name, 'globals', 'Any');
		$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($rb_name, 'globals', 'Any');
		$trules[$rnr]{TR_SOURCE}		= HTMLObjectList($rb_name, $table, $name, ($_NAT+$_HREF));
		$trules[$rnr]{TR_DESTINATION}	= [txt2html('Original', 'Original', 'black')];
		$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
		if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
			$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
		} else {
			$trules[$rnr]{INSTALL}	    = HTMLObjectList($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
		}
		$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);

		if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
			# Static NAT only: From outside to inside NAT ...
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_NO}		= $rulenr;

			$trules[$rnr]{ORG_SOURCE}		= HTMLObjectList($rb_name, 'globals', 'Any');
			$trules[$rnr]{ORG_DESTINATION}	= HTMLObjectList($rb_name, $table, $name, ($_NAT+$_HREF));
			$trules[$rnr]{ORG_SERVICE}		= HTMLObjectList($rb_name, 'globals', 'Any');
			$trules[$rnr]{TR_SOURCE}		= [txt2html('Original', 'Original', 'black')];
			$trules[$rnr]{TR_DESTINATION}	= HTMLObjectList($rb_name, $table, $name, $_HREF);
			$trules[$rnr]{TR_SERVICE}		= [txt2html('Original', 'Original', 'black')];
			if (GetKey($obj, 'the_firewalling_obj:Name') eq 'Any') {
				$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{INSTALL}	    = HTMLObjectList($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF);
			}
			$trules[$rnr]{RUL_COMMENTS}	    = text2html('Automatic NAT rule', $_MAX_COMMENT_LENGTH);
		}
	}

	$template->param(NAT_AUTO_RULES=>\@trules);

	# Insert NAT rules LAST

	# Initialize tables for this section ...
	my (@trules); 

	$rnr=-1;
	foreach $rule (@rules) {
		if (GetKey($rule, 'rule_block_number') eq '1') {
			$rnr += 1;
			$rulenr += 1;
			$trules[$rnr]{RUL_NO}		= $rulenr;
	
			$trules[$rnr]{ORG_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr'), $_HREF);
			$trules[$rnr]{ORG_DESTINATION}	= HTMLMemberList($rb_name, GetKey($rule, 'dst_adtr'), $_HREF);
			$trules[$rnr]{ORG_SERVICE}		= HTMLMemberList($rb_name, GetKey($rule, 'services_adtr'), $_HREF);
			if (GetKey($rule.':services_adtr', 'adtr_method') eq 'adtr_method_static') {
				$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_STATIC);
			} else {
				$trules[$rnr]{TR_SOURCE}		= HTMLMemberList($rb_name, GetKey($rule, 'src_adtr_translated'), $_HREF+$_NAT_HIDE);
			}
			$trules[$rnr]{TR_DESTINATION}	= HTMLMemberList($rb_name, GetKey($rule, 'dst_adtr_translated'), $_HREF+$_NAT_STATIC);
			$trules[$rnr]{TR_SERVICE}		= HTMLMemberList($rb_name, GetKey($rule, 'services_adtr_translated'), $_HREF+$_NAT_STATIC);
			@install=GetMembers(GetKey($rule, 'install'));
			if (GetKey($install[0], 'Name') eq 'Any') {
				$trules[$rnr]{INSTALL}      = [txt2html('Policy Targets', 'any_object', 'black')]
			} else {
				$trules[$rnr]{INSTALL}	    = HTMLMemberList($rb_name, GetKey($rule, 'install'), $_HREF);
			}
			$trules[$rnr]{RUL_COMMENTS}		= text2html(stripquotes(GetKey($rule, 'comments')), $_MAX_COMMENT_LENGTH);
		}
	}
	$template->param(NAT_RULES_LAST=>\@trules);
	
}

##########################################################################
# Routine: checkpoint2html
#
# Description:
# Routine to create the CHECKPOINT table
##########################################################################
sub checkpoint2html {
	use strict;
	my ($obj, $rb_name, $name, $type, $current_type);
	my ($cnt, @tobj, @tint, $intcnt, $iobj);
	my ($var, %var, @var, $val, $AItype);
	
	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};

	# Determine rulebase name ...
	my $rb_name = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                  CHECKPOINT to HTML                      ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}
		
		$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
		$tobj[$cnt]{OBJ_TYPE} = $type;
		$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($rb_name, $obj, $name);
		
		$tobj[$cnt]{OBJ_IPADDRESS} = GetKey($obj, 'ipaddr');
		if (GetKey($obj, 'ipaddr') eq '0.0.0.1') {
			$tobj[$cnt]{OBJ_IPADDRESS} = 'Dynamic Address'
		}
		
		#------------------------------------------------
		# GENERAL tab/column of Checkpoint devices
		#------------------------------------------------
		# Checkpoint version and option level ...
		my @section;
		if (GetKey($obj, 'cp_products_installed') eq 'true') {
			$var = '';
			$AItype = 0;
			if (GetKey($obj, 'cpver') eq '5.0') {
				$var .= 'NG';
				$val = GetKey($obj, 'option_pack');
				if ($val ge '4') { 
					$var .= ' with Application Intelligence';
					$AItype = 1;
				} elsif ($val ne '0') {
					$var .= ' Feature Pack '.$val
				}
			} else {
				$var .= GetKey($obj, 'fwver');
			}
			push @section, { R2 => 1, OBJ_VAR => html('Version:'), OBJ_SET => html($var) };
			
			if ($AItype) {
				$var = (GetKey($obj, 'cp_suite_type') eq 'pro')?'Check Point Enterprise/Pro':'Check Point Express';
				push @section, { R2 => 1, OBJ_VAR => html('Type:'), OBJ_SET => html($var) };
			}
			
			# Check Point products installed
			push @section, { H1 => 1, OBJ_VAR => html('Products') };
			@var = ();
			if (GetKey($obj, 'firewall') eq 'installed') 		{ push @var, 'FireWall-1' }
			if (GetKey($obj, 'VPN_1')    eq 'true')				{ push @var, 'VPN-1 Pro' }
			if (GetKey($obj, 'vpnddcate') eq 'true')			{ push @var, 'VPN-1 Net' }
			if (GetKey($obj, 'floodgate') eq 'installed')		{ push @var, 'Floodgate-1' }
			if (GetKey($obj, 'policy_server') eq 'installed')	{ push @var, 'SecureClient Policy Server' }
			if (GetKey($obj, 'SDS') eq 'installed')				{ push @var, 'SecureClient Software Distribution Server' }
			if (GetKey($obj, 'management') eq 'true')			{ push @var, 'Secondary Management Station' }
			if (GetKey($obj, 'cpver') gt '4.1')					{ push @var, 'SVN Foundation' }
			if (GetKey($obj, 'log_server') eq 'true')			{ push @var, 'Log Server' }
			if (GetKey($obj, 'reporting_server') eq 'true')		{ push @var, 'SmartView Reporter' }
			if (GetKey($obj, 'real_time_monitor') eq 'true')	{ push @var, 'SmartView Monitor' }
			if (GetKey($obj, 'UA_Server') eq 'true')			{ push @var, 'UserAuthority Server' }
			if (GetKey($obj, 'UA_WebAccess') eq 'true')			{ push @var, 'UserAuthority WebAccess' }
			if (GetKey($obj, 'Everest') eq 'true')				{ push @var, 'Firewall-1 GX' }
			if (@var) {
				foreach $var (@var) {
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}

			push @{$tobj[$cnt]{OBJ_GENERAL}}, { S0 => html('Check Point Products'), S1 => \@section };
		}
		# Cluster members ...
		if ($type eq 'Gateway Cluster') {
			my @section;
			# Find Members ...
			@var = GetMembers(GetKey($obj, 'cluster_members'));
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{OBJ_GENERAL}}, { S0 => html('Cluster Members'), S1 => \@section };
		}
		if (GetKey($obj, 'type') eq 'sofaware_gateway') {
			my @section;
			if (GetKey($obj, 'sofaware_vpn') eq 'true') {
				$var = (GetKey($obj, 'sofaware_mode') eq 'SiteToSite')?'Site To Site':'Remote Access';
				push @section, { R2 => 1, OBJ_VAR => $CheckTick.html('VPN Enabled:'), OBJ_SET => html($var) };
			}
			
			# It is a reference to an object, but not displayed as such in the SmartDashboard ... ?
			$var = GetKey($obj, 'product_type:Name');
			push @section, { R2 => 1, OBJ_VAR => html('Type:'), OBJ_SET => html($var) };

			%var = %{obj2html($rb_name, GetKey($obj, 'profile:Table'), GetKey($obj, 'profile:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('VPN-1 Edge Profile:'), OBJ_SET => $var };

			if (GetKey($obj, 'product_key')) {
				push @section, { R2 => 1, OBJ_VAR => html('Product Key:'), OBJ_SET => GetKey($obj, 'product_key') };
			}
			if (GetKey($obj, 'MAC_address')) {
				push @section, { R2 => 1, OBJ_VAR => html('MAC Address:'), OBJ_SET => GetKey($obj, 'MAC_address') };
			}
			if (GetKey($obj, 'external') eq 'true') {
				push @section, { R3 => 1, OBJ_VAR => html('Externally Managed Gateway'), OBJ_SET => $CheckTick };
			}

			push @{$tobj[$cnt]{OBJ_GENERAL}}, { S1 => \@section };
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_GENERAL}}) { push @{$tobj[$cnt]{OBJ_GENERAL}}, { VOID => 1 } }

		#------------------------------------------------
		# TOPOLOGY tab/column of Checkpoint devices
		#------------------------------------------------
		# Interfaces ...
		if ($type ne 'Gateway Profile') {
			my @section;
			@var = GetMembers(GetKey($obj, 'interfaces'));
			if (@var) {
				push @section, { R0 => 1,	OBJ_VAL1 => html('Name'), 
											OBJ_VAL2 => html('IP address'), 
											OBJ_VAL3 => html('Allowed addresses')
										};
				my %list;
				foreach $var (@var) {
					$list{GetKey($var, 'officialname')}=$var;
				}					
				foreach $var (sort keys %list) {
					# AntiSpoofing information
					$var = $list{$var};
					%var = {};
					$val = '';
					if (($type eq 'Cluster Member') && (GetKey($var, 'shared') eq 'true')) {
						$val=html('(cluster interface) ');
					} elsif (GetKey($var, 'netaccess:leads_to_internet') eq 'true') {
						%var = %{txt2html('Internet', 'any_object', 'black')};
					} elsif (GetKey($var, 'netaccess:access') eq 'this') {
						%var = %{txt2html('This Net', 'network', 'black')};
					} elsif (GetKey($var, 'netaccess:access') eq 'specific') {
						%var = %{obj2html($rb_name, GetKey($var, 'netaccess:allowed:Table'), GetKey($var, 'netaccess:allowed:Name'), $_HREF)};
					} elsif (GetKey($var, 'netaccess:access') eq 'undefined') {
						$val = html('(not defined) ');
					}
					if (!$val) { $val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }
	
					push @section, {
						R1 => 1, 	
						OBJ_VAL1 => html(GetKey($var, 'officialname')), 
						OBJ_VAL2 => GetKey($var, 'ipaddr').'/'.$Subnet{GetKey($var, 'netmask')},
						OBJ_VAL3 => $val
					}
				}
				push @{$tobj[$cnt]{OBJ_TOPO}}, { T1 => \@section, C3 => 1};
			}
		}
		# VPN Domain ...
		if ((GetKey($obj, 'VPN_1') eq 'true') || (GetKey($obj, 'third_party_encryption') eq 'true')) {
			my @section;
			if ($type =~ /Host/) {
				push @section, { R3 => 1, OBJ_VAR => html('This Host'), OBJ_SET => $CheckTick };
			} elsif (GetKey($obj, 'encdomain') eq 'addresses_behind_gw') {
				push @section, { R3 => 1, OBJ_VAR => html('All IP addresses behind interfaces'), OBJ_SET => $CheckTick };
			} elsif (GetKey($obj, 'encdomain') eq 'manual') {
				%var = %{obj2html($rb_name, GetKey($obj, 'manual_encdomain:Table'), GetKey($obj, 'manual_encdomain:Name'), $_HREF)};
				$val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('Manually defined:'), OBJ_SET => $val };
			}
			push @{$tobj[$cnt]{OBJ_TOPO}}, { S0 => html('VPN Domain'), S1 => \@section };
		} elsif (GetKey($obj, 'type') eq 'sofaware_gateway') {
			my @section;
			if (GetKey($obj, 'encdomain') eq 'addresses_behind_gw') {
				push @section, { R3 => 1, OBJ_VAR => html('This gateway'), OBJ_SET => $CheckTick };
			} elsif (GetKey($obj, 'encdomain') eq 'manual') {
				%var = %{obj2html($rb_name, GetKey($obj, 'manual_encdomain:Table'), GetKey($obj, 'manual_encdomain:Name'), $_HREF)};
				$val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('Manually defined:'), OBJ_SET => $val };
			}
			push @{$tobj[$cnt]{OBJ_TOPO}}, { S0 => html('VPN Domain'), S1 => \@section };
		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_TOPO}}) { push @{$tobj[$cnt]{OBJ_TOPO}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}

	$template->param(CHECKPOINT=>\@tobj);

}


##########################################################################
# Routine: nodes2html
#
# Description:
# Routine to create the NODES table
##########################################################################
sub nodes2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my ($var, %var, $val);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                    NODES to HTML                         ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}
	
		$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
		
		#------------------------------------------------
		# OPTIONS column of Node
		#------------------------------------------------
		# Web Server configuration
		my @section;
		if (GetKey($obj, 'additional_products:is_web_server') eq 'true') {
			$var = (GetKey($obj, 'additional_products:web_server_prop:web_server_ports:0:has_standard_port') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Server uses Standard Port: 80'), OBJ_SET => $var };
			%var = %{obj2html($rb_name, GetKey($obj, 'additional_products:web_server_prop:protected_by_obj:Table'), GetKey($obj, 'additional_products:web_server_prop:protected_by_obj:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Protected by:'), OBJ_SET => $var };

			$var = 'OFF';
			if (GetKey($obj, 'additional_products:web_server_prop:cross_sites_scripting') eq 'true') {
				$val = GetKey($obj, 'additional_products:web_server_prop:strip');
				if ($val eq 'strip_HTML')     { $var = 'Block all HTML Tags' }
				if ($val eq 'strip_tags')     { $var = 'Block SCRIPT Tags' }
				if ($val eq 'strip_all_tags') { $var = 'Block all Tags' }
			}
			push @section, { R2 => 1, OBJ_VAR => html('Cross Site Scripting:'), OBJ_SET => html($var) };

			push @{$tobj[$cnt]{OBJ_OPTIONS}}, { S0 => html('Web Server'), S1 => \@section };

		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_OPTIONS}}) { push @{$tobj[$cnt]{OBJ_OPTIONS}}, { VOID => 1 } }
				
		#------------------------------------------------
		# ADVANCED column of Node
		#------------------------------------------------
		# NAT configuration
		my @section;
		if (GetKey($obj, 'add_adtr_rule') eq 'true') {
			if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Static') };
				$var = GetKey($obj, 'valid_ipaddr');
				push @section, { R2 => 1, OBJ_VAR => html('Translate to IP address:'), OBJ_SET => $var };
			} else {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Hide') };
				$var = GetKey($obj, 'valid_ipaddr');
				if ($var eq '0.0.0.0') {
					push @section, { R3 => 1, OBJ_VAR => html('Hide behind Gateway'), OBJ_SET => $CheckTick };
				} else {
					push @section, { R2 => 1, OBJ_VAR => html('Hide behind IP Address:'), OBJ_SET => $var };
				}
			}
			%var = %{obj2html($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Install on Gateway:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_ADVANCED}}, { S0 => html('Network Address Translation'), S1 => \@section };
		}            
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_ADVANCED}}) { push @{$tobj[$cnt]{OBJ_ADVANCED}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(NODES=>\@tobj);
}


##########################################################################
# Routine: networks2html
#
# Description:
# Routine to create the NETWORKS table
##########################################################################
sub networks2html {
	use strict;
	my (@ObjList, @tobj, $rb_name);
	my ($cnt, $obj, $name);
	my ($type, $current_type, $var, %var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                   NETWORKS to HTML                       ---
	#----------------------------------------------------------------
	$cnt   = -1;
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');
		$tobj[$cnt]{OBJ_NETMASK}    = GetKey($obj, 'netmask');

		#------------------------------------------------
		# Advanced column
		#------------------------------------------------
		# NAT configuration
		my @section;
		if (GetKey($obj, 'add_adtr_rule') eq 'true') {
			if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Static') };
				$var = GetKey($obj, 'valid_ipaddr');
				push @section, { R2 => 1, OBJ_VAR => html('Translate to IP address:'), OBJ_SET => $var };
			} else {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Hide') };
				$var = GetKey($obj, 'valid_ipaddr');
				if ($var eq '0.0.0.0') {
					push @section, { R3 => 1, OBJ_VAR => html('Hide behind Gateway'), OBJ_SET => $CheckTick };
				} else {
					push @section, { R2 => 1, OBJ_VAR => html('Hide behind IP Address:'), OBJ_SET => $var };
				}
			}
			%var = %{obj2html($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Install on Gateway:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_ADVANCED}}, { S0 => html('Network Address Translation'), S1 => \@section };
		}            
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_ADVANCED}}) { push @{$tobj[$cnt]{OBJ_ADVANCED}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(NETWORKS=>\@tobj);
}

##########################################################################
# Routine: domains2html
#
# Description:
# Routine to create the DOMAINS table
##########################################################################
sub domains2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my (%var, $var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                     DOMAINS to HTML                      ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}     = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}     = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(DOMAINS=>\@tobj);
}


##########################################################################
# Routine: accesspoints2html
#
# Description:
# Routine to create the ACCESSPOINTS table
##########################################################################
sub accesspoints2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $sct);
	my (%var, $var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                 ACCESSPOINTS to HTML                     ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		$sct = 0;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_APN}  = GetKey($obj, 'apn');

		#------------------------------------------------
		# ENDUSERDOMAIN column
		#------------------------------------------------
		if (GetKey($obj, 'enforce_domain') eq 'true') {
			my @section;

			%var = %{obj2html($rb_name, GetKey($obj, 'domain:Table'), GetKey($obj, 'domain:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('End User Domain:'), OBJ_SET => $var };

			$var = (GetKey($obj, 'enforce_other_apns') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Block MS to MS traffic between this and other APN End User domains'), OBJ_SET => $var };

			$var = (GetKey($obj, 'enforce_this_apn') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Block MS to MS traffic within this End User domain'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_ENDUSERDOMAIN}}, { S0 => html('Intra Tunnel Policy Enforcement'), S1 => \@section };
		}			

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(ACCESSPOINTS=>\@tobj);
}


##########################################################################
# Routine: osedevices2html
#
# Description:
# Routine to create the OSEDEVICES table
##########################################################################
sub osedevices2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my ($var, %var, @var, $val);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                  OSEDEVICES to HTML                      ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}
		
		$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
		$tobj[$cnt]{OBJ_TYPE} = $type;
		$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_IPADDRESS} = GetKey($obj, 'ipaddr');
		
		#------------------------------------------------
		# TOPOLOGY tab/column of OSE Device
		#------------------------------------------------
		# Interfaces ...
		@var = GetMembers(GetKey($obj, 'interfaces'));
		if (@var) {
			my @section;
			push @section, { R0 => 1,	OBJ_VAL1 => html('Name'), 
										OBJ_VAL2 => html('IP address'), 
										OBJ_VAL3 => html('Allowed addresses')
										};
			foreach $var (@var) {
				# AntiSpoofing information
				%var = {};
				$val = '';
				if (GetKey($var, 'netaccess:leads_to_internet') eq 'true') {
					%var = %{txt2html('Internet', 'any_object', 'black')};
				} elsif (GetKey($var, 'netaccess:access') eq 'this') {
					%var = %{txt2html('This Net', 'network', 'black')};
				} elsif (GetKey($var, 'netaccess:access') eq 'specific') {
					%var = %{obj2html($rb_name, GetKey($var, 'netaccess:allowed:Table'), GetKey($var, 'netaccess:allowed:Name'), $_HREF)};
				} elsif (GetKey($var, 'netaccess:access') eq 'undefined') {
					$val = ' ';
				}
				if (!$val) { $val = $var{OBJ_ICON}.' '.$var{OBJ_TEXT} }

				push @section, { 	
					R1 => 1,
					OBJ_VAL1 => html(GetKey($var, 'officialname')), 
					OBJ_VAL2 => GetKey($var, 'ipaddr').'/'.$Subnet{GetKey($var, 'netmask')},
					OBJ_VAL3 => $val
				}
			}
			push @{$tobj[$cnt]{OBJ_TOPO}}, { T1 => \@section, C3 => 1};
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_TOPO}}) { push @{$tobj[$cnt]{OBJ_TOPO}}, { VOID => 1 } }

		#------------------------------------------------
		# SETUP column/tab of OSE Device
		#------------------------------------------------
		if ($type =~ /Cisco/i) {
			my @section;
			$var = GetKey($obj, 'alnum');
			push @section, { R2 => 1, OBJ_VAR => html('Access List No.:'), OBJ_SET => $var };
			$var = GetKey($obj, 'version');
			push @section, { R2 => 1, OBJ_VAR => html('Version:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_SETUP}}, { S0 => html('Device specifics'), S1 => \@section };
		} elsif ($type =~ /Nortel/i) {
			my @section;
			$var = GetKey($obj, 'slot_num');
			push @section, { R2 => 1, OBJ_VAR => html('Volume:'), OBJ_SET => $var };
			$var = GetKey($obj, 'config_file');
			push @section, { R2 => 1, OBJ_VAR => html('Config File:'), OBJ_SET => $var };
			$var = GetKey($obj, 'version');
			push @section, { R2 => 1, OBJ_VAR => html('Version:'), OBJ_SET => $var };
			
			push @{$tobj[$cnt]{OBJ_SETUP}}, { S0 => html('Device specifics'), S1 => \@section };
		}

		if ($type =~ /Cisco|3Com/i) {
			my @section;
			$var = GetKey($obj, 'rules_dir');
			push @section, { R2 => 1, OBJ_VAR => html('Rules:'), OBJ_SET => $var };
			$var = GetKey($obj, 'spoof_dir');
			push @section, { R2 => 1, OBJ_VAR => html('Spoof rules:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_SETUP}}, { S0 => html('Interface Directions'), S1 => \@section };
		}
		if ($type =~ /3Com/i) {
			my @section;
			$var = (GetKey($obj, 'deny_record_route') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny Route Recording'), OBJ_SET => $var };
			$var = (GetKey($obj, 'deny_src_route') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny Src Routing'), OBJ_SET => $var };
			$var = (GetKey($obj, 'deny_tiny_fragment') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny Tiny Fragments'), OBJ_SET => $var };
			$var = (GetKey($obj, 'deny_time_stamp') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny Time Stamping'), OBJ_SET => $var };
			$var = (GetKey($obj, 'deny_ip_tunnel') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny IP Tunnels'), OBJ_SET => $var };
			$var = (GetKey($obj, 'deny_src_spoofing') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deny Src Spoofing (3Com)'), OBJ_SET => $var };
			$var = (GetKey($obj, 'generate_icmp') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Generate ICMP Errors'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_SETUP}}, { S0 => html('Service Independent Filters'), S1 => \@section };
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_SETUP}}) { push @{$tobj[$cnt]{OBJ_SETUP}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENT column
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(OSEDEVICES=>\@tobj);
}


##########################################################################
# Routine: objectgroups2html
#
# Description:
# Routine to create the ObjectGROUPS table
##########################################################################
sub objectgroups2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                    GROUPS to HTML                        ---
	#----------------------------------------------------------------
	$cnt   = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}
		
		$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($rb_name, $obj, $name);
		if ($type =~ /Simple/i) {
			$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($rb_name, $obj, $_HREF);
			$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
		} elsif ($type =~ /Exclusion/i) {
			$tobj[$cnt]{OBJ_MEMBERS}    = HTMLObjectList($rb_name, GetKey($obj, 'base:Table'), GetKey($obj, 'base:Name'), $_HREF);
			$tobj[$cnt]{OBJ_EXCLUSION}  = HTMLObjectList($rb_name, GetKey($obj, 'exception:Table'), GetKey($obj, 'exception:Name'), $_HREF);
		} elsif ($type =~ /User Authority/i) {
			$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($rb_name, GetKey($obj, 'members'), $_HREF);
			$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
		} elsif ($type =~ /GSN Handover/i) {
			$tobj[$cnt]{OBJ_MEMBERS}    = HTMLMemberList($rb_name, $obj, $_HREF);
			$tobj[$cnt]{OBJ_EXCLUSION}  = [{OBJ_TEXT => html(' ')}];
		}
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(OBJECTGROUPS=>\@tobj);
}


##########################################################################
# Routine: logicalservers2html
#
# Description:
# Routine to create the LOGICALSERVERS table
##########################################################################
sub logicalservers2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my ($var, $val, %var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                LOGICALSERVERS to HTML                    ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}       = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}       = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_IPADDRESS}  = GetKey($obj, 'ipaddr');

		my @section;
		$var = (GetKey($obj, 'servers_type') eq 'http')?'HTTP':'Other';
		push @section, { R2 => 1, OBJ_VAR => html("Server's type:"), OBJ_SET => html($var) };

		%var = %{obj2html($rb_name, GetKey($obj, 'servers:Table'), GetKey($obj, 'servers:Name'), $_HREF)};
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		push @section, { R2 => 1, OBJ_VAR => html('Servers group:'), OBJ_SET => $var };

		$var = '';
		$val = $Check;
		if (GetKey($obj, 'use_cache') eq 'true') {
			$val = $CheckTick;
			$var = (GetKey($obj, 'cache_type') eq 'by_service')?'Persistency by service':'Persistency by server';
		}
		push @section, { R2 => 1, OBJ_VAR => $val.'Persistent mode:', OBJ_SET => html($var) };

		$val = GetKey($obj, 'method');
		$var = 'Server Load' if ($val eq '"server load"');
		$var = 'Round Trip' if ($val eq '"round trip"');
		$var = 'Round Robin' if ($val eq '"round robin"');
		$var = 'Random' if ($val eq 'random');
		$var = 'Domain' if ($val eq 'domain');
		push @section, { R2 => 1, OBJ_VAR => html('Balance method:'), OBJ_SET => html($var) };

		push @{$tobj[$cnt]{OBJ_GENERAL}}, { S1 => \@section };
		
		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(LOGICALSERVERS=>\@tobj);
}


##########################################################################
# Routine: addressranges2html
#
# Description:
# Routine to create the ADDRESSRANGES table
##########################################################################
sub addressranges2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my (%var, $var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                ADDRESSRANGES to HTML                    ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}    = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}    = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_FIRSTIP} = GetKey($obj, 'ipaddr_first');
		$tobj[$cnt]{OBJ_LASTIP}  = GetKey($obj, 'ipaddr_last');

		# NAT configuration
		my @section;
		if (GetKey($obj, 'add_adtr_rule') eq 'true') {
			if (GetKey($obj, 'netobj_adtr_method') eq 'adtr_static') {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Static') };
				$var = GetKey($obj, 'valid_ipaddr');
				push @section, { R2 => 1, OBJ_VAR => html('Translate to IP address:'), OBJ_SET => $var };
			} else {
				push @section, { R2 => 1, OBJ_VAR => html('Translation method:'), OBJ_SET => html('Hide') };
				$var = GetKey($obj, 'valid_ipaddr');
				if ($var eq '0.0.0.0') {
					push @section, { R3 => 1, OBJ_VAR => html('Hide behind Gateway'), OBJ_SET => $CheckTick };
				} else {
					push @section, { R2 => 1, OBJ_VAR => html('Hide behind IP Address:'), OBJ_SET => $var };
				}
			}
			%var = %{obj2html($rb_name, GetKey($obj, 'the_firewalling_obj:Table'), GetKey($obj, 'the_firewalling_obj:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Install on Gateway:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_ADVANCED}}, { S0 => html('Network Address Translation'), S1 => \@section };
		}            
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OBJ_ADVANCED}}) { push @{$tobj[$cnt]{OBJ_ADVANCED}}, { VOID => 1 } }

		$tobj[$cnt]{OBJ_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(ADDRESSRANGES=>\@tobj);
}


##########################################################################
# Routine: dynamicobjects2html
#
# Description:
# Routine to create the DYNAMICOBJECTS table
##########################################################################
sub dynamicobjects2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                DYNAMICOBJECTS to HTML                    ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{OBJ_TEXT}     = stripquotes($name);
		$tobj[$cnt]{OBJ_NAME}     = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(DYNAMICOBJECTS=>\@tobj);
}


##########################################################################
# Routine: voipdomains2html
#
# Description:
# Routine to create the VOIPDOMAINS table
##########################################################################
sub voipdomains2html {
	use strict;
	my (@ResourceList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $spec);

	my (@var, %var, $var, $val);
	my (@tmp, $none);
	my ($Usage, $share, $server);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                  VOIPDOMAINS to HTML                     ---
	#----------------------------------------------------------------
	$cnt  = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);
		$spec = GetKey($obj, 'spec_select') || GetKey($obj, 'resource_with_cvp');

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{OBJ_TEXT} = stripquotes($name);
		$tobj[$cnt]{OBJ_TYPE} = $type;
		$tobj[$cnt]{OBJ_NAME} = HTMLObjectList($rb_name, $obj, $name);

		#------------------------------------------------
		# GENERAL tab/column in VoIP Domain
		#------------------------------------------------
		my @section;
		%var = %{obj2html($rb_name, GetKey($obj, 'zone_range:Table'), GetKey($obj, 'zone_range:Name'), $_HREF)};
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		push @section, { R2 => 1, OBJ_VAR => html('Related endpoints domain:'), OBJ_SET => $var };

		%var = %{obj2html($rb_name, GetKey($obj, 'server:Table'), GetKey($obj, 'server:Name'), $_HREF)};
		$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
		push @section, { R2 => 1, OBJ_VAR => html('VoIP installed at:'), OBJ_SET => $var };

		push @{$tobj[$cnt]{OBJ_GENERAL}}, { S1 => \@section };

		#------------------------------------------------
		# ROUTING Mode tab/column in VoIP Domain
		#------------------------------------------------
		if ($type =~ /H.323/) {
			my @section;
			if ($type =~ /Gatekeeper/) {
				@var = GetMembers(GetKey($obj, 'h323_gatekeeper_protocols'));
			} elsif ($type =~ /Gateway/) {
				@var = GetMembers(GetKey($obj, 'h323_gateway_protocols'));
			}
			foreach $var (@var) { 
				if ($var eq 'H.225')    { $val += 1 }
				if ($var eq 'H.245')    { $val += 2 }
				if ($var eq 'RTP/RTCP') { $val += 4 }
			}
			if ($type =~ /Gatekeeper/) {
				$var = ($val&1)?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Direct'), OBJ_SET => $var };
			}
			$var = ($val&2)?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Call Setup (Q.931)'), OBJ_SET => $var };
			$var = ($val&4)?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Call Setup (Q.931) and Call Control (H.245)'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OBJ_ROUTING}}, { S0 => html('Allowed Routing Mode'), S1 => \@section };
		}

		#------------------------------------------------
		# COMMENTS tab/column in VoIP Domain
		#------------------------------------------------
		$tobj[$cnt]{OBJ_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(VOIPDOMAINS=>\@tobj);
}


##########################################################################
# Routine: services2html
#
# Description:
# Routine to create the SERVICES table
##########################################################################
sub services2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $var);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                   SERVICES to HTML                       ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OBJ_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{SVC_TEXT}       = stripquotes($name);
		$tobj[$cnt]{SVC_NAME}       = HTMLObjectList($rb_name, $obj, $name);

		#------------------------------------------------
		# SPECIFICATION column of the SERVICES
		#------------------------------------------------
		my @section;
		if ($type eq 'TCP' || $type eq 'UDP') {
			push @section, { R2 => 1, OBJ_VAR => html('Port:'), OBJ_SET => stripquotes(GetKey($obj, 'port')) };
		} elsif ($type eq 'Compound TCP') {
			push @section, { R2 => 1, OBJ_VAR => html('Port:'), OBJ_SET => stripquotes(GetKey($obj, 'port')) };
			push @section, { R2 => 1, OBJ_VAR => html('Compound Service:'), OBJ_SET => GetKey($obj, 'svc_type') };
		} elsif ($type eq 'Citrix TCP') {
			push @section, { R2 => 1, OBJ_VAR => html('Port:'), OBJ_SET => stripquotes(GetKey($obj, 'port')) };
		} elsif ($type eq 'RPC') {
			push @section, { R2 => 1, OBJ_VAR => html('Program Number:'), OBJ_SET => stripquotes(GetKey($obj, 'port')) };
		} elsif ($type eq 'ICMP') {
			push @section, { R2 => 1, OBJ_VAR => html('Type:'), OBJ_SET => GetKey($obj, 'icmp_type') };
			push @section, { R2 => 1, OBJ_VAR => html('Code:'), OBJ_SET => GetKey($obj, 'icmp_code') };
		} elsif ($type eq 'Other') {
			push @section, { R2 => 1, OBJ_VAR => html('IP Protocol:'), OBJ_SET => GetKey($obj, 'protocol') };
		} elsif ($type eq 'DCE-RPC') {
			push @section, { R2 => 1, OBJ_VAR => html('Interface UUID:'), OBJ_SET => GetKey($obj, 'uuid') };
			push @section, { R2 => 1, OBJ_VAR => html('Protocol Type:'), OBJ_SET => GetKey($obj, 'proto_type') };
		} elsif ($type eq 'GTP') {
			$var = (GetKey($obj, 'type') eq 'gtp_v1')?'1':'0';
			push @section, { R2 => 1, OBJ_VAR => html('Version:'), OBJ_SET => $var };
			push @section, { R2 => 1, OBJ_VAR => html('Port:'), OBJ_SET => stripquotes(GetKey($obj, 'port')) };
		}
			
		push @{$tobj[$cnt]{SVC_SPEC}}, { S1 => \@section };
		
		#------------------------------------------------
		# COMMENTS column of the SERVICES
		#------------------------------------------------
		$tobj[$cnt]{SVC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(SERVICES=>\@tobj);
}

##########################################################################
# Routine: servicegroups2html
#
# Description:
# Routine to create the SERVICEGROUPS table
##########################################################################
sub servicegroups2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                 SERVICEGROUPS to HTML                    ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable, $name);

		$tobj[$cnt]{SVC_TEXT}       = stripquotes($name);
		$tobj[$cnt]{SVC_NAME}       = HTMLObjectList($rb_name, $obj, $name);
		$tobj[$cnt]{SVC_MEMBERS}    = HTMLMemberList($rb_name, $obj, $_HREF);
		$tobj[$cnt]{SVC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(SERVICEGROUPS=>\@tobj);
}

##########################################################################
# Routine: resources2html
#
# Description:
# Routine to create the RESOURCES table
##########################################################################
sub resources2html {
	use strict;
	my (@ResourceList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $spec);

	my (@var, %var, $var, $val, $v1, $v2);
	my (@tmp, $none);
	my ($Usage, $share, $server);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                   RESOURCES to HTML                      ---
	#----------------------------------------------------------------
	$cnt  = -1;
	$none = '';
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);
		$spec = GetKey($obj, 'spec_select') || GetKey($obj, 'resource_with_cvp');

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{RSC_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{RSC_TEXT} = stripquotes($name);
		$tobj[$cnt]{RSC_TYPE} = $type;
		$tobj[$cnt]{RSC_NAME} = HTMLObjectList($rb_name, $obj, $name);
		
		#------------------------------------------------
		# GENERAL tab/column in Resource
		#------------------------------------------------
		if ($type eq 'URI') {
			my @section;
			$Usage = 0;
			if (GetKey($obj, 'url_logging') eq 'true') {
				$var = 'Optimize URL logging';
				$Usage = 1;
			} elsif (GetKey($obj, 'url_logging') eq 'enhance_ufp') {
				$var = 'Enhance UFP performance';
				$Usage = 3;
			} else {
				$var = 'Enforce URI capabilities';
				$Usage = 2;
			}
			push @section, { R2 => 1, OBJ_VAR => html('Use resource to:'), OBJ_SET => html($var) };
			
			push @{$tobj[$cnt]{RSC_GENERAL}}, { S0 => html('URI Specifications'), S1 => \@section };

			if ($Usage == 2) {
				my @section;
				push @section, { H1 => 1, OBJ_VAL1 => html('Connection Methods'), OBJ_VAL2 => html('URI Match Type') };

				# Connection Methods are stored in a list ...
				@var = GetMembers(GetKey($obj, 'connection_method'));
				%var = ();
				foreach $var (@var) { $var{$var} = 'true' }

				# URI Match Specification Type ...
				$val = 0;
				if ($spec eq 'ufp')             { $val = 4 }
				elsif ($spec eq '"wild cards"') { $val = 1 }
				elsif ($spec eq 'file')         { $val = 2 }
				
				push @section, { R0 => 1, 	OBJ_VAL1 => (($var{Transparent})?$CheckTick:$Check).html(' Transparent'), 
											OBJ_VAL2 => (($val & 1)?$MarkDot:$Mark).html(' Wildcards') };
				push @section, { R0 => 1, 	OBJ_VAL1 => (($var{Proxy})?$CheckTick:$Check).html(' Proxy'), 
											OBJ_VAL2 => (($val & 2)?$MarkDot:$Mark).html(' File') };
				push @section, { R0 => 1, 	OBJ_VAL1 => (($var{Tunneling})?$CheckTick:$Check).html(' Tunneling'), 
											OBJ_VAL2 => (($val & 4)?$MarkDot:$Mark).html(' UFP') };

				push @{$tobj[$cnt]{RSC_GENERAL}}, { S2 => \@section };
			}
		} 
	
		if ($type eq 'SMTP') {
			# Mail delivery ...
			my @section;

			$var = GetKey($obj, 'default_server');
			if ($var) {
				push @section, { R2 => 1, OBJ_VAR => html('Server:'), OBJ_SET => $var };
			}
			$var = (GetKey($obj, 'resolve_recipient_domain') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Deliver messages using DNS/MX records'), OBJ_SET => $var };

			$var = (GetKey($obj, 'check_rulebase_again') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Check Rule Base with new destination'), OBJ_SET => $var };

			push @{$tobj[$cnt]{RSC_GENERAL}}, { S0 => html('Mail delivery'), S1 => \@section };

			if (GetKey($obj, 'err_notify') eq 'true') {
				# Error notification delivery ...
				my @section;
				$var = (GetKey($obj, 'err_notify') eq 'true')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Notify sender on error'), OBJ_SET => $var };
				
				if (GetKey($obj, 'err_notify') eq 'true') {
					$var = GetKey($obj, 'error_server');
					push @section, { R2 => 1, OBJ_VAR => html('Server:'), OBJ_SET => $var };

					$var = (GetKey($obj, 'resolve_sender_domain') eq 'true')?$CheckTick:$Check;
					push @section, { R3 => 1, OBJ_VAR => html('Deliver messages using DNS/MX records'), OBJ_SET => $var };
				
					$var = (GetKey($obj, 'err_check_rulebase_again') eq 'true')?$CheckTick:$Check;
					push @section, { R3 => 1, OBJ_VAR => html('Check Rule Base with new destination'), OBJ_SET => $var };
				}
				push @{$tobj[$cnt]{RSC_GENERAL}}, { S0 => html('ERROR Mail delivery'), S1 => \@section };
			}
		}

		if ($type eq 'TCP') {
			# TCP Resource type ...
			my @section;
			$var = GetKey($obj, 'resource_with_cvp');
			push @section, { R0 => 1, 	OBJ_VAL1 => (($var eq 'true')?$MarkDot:$Mark).html(' UFP'), 
										OBJ_VAL2 => (($var eq 'false')?$MarkDot:$Mark).html(' CVP') };
				
			push @{$tobj[$cnt]{RSC_GENERAL}}, { S0 => html('Type'), S2 => \@section };
		}
		if ((($type eq 'URI') && !($Usage == 1)) || 
			($type eq 'SMTP') || ($type eq 'TCP') || ($type eq 'FTP')) {
				# Exception Track ...
				my @section;
				$var = GetKey($obj, 'except_track:Name');
				$val = 0;
				if ($var eq 'None')                 { $val = 1 }
				elsif ($var eq '"Exception Log"')   { $val = 2 }
				elsif ($var eq '"Exception Alert"') { $val = 4 }
				push @section, { R0 => 1, 	OBJ_VAL1 => (($val & 1)?$MarkDot:$Mark).html(' None'), 
											OBJ_VAL2 => (($val & 2)?$MarkDot:$Mark).html(' Log'), 
											OBJ_VAL3 => (($val & 4)?$MarkDot:$Mark).html(' Alert') };
				
				push @{$tobj[$cnt]{RSC_GENERAL}}, { S0 => html('Exception Track'), S2 => \@section };
			}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{RSC_GENERAL}}) { push @{$tobj[$cnt]{RSC_GENERAL}}, { VOID => 1 } }

		#------------------------------------------------
		# MATCH tab/column in Resource
		#------------------------------------------------
		if (($type eq 'URI') && ($Usage == 2) && ($spec eq '"wild cards"')) {
			# URI resource with wild cards ...

			# Schemes used ...
			my @section;
			push @section, {	H0 => 1, OBJ_VAL1 => html('Schemes') };

			$val = GetKey($obj, 'multi_scheme_val');
			push @section, { 	R0=> 1, 
								OBJ_VAL1 => (($val & $Scheme{http})?$CheckTick:$Check).html(' http'),
								OBJ_VAL2 => (($val & $Scheme{ftp})?$CheckTick:$Check).html(' ftp'),
								OBJ_VAL3 => (($val & $Scheme{gopher})?$CheckTick:$Check).html(' gopher'),
								OBJ_VAL4 => ''};
			push @section, { 	R0=> 1, 
								OBJ_VAL1 => (($val & $Scheme{mailto})?$CheckTick:$Check).html(' mailto'),
								OBJ_VAL2 => (($val & $Scheme{news})?$CheckTick:$Check).html(' news'),
								OBJ_VAL3 => (($val & $Scheme{wais})?$CheckTick:$Check).html(' wais'),
								OBJ_VAL4 => html('Other: '.stripquotes(GetKey($obj, 'other_scheme'))) };

			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('Wild card specifications'), S2 => \@section };
								
			# Methods used ...
			my @section;
			push @section, {	H0 => 1, OBJ_VAL1 => html('Methods') };

			$val = GetKey($obj, 'multi_method_val');
			push @section, { 	R0=> 1, 
								OBJ_VAL1 => (($val & $MethodHTTP{get})?$CheckTick:$Check).html(' GET'),
								OBJ_VAL2 => (($val & $MethodHTTP{post})?$CheckTick:$Check).html(' POST'),
								OBJ_VAL3 => (($val & $MethodHTTP{head})?$CheckTick:$Check).html(' HEAD'),
								OBJ_VAL4 => (($val & $MethodHTTP{put})?$CheckTick:$Check).html(' PUT'),
								OBJ_VAL5 => html('Other: '.stripquotes(GetKey($obj, 'other_method'))) 
							};

			push @{$tobj[$cnt]{RSC_MATCH}}, { S2 => \@section };
	
			my @section;
			# Host, Path and Query ...
			push @section, { H0 => 1 };
			$var = (GetKey($obj, 'host'))?stripquotes(GetKey($obj, 'host')):$none;
			push @section, { R2 => 1, OBJ_VAR => html('Host:'), OBJ_SET => $var };
			$var = (GetKey($obj, 'path'))?stripquotes(GetKey($obj, 'path')):$none;
			push @section, { R2 => 1, OBJ_VAR => html('Path:'), OBJ_SET => $var };
			$var = (GetKey($obj, 'query'))?stripquotes(GetKey($obj, 'query')):$none;
			push @section, { R2 => 1, OBJ_VAR => html('Query:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{RSC_MATCH}}, { S1 => \@section };

		} elsif ((($type eq 'URI') && ($Usage != 1) && ($spec eq 'ufp')) ||
				 (($type eq 'TCP') && ($spec eq 'false'))) {
			# UFP options ...
			my @section;
			
			# UFP server ...
			%var = %{obj2html($rb_name, GetKey($obj, 'ufpserver:Table'), GetKey($obj, 'ufpserver:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('UFP server:'), OBJ_SET => $var };
			
			# UFP caching control ...
			$var = GetKey($obj, 'cacheufp');
			if ($var eq 'none') 		{ $var = 'No caching' }
			elsif ($var eq 'server') 	{ $var = 'UFP server' }
			elsif ($var eq 'client') 	{ $var = 'VPN-1 & FireWall-1 (one request)' }
			elsif ($var eq 'client2nd')	{ $var = 'VPN-1 & FireWall-1 (two requests)' }
			push @section, { R2 => 1, OBJ_VAR => html('UFP caching control:'), OBJ_SET => html($var) };
			
			# UFP-Categories ...
			@var = GetMembers(GetKey($obj, 'ufp_categories'));
			push @section, { H1 => 1, OBJ_VAR => html('Categories') };
			if (@var) { 
				foreach $var (@var) { 
					if (GetKey($var, 'selected') eq 'true') {
						push @tmp, stripquotes(GetKey($var, 'name'));
					}
				}
				foreach $var (sort @tmp) {
					push @section, { R1 => 1, OBJ_VAR => html($var) };
				}
			} else {
				push @section, {  R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('UFP Options'), S1 => \@section };

			# UFP server failure ...
			my @section;
			push @section, { H0 => 1 };
			$var = (GetKey($obj, 'ufp_fail_allowed') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Ignore UFP server after connection failure'), OBJ_SET => $var };
			if (GetKey($obj, 'ufp_fail_allowed') eq 'true') {
				$var = GetKey($obj, 'ufp_skip_threshold');
				push @section, { R2 => 1, OBJ_VAR => html('Number of failures before ignoring the UFP server:'), OBJ_SET => html($var) };
				$var = GetKey($obj, 'ufp_skip_time');
				push @section, { R2 => 1, OBJ_VAR => html('Timeout before reconnect to UFP server (seconds):'), OBJ_SET => html($var) };
			}
			push @{$tobj[$cnt]{RSC_MATCH}}, { S1 => \@section };
				
		} elsif ($type eq 'SMTP') {
			# SMTP Resource ....
			my @section;
			
			@var = @{GetKey($obj, 'match_from')};
			$val = 'Sender:';
			foreach $var (@var) {
				push @section, { R2 => 1, OBJ_VAR => $val, OBJ_SET => stripquotes($var) };
				$val = '';
			}
			@var = @{GetKey($obj, 'match_to')};
			$val = 'Recipient:';
			foreach $var (@var) {
				push @section, { R2 => 1, OBJ_VAR => $val, OBJ_SET => stripquotes($var) };
				$val = '';
			}
			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('SMTP inspection'), S1 => \@section };
				
		} elsif ($type eq 'FTP') {
			# FTP Resource ....
			my @section;
			if (GetKey($obj, 'path')) { $var = stripquotes(GetKey($obj, 'path')) } else { $var = $none }
			push @section, { R2 => 1, OBJ_VAR => html('Path:'), OBJ_SET => $var };
			
			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('FTP inspection'), S1 => \@section };

			# Methods used in MATCH tab ...
			my @section;
			push @section, { H0 => 1, OBJ_VAL1 => html('Methods') };
			@var = ();
			$val = GetKey($obj, 'method_val');
			push @section, { 	R0 => 1, 
								OBJ_VAL1 => (($val & $MethodFTP{GET})?$CheckTick:$Check).html(' GET'),
								OBJ_VAL2 => (($val & $MethodFTP{PUT})?$CheckTick:$Check).html(' PUT') };

			push @{$tobj[$cnt]{RSC_MATCH}}, { S2 => \@section };

		} elsif ($type eq 'CIFS') {
			# CIFS Resource (allowed share list) ...
			my @section;

			push @section, { R0 => 1, OBJ_VAL1 => html('Server Name'), OBJ_VAL2 => html('Share Name') };

			@var = @{GetKey($obj, 'allowed_shares_list')};
			foreach $var (@var) {
				($server, $share) = split(/#/, stripquotes($var));
				push @section, { 	R1 => 1, 
									OBJ_VAL1 => $server, 
									OBJ_VAL2 => $share };
			}

			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('Allowed Disk\Print Shares'), T1 => \@section, C2 => 1 };

			# CIFS log settings ...			
			my @section;
			push @section, { H0 => 1 };
			$var = GetKey($obj, 'log_mapped_shares');
			push @section, { R3 => 1, OBJ_VAR => html('Log mapped shares'), OBJ_SET => ($var eq 'true')?$CheckTick:$Check };
			$var = GetKey($obj, 'log_access_violation');
			push @section, { R3 => 1, OBJ_VAR => html('Log access violation'), OBJ_SET => ($var eq 'true')?$CheckTick:$Check };
			$var = GetKey($obj, 'block_remote_registry');
			push @section, { R3 => 1, OBJ_VAR => html('Block remote registry access'), OBJ_SET => ($var eq 'true')?$CheckTick:$Check };
			
			push @{$tobj[$cnt]{RSC_MATCH}}, { S1 => \@section };
			
		} elsif ($type eq 'URI for QoS') {
			# URI for QOS Resource (used for floodgate) ...
			my @section;
			
			if (GetKey($obj, 'path')) { $var = stripquotes(GetKey($obj, 'path')) } else { $var = $none }
			push @section, { R2 => 1, OBJ_VAR => html('URL:'), OBJ_SET => $var };
			
			push @{$tobj[$cnt]{RSC_MATCH}}, { S0 => html('Search definition'), S1 => \@section };
				
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{RSC_MATCH}}) { push @{$tobj[$cnt]{RSC_MATCH}}, { VOID => 1 } }

		#------------------------------------------------
		# ACTIONS tab/column in Resource
		#------------------------------------------------
		if (($type eq 'URI') && ($Usage == 2)) {
			# URI Resources all have the same ACTIONS tab/column
			my @section;
			
			# Replacement URI ...
			$var = stripquotes(GetKey($obj, 'redirect_to'));
			push @section, { R2 => 1, OBJ_VAR => html('Replacement URI:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{RSC_ACTIONS}}, { S1 => \@section };

			# HTML Weeding & Response Scanning ...
			my @section;
			push @section, { H1 => 1, OBJ_VAL1 => html('HTML Weeding'), OBJ_VAL2 => html('Response Scanning') };
			
			$var = (GetKey($obj, 'response:strip_script_tags') eq 'true')?$CheckTick:$Check;
			$val = (GetKey($obj, 'response:strip_java') eq 'true')?$CheckTick:$Check;
			push @section, { R0=> 1, OBJ_VAL1 => $var.html(' Strip SCRIPT Tags'), OBJ_VAL2 => $val.html(' Block JAVA Code') };
			$var = (GetKey($obj, 'response:strip_applet_tags') eq 'true')?$CheckTick:$Check;
			push @section, { R0=> 1, OBJ_VAL1 => $var.html(' Strip Applet Tags') };
			$var = (GetKey($obj, 'response:strip_active_tags') eq 'true')?$CheckTick:$Check;
			push @section, { R0=> 1, OBJ_VAL1 => $var.html(' Strip ActiveX Tags') };
			$var = (GetKey($obj, 'response:http_erase_ftp_link') eq 'true')?$CheckTick:$Check;
			push @section, { R0=> 1, OBJ_VAL1 => $var.html(' Strip FTP Links') };
			$var = (GetKey($obj, 'response:http_erase_port_cmd') eq 'true')?$CheckTick:$Check;
			push @section, { R0=> 1, OBJ_VAL1 => $var.html(' Strip Port Strings') };

			push @{$tobj[$cnt]{RSC_ACTIONS}}, { S2 => \@section };
			
			# CVP Options in URI resource ...
			if (GetKey($obj, 'response:av_use') eq 'true') {
				my @section;
				%var = %{obj2html($rb_name, GetKey($obj, 'response:av_server:Table'), GetKey($obj, 'response:av_server:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('CVP server:'), OBJ_SET => $var };
				
				$var = (GetKey($obj, 'response:av_setting') eq 'cure')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('CVP server is allowed to modify content'), OBJ_SET => $var };
				$var = (GetKey($obj, 'response:av_headers') eq 'true')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Send HTTP headers to CVP server'), OBJ_SET => $var };
				$var = (GetKey($obj, 'response:http_cvp_on_request') eq 'true')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Send HTTP requests to CVP server'), OBJ_SET => $var };
				$var = (GetKey($obj, 'response:http_cvp_only_unsafe') eq 'true')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Send only unsafe file types to CVP server'), OBJ_SET => $var };

				# Reply Order
				push @section, { H1 => 1, OBJ_VAR => html('Reply Order') };
				$var = (GetKey($obj, 'response:reply_mode') eq 'reply_first')?$MarkDot:$Mark;
				push @section, { R3 => 1, OBJ_VAR => html('Return data after content is approved'), OBJ_SET => $var };
				$var = (GetKey($obj, 'response:reply_mode') eq 'reply_first')?$Mark:$MarkDot;
				push @section, { R3 => 1, OBJ_VAR => html('Return data before content is approved'), OBJ_SET => $var };
				
				push @{$tobj[$cnt]{RSC_ACTIONS}}, { S0 => html('CVP Options'), S1 => \@section };
			}
	
			# SOAP Inspection ...
			# if URI spec type is wildcards AND schemes include http ...
			if (($spec eq '"wild cards"') &&
				((GetKey($obj, 'multi_scheme_val') & $Scheme{http}) || (GetKey($obj, 'other_scheme') eq '"*"'))) {

				my @section;
				$var = (GetKey($obj, 'request:soap_accept_all_conn') eq 'accept_all_conns')?$MarkDot:$Mark;
				push @section, { R3 => 1, OBJ_VAR => html('Allow all SOAP requests'), OBJ_SET => $var };
				$var = (GetKey($obj, 'request:soap_accept_all_conn') eq 'accept_by_file_id')?$MarkDot:$Mark;
				push @section, { R3 => 1, OBJ_VAR => html('Allow SOAP requests as specified in file'), OBJ_SET => $var };
				if ($var eq $MarkDot) {
					push @section, { R2 => 1, OBJ_VAR => $Spacer.html('File ID:'), OBJ_SET => stripquotes(GetKey($obj, 'request:soap_file_id')) };
				}
				push @{$tobj[$cnt]{RSC_ACTIONS}}, { S0 => html('SOAP Inspection'), S1 => \@section };

				my @section;
				push @section, { H1 => 1, OBJ_VAR => html('Track') };
				$var = (GetKey($obj, 'request:soap_track') eq 'none')?'None':'Log';
				push @section, { R2 => 1, OBJ_VAR => html('Track SOAP connections:'), OBJ_SET => $var };

				push @{$tobj[$cnt]{RSC_ACTIONS}}, { S1 => \@section };
			}
		} 
		
		if ($type eq 'SMTP') {
			# Rewriting rules ...
			my @section;
			@var = GetMembers(GetKey($obj, 'from'));
			push @section, { 	R24 => 1,
								OBJ_VAL1 => html('Sender:'), 
								OBJ_VAL2 => html(stripquotes($var[0])), 
								OBJ_VAL3 => html(' -> '),
								OBJ_VAL4 => html(stripquotes($var[1])) };

			@var = GetMembers(GetKey($obj, 'to'));
			push @section, { 	R24 => 1,
								OBJ_VAL1 => html('Recipient:'), 
								OBJ_VAL2 => html(stripquotes($var[0])), 
								OBJ_VAL3 => html(' -> '),
								OBJ_VAL4 => html(stripquotes($var[1])) };

			@var = GetMembers(GetKey($obj, 'user_field'));
			if ($var[0]) {
				push @section, { 	R24 => 1,
									OBJ_VAL1 => html(stripquotes($var[0]).':'), 
									OBJ_VAL2 => html(stripquotes($var[1])), 
									OBJ_VAL3 => html(' -> '),
									OBJ_VAL4 => html(stripquotes($var[2])) };
			}
			push @{$tobj[$cnt]{RSC_ACTIONS}}, { SECTION => html('Rewriting Rules'), S2 => \@section };

			# Strip code ...
			@var = GetMembers(GetKey($obj, 'content_type'));
			$v1 = stripquotes($var[0]);
			@var = GetMembers(GetKey($obj, 'forbiddenfiles'));
			$v2 = stripquotes($var[0]);
			push @{$tobj[$cnt]{RSC_ACTIONS}},
				{ 	SECTION => html('Attachment handling'), 
					S2 => 	[{ 	R2 => 1, 
								OBJ_VAL1 => html('Strip MIME of type:'), 
								OBJ_VAL2 => $v1 },
							 { 	R2 => 1, 
							 	OBJ_VAL1 => html('Strip file by name:'), 
							 	OBJ_VAL2 => $v2 }] };

			# Other parameters ...
			push @{$tobj[$cnt]{RSC_ACTIONS}}, 
				{ 	S2 => 	[{ R2 => 1, 
					OBJ_VAL1 => html('Do not send mail larger than:'), 
					OBJ_VAL2 => stripquotes(GetKey($obj, 'maxsize')),
					OBJ_VAL3 => html('KB') }] };

			# Other parameters ...
			push @{$tobj[$cnt]{RSC_ACTIONS}}, (
				{ 	SECTION => html('Allowed Characters'),
					S2 => 	[{ 	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'allowed_chars') eq '"8 bit"')?$MarkDot:$Mark,
								OBJ_VAL2 => html('8-bit') },
							 { 	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'allowed_chars') eq '"8 bit"')?$Mark:$MarkDot,
								OBJ_VAL2 => html('7-bit (no CTRL chars)') }] }
			);

			# Weeding ...
			push @{$tobj[$cnt]{RSC_ACTIONS}}, 
				{ 	SECTION => html('Weeding'),
					S2 => 	[{	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'smtp_strip_script_tags') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Strip SCRIPT Tags') },
							 {	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'smtp_strip_applet_tags') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Strip Applet Tags') },
							 {	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'smtp_strip_active_tags') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Strip ActiveX Tags') },
							 {	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'smtp_strip_ftp_tags') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Strip FTP Links') },
							 {	R1 => 1, 
								OBJ_VAL1 => (GetKey($obj, 'smtp_strip_port_tags') eq 'true')?$CheckTick:$Check,
								OBJ_VAL2 => html('Strip Port Strings') }] 
				};
		}

		# CVP Options in non-URI resources ...
		if (($type eq 'SMTP') || ($type eq 'FTP') || ($type eq 'TCP')) {
			if (GetKey($obj, 'av_use') eq 'true') {
				# CVP options ...
				my @section;
	
				%var = %{obj2html($rb_name, GetKey($obj, 'av_server:Table'), GetKey($obj, 'av_server:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('CVP server:'), OBJ_SET => $var };
				
				$var = (GetKey($obj, 'av_setting') eq 'cure')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('CVP server is allowed to modify content'), OBJ_SET => $var };
				if ($type eq 'SMTP') {
					$var = (GetKey($obj, 'av_headers') eq 'true')?$CheckTick:$Check;
					push @section, { R3 => 1, OBJ_VAR => html('Send SMTP headers to CVP server'), OBJ_SET => $var };
				}

				# Reply Order
				push @section, { H1 => 1, OBJ_VAR => html('Reply Order') };
				$var = (GetKey($obj, 'reply_mode') eq 'reply_first')?$MarkDot:$Mark;
				push @section, { R3 => 1, OBJ_VAR => html('Return data after content is approved'), OBJ_SET => $var };
				$var = (GetKey($obj, 'reply_mode') eq 'reply_first')?$Mark:$MarkDot;
				push @section, { R3 => 1, OBJ_VAR => html('Return data before content is approved'), OBJ_SET => $var };
				
				push @{$tobj[$cnt]{RSC_ACTIONS}}, { S0 => html('CVP Options'), S1 => \@section };
			}
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{RSC_ACTIONS}}) { push @{$tobj[$cnt]{RSC_ACTIONS}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENTS tab/column of Resource
		#------------------------------------------------
		$tobj[$cnt]{RSC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
			
	}
		
	$template->param(RESOURCES=>\@tobj);
	
}

##########################################################################
# Routine: servers2html
#
# Description:
# Routine to create the SERVERS table
##########################################################################
sub servers2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $spec);

	my (@var, %var, $var, $val);
	my (@tmp, $tmp, $opsec);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                    SERVERS to HTML                       ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{SRV_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{SRV_TEXT} = stripquotes($name);
		$tobj[$cnt]{SRV_NAME} = HTMLObjectList($rb_name, $obj, $name);

		#------------------------------------------------
		# General column/tab ...
		#------------------------------------------------
		if (($type eq 'RADIUS') || ($type eq 'TACACS') || ($type eq 'SecuRemote DNS')) {
			my @section;
			# Host ...
			%var = %{obj2html($rb_name, GetKey($obj, 'server:Table'), GetKey($obj, 'server:Name'), $_HREF)};
			push @section, {	R2 => 1, 
								OBJ_VAL1 => html('Host:'), 
								OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
		
			if ($type eq 'RADIUS') {
				# Service ...
				%var = %{obj2html($rb_name, GetKey($obj, 'service:Table'), GetKey($obj, 'service:Name'), $_HREF)};
				push @section, { 	R2 => 1, 
									OBJ_VAL1 => html('Service:'), 
									OBJ_VAL2 => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
	
				# Version & Priority ...
				$var = stripquotes(GetKey($obj, 'version'));
				push @section, {	R2 => 1, 
									OBJ_VAL1 => html('Version:'), 
									OBJ_VAL2 => ($var =~ /1\.0/)?'RADIUS Ver. 1.0 Compatible':'RADIUS Ver. 2.0 Compatible' };
				push @section, { 	R2 => 1, 
									OBJ_VAL1 => html('Priority:'), 
									OBJ_VAL2 => GetKey($obj, 'priority') };
			}
				 
			if ($type eq 'TACACS') {
				# Tacacs type ...
				$val = stripquotes(GetKey($obj, 'tacacs_server_type'));
				push @section, { 	R0 => 1, 
									OBJ_VAL1 => html('Type:'), 
									OBJ_VAL2 => (($val eq 'TACACS')?$MarkDot:$Mark).html(' TACACS  '),
									OBJ_VAL3 => (($val eq 'TACACS')?$Mark:$MarkDot).html(' TACACS +') };
	
				if ($val eq 'TACACS') {
					# Service ...
					%var = %{obj2html($rb_name, GetKey($obj, 'service:Table'), GetKey($obj, 'service:Name'), $_HREF)};
					push @section, { 	R2 => 1, 
										OBJ_VAR => html('Service:'), 
										OBJ_SET => $var{OBJ_ICON}.' '.$var{OBJ_TEXT} };
				}
			}
			push @{$tobj[$cnt]{SRV_GENERAL}}, { S2 => \@section };
		}


		if ($type eq 'RADIUS Group') {
			my @section;
			# Find Members ...
			@var = GetMembers($obj);
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 =>1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{SRV_GENERAL}}, { S0 => html('In Group:'), S1 => \@section }
		}

		if ($type eq 'LDAP Account Unit') {
			my @section;
			# Account Unit usage ...
			@var = GetMembers(GetKey($obj, 'ldap_usage'));
			%var = ();
			foreach $var (@var) { $var{$var} = 'true' }

			$val = ($var{'crl'})?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('CRL retrieval'), OBJ_SET => $val };
			$val = ($var{'user'})?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('User management'), OBJ_SET => $val };

			# Profile ...
			%var = %{obj2html($rb_name, GetKey($obj, 'ldap_policy:Table'), GetKey($obj, 'ldap_policy:Name'), $_HREF)};
			# No Icons for LDAP profile/policy ...
			$var = $var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Profile:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{SRV_GENERAL}}, { S0 => html('Account Unit usage'), S1 => \@section }
		}
		
		if ($type eq 'Certificate Authority') {
			my @section;
			# Certificate Authority type ...
			$var = stripquotes(GetKey($obj, 'ca_type'));
			if ($var eq 'internal') { $var = html('Local management server') }
			push @section, { R2 => 1, OBJ_VAR => html('Certificate Authority:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{SRV_GENERAL}}, { S1 => \@section }
		}
		
		# Check for empty cell and correct if necessary ...
		if (!defined $tobj[$cnt]{SRV_GENERAL}) { 
			push @{$tobj[$cnt]{SRV_GENERAL}}, { VOID => 1 }
		}

		#------------------------------------------------
		# OTHER Settings ...
		#------------------------------------------------
		if ($type eq 'SecuRemote DNS') {
			my @section;
			push @section, {	R0 => 1, 
								OBJ_VAL1 => html('Name'), 
								OBJ_VAL2 => html('Max. Label Count') };
			@var = GetMembers(GetKey($obj, 'domain'));
			foreach $var (@var) {
				push @section, { 	R1 => 1,
									OBJ_VAL1 => html(GetKey($var, 'domain')), 
									OBJ_VAL2 => GetKey($var, 'dns_label_count')};
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('Domain Patterns'), T1 => \@section, C2 => 1 };
		} 
		
		if ($type eq 'LDAP Account Unit') {
			# LDAP Servers
			my @section;
			my %server;
			push @section, { 	R0 => 1, 	
								OBJ_VAL1 => html('Host'), 
								OBJ_VAL2 => html('Port'), 
								OBJ_VAL3 => html('Priority') };
			@var = GetMembers(GetKey($obj, 'ldap_servers'));
			foreach $var (@var) {
				# LDAP Server ...
				%var = %{obj2html($rb_name, GetKey($var, 'server:Table'), GetKey($var, 'server:Name'), $_HREF)};
				$tmp = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { 	R1 => 1,
									OBJ_VAL1 => $tmp,
									OBJ_VAL2 => GetKey($var, 'ldap_ssl_port'), 
									OBJ_VAL3 => GetKey($var, 'priority') };
				$server{GetKey($var, 'AdminInfo:chkpf_uid')}{Name}  = GetKey($var, 'server:Name');
				$server{GetKey($var, 'AdminInfo:chkpf_uid')}{Table} = GetKey($var, 'server:Table');
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('LDAP servers'), T1 => \@section, C3 => 1 };

			# Early version compatibility server
			my @section;
			$tmp = GetKey($obj, 'bc_server');
			foreach $val (keys %server) {
				if ($val eq $tmp) {
					%var = %{obj2html($rb_name, $server{$val}{Table}, $server{$val}{Name}, $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R2=> 1, OBJ_VAR => html('Early Versions server:'), OBJ_SET => $var };
				}
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S1 => \@section };
						
			# Objects Management
			my @section;
			$tmp = GetKey($obj, 'editable_server');
			foreach $val (keys %server) {
				if ($val eq $tmp) {
					%var = %{obj2html($rb_name, $server{$val}{Table}, $server{$val}{Name}, $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R2=> 1, OBJ_VAR => html('Manage objects on:'), OBJ_SET => $var };
				}
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('Object Management'), S1 => \@section };

			# Branches ...
			my @section;
			@var = GetMembers(GetKey($obj, 'branches'));
			if (@var) {
				foreach $var (@var) {
					push @section, { R1 => 1, OBJ_VAR => html(stripquotes($var)) }
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('Branches in use'), S1 => \@section };
		} 
		
		if ($type eq 'Certificate Authority') {
			# Certificate Authority CRL Retrieval ...
			my @section;
			$var = (GetKey($obj, 'crl_ldap') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('LDAP Server(s)'), OBJ_SET => $var };
			$var = (GetKey($obj, 'crl_http') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('HTTP Server(s)'), OBJ_SET => $var };

			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('Retrieve CRL From:'), S1 => \@section };

			# Certificate Authority CRL Caching ...
			my @section;
			if (GetKey($obj, 'crl_cache_type') eq 'exp_Date') {
				push @section, { R3 => 1, OBJ_VAR => html('Cache CRL on the module'), OBJ_SET => $CheckTick };
				push @section, { R3 => 1, OBJ_VAR => html('Fetch new CRL when expires') }
			} elsif (GetKey($obj, 'crl_cache_type') eq 'Timeout') {
				push @section, { R3 => 1, OBJ_VAR => html('Cache CRL on the module'), OBJ_SET => $CheckTick };
				$var = html(GetKey($obj, 'crl_cache_timeout').' seconds');
				push @section, { R2 => 1, OBJ_VAR => html('Fetch new CRL after:'), OBJ_SET => $var }
			} else { 
				# No CRL caching ...
				push @section, { R1 => 1, OBJ_VAR => html('Do not cache CRL on the module'), OBJ_SET => $Check };
			}
			push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('CRL Caching'), S1 => \@section };

			# Certificate Authority
			if (GetKey($obj, 'permissions_type') eq 'accept_only') {
				my @section;
				push @section, { R3 => 1, OBJ_VAR => html('Allow only certificates from the listed branches'), OBJ_SET => $CheckTick };
				@var = GetMembers(GetKey($obj, 'permissions_strings'));
				if (@var) {
					foreach $var (@var) {
						push @section, { R1 => 1, OBJ_VAR => html(stripquotes($var)) }
					}
				} else {
					push @section, { R2 => 1, OBJ_VAR => $None };
				}
					
				push @{$tobj[$cnt]{SRV_OTHER}}, { S0 => html('Certificate Authority'), S1 => \@section };
			}
		}
		
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{SRV_OTHER}}) { push @{$tobj[$cnt]{SRV_OTHER}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENTS tab/column of Resource
		#------------------------------------------------
		$tobj[$cnt]{SRV_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
			
	}
		
	$template->param(SERVERS=>\@tobj);
}


##########################################################################
# Routine: opsec2html
#
# Description:
# Routine to create the OPSEC Application table
##########################################################################
sub opsec2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type, $spec);

	my (@var, %var, $var, $val);
	my (@tmp, $tmp, $opsec);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---              OPSEC Applications to HTML                  ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);
		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{OPSEC_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{OPSEC_TEXT} = stripquotes($name);
		$tobj[$cnt]{OPSEC_NAME} = HTMLObjectList($rb_name, $obj, $name);

		#------------------------------------------------
		# General column/tab
		#------------------------------------------------
		if ($type eq 'OPSEC Application') {
			my @section;
			# Host ...
			%var = %{obj2html($rb_name, GetKey($obj, 'host:Table'), GetKey($obj, 'host:Name'), $_HREF)};
			$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
			push @section, { R2 => 1, OBJ_VAR => html('Host:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{OPSEC_GENERAL}}, { S1 => \@section };

			# Application properties ...
			my @section;
			# Vendor, Product & Version ...
			$var = GetKey(GetKey($obj, 'product:Table').':'.GetKey($obj, 'product:Name'));
			push @section, { R2 => 1, OBJ_VAR => html('Vendor:'),  OBJ_SET => html(stripquotes(GetKey($var, 'product_info:vendor'))) };
			$opsec = $obj;
			if (stripquotes(GetKey($var, 'product_info:vendor')) ne 'User defined') {
				push @section, { R2 => 1, OBJ_VAR => html('Product:'), OBJ_SET => html(stripquotes(GetKey($var, 'product_info:product'))) };
				push @section, { R2 => 1, OBJ_VAR => html('Version:'), OBJ_SET => html(stripquotes(GetKey($var, 'product_info:version'))) };
				$opsec = $var;
			}

			push @{$tobj[$cnt]{OPSEC_GENERAL}}, { S0 => html('Application properties'), S1 => \@section };

			# Server and Client entities special table format ...
			my @section;
			push @section, { H1 => 1, OBJ_VAL1 => html('Server Entities'), OBJ_VAL2 => html('Client Entities') };
			
			$var = (GetKey($opsec, 'CVP') eq 'true')?$CheckTick:$Check;
			$val = (GetKey($opsec, 'ELA') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL1 => $var.html(' CVP'), OBJ_VAL2 => $val.html(' ELA') };
			$var = (GetKey($opsec, 'UFP') eq 'true')?$CheckTick:$Check;
			$val = (GetKey($opsec, 'LEA') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL1 => $var.html(' UFP'), OBJ_VAL2 => $val.html(' LEA') };
			$var = (GetKey($opsec, 'AMON') eq 'true')?$CheckTick:$Check;
			$val = (GetKey($opsec, 'SAM') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL1 => $var.html(' AMON'), OBJ_VAL2 => $val.html(' SAM') };
			$val = (GetKey($opsec, 'CPMI') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL2 => $val.html(' CPMI') };
			$val = (GetKey($opsec, 'OMI') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL2 => $val.html(' OMI') };
			$val = (GetKey($opsec, 'UAC') eq 'true')?$CheckTick:$Check;
			push @section, { R0 => 1, OBJ_VAL2 => $val.html(' UAA') };
			
			push @{$tobj[$cnt]{OPSEC_GENERAL}}, { S2 => \@section };

		} elsif ($type =~ /Group/) {
			# Group Members ...
			my @section;
			@var = GetMembers(GetKey($obj, 'members'));
			foreach $var (@var) {
				%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R1 => 1, OBJ_VAR => $var };
			}
			if (defined @section) { push @{$tobj[$cnt]{OPSEC_GENERAL}}, { S0 => html('In Group:'), S1 => \@section }	}
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OPSEC_GENERAL}}) { push @{$tobj[$cnt]{OPSEC_GENERAL}}, { VOID => 1 } }
		
		#------------------------------------------------
		# Options column
		#------------------------------------------------
		if ($type eq 'OPSEC Application') {
			if (GetKey($opsec, 'CVP') eq 'true') {
				# CVP Options ...
				my @section;
				# Service ...
				%var = %{obj2html($rb_name, GetKey($obj, 'CVP_settings:service:Table'), GetKey($obj, 'CVP_settings:service:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('Service:'), OBJ_SET => $var };
				
				# Early version compatibility mode
				$val = GetKey($obj, 'CVP_settings:backwards_compatibility_security_method');
				push @section, { R3 => 1, OBJ_VAR => html('Use early version compatibility mode'), OBJ_SET => $val?$CheckTick:$Check };
				if ($val) {
					push @section, { R3 => 1, 	OBJ_VAR => html('Clear (opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC Authentication (auth_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'auth_opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC SSL (ssl_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'ssl_opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC SSL Clear (ssl_clear_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'ssl_clear_opsec')?$MarkDot:$Mark) };
				}
				
				push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('CVP Options'), S1 => \@section };
			}
			
			if (GetKey($opsec, 'UFP') eq 'true') {
				# UFP Options ...

				# Service ...
				my @section;
				%var = %{obj2html($rb_name, GetKey($obj, 'UFP_settings:service:Table'), GetKey($obj, 'UFP_settings:service:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('Service:'), OBJ_SET => $var };

				# Early version compatibility mode
				$val = GetKey($obj, 'UFP_settings:backwards_compatibility_security_method');
				push @section, { R3 => 1, OBJ_VAR => html('Use early version compatibility mode'), OBJ_SET => $val?$CheckTick:$Check };
				if ($val) {
					push @section, { R3 => 1, 	OBJ_VAR => html('Clear (opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC Authentication (auth_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'auth_opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC SSL (ssl_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'ssl_opsec')?$MarkDot:$Mark) };
					push @section, { R3 => 1, 	OBJ_VAR => html('OPSEC SSL Clear (ssl_clear_opsec)'), 
												OBJ_SET => $Spacer.(($val eq 'ssl_clear_opsec')?$MarkDot:$Mark) };
				}
				push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('UFP Options'), S1 => \@section };

				# Dictionary ...
				my @section;
				$var = GetKey($obj, 'UFP_settings:description');
				push @section, { R2 => 1, OBJ_VAR => html('Description:'), OBJ_SET => html(stripquotes($var)) };

				# UFP Categories ...
				@var = GetMembers(GetKey($obj, 'UFP_settings:categories'));
				@tmp = ();
				foreach $var (@var) { push @tmp, stripquotes(GetKey($var, 'name')) }
				
				push @section, { H1 => 1, OBJ_VAR => html('Categories in Dictionary') };
				if (@tmp) {
					foreach $var (sort @tmp) {
						push @section, { R1 => 1, OBJ_VAR => html($var) };
					}
				} else {
					push @section, { R2 => 1, OBJ_VAR => $None };
				}
				push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('Dictionary'), S1 => \@section };

			}

			if (GetKey($opsec, 'AMON') eq 'true') {
				# CVP Options ...
				my @section;
				# Service ...
				%var = %{obj2html($rb_name, GetKey($obj, 'AMON_settings:service:Table'), GetKey($obj, 'AMON_settings:service:Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1, OBJ_VAR => html('Service:'), OBJ_SET => $var };
				
				# Identifier ...
				$var = stripquotes(GetKey($obj, 'AMON_class_name'));
				push @section, { R2 => 1, OBJ_VAR => html('Identifier:'), OBJ_SET => html($var) };

				push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('AMON Options'), S1 => \@section };
			}
		
		} elsif ($type eq 'CVP Group') {
			# CVP Group options ...
			my @section;
			$var = stripquotes(GetKey($obj, 'method'));
			push @section, { R4 => 1, 	OBJ_VAR => (($var eq 'load sharing')?$MarkDot:$Mark).html(' Load sharing'), 
										OBJ_SET => (($var eq 'load sharing')?$Mark:$MarkDot).html(' Chaining') };
			if ($var eq 'load sharing') {
				$var = stripquotes(GetKey($obj, 'load_sharing_method'));
				push @section, { R2 => 1, OBJ_VAR => html('Load sharing method:'), OBJ_SET => $var };
				$var = stripquotes(GetKey($obj, 'load_sharing_suspend_timeout')).' minutes';
				push @section, { R2 => 1, OBJ_VAR => html('Load sharing suspend timeout:'), OBJ_SET => html($var) };
			} else {
				$var = stripquotes(GetKey($obj, 'drop_on_unsafe'));
				push @section, { R3 => 1, OBJ_VAR => html('Abort chaining upon Unsafe reply'), OBJ_SET => ($var eq 'true')?$CheckTick:$Check };
			}
			push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('Work distribution method'), S1 => \@section };

		} elsif ($type eq 'UFP Group') {
			# UFP Group options ...
			my @section;
			$var = stripquotes(GetKey($obj, 'load_sharing_method'));
			push @section, { R2 => 1, OBJ_VAR => html('Load sharing method:'), OBJ_SET => $var };
			$var = stripquotes(GetKey($obj, 'load_sharing_suspend_timeout')).' minutes';
			push @section, { R2 => 1, OBJ_VAR => html('Load sharing suspend timeout:'), OBJ_SET => html($var) };

			push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { S0 => html('Work distribution method'), S1 => \@section };

		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{OPSEC_OPTIONS}}) { push @{$tobj[$cnt]{OPSEC_OPTIONS}}, { VOID => 1 } }

		#------------------------------------------------
		# COMMENTS tab/column of Resource
		#------------------------------------------------
		$tobj[$cnt]{OPSEC_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(OPSEC=>\@tobj);
}


##########################################################################
# Routine: times2html
#
# Description:
# Routine to create the TIME object table
##########################################################################
sub	times2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($type, $current_type);
	my (@var, %var, $var, $val);
	my (@tmp, $tmp);
	my ($h1, $m1, $h2, $m2, $h, $m, $s);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                  Time objects to HTML                    ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{TIME_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{TIME_TEXT} = stripquotes($name);
		$tobj[$cnt]{TIME_NAME} = HTMLObjectList($rb_name, $obj, $name);

		#------------------------------------------------
		# General column/tab
		#------------------------------------------------
		if ($type eq 'Time') {
			my @section;
			if (GetKey($obj, 'from_hour1')||GetKey($obj, 'from_minute1')||GetKey($obj, 'to_hour1')||GetKey($obj, 'to_minute1')) {
				$h1 = sprintf("%02d", GetKey($obj, 'from_hour1'));
				$m1 = sprintf("%02d", GetKey($obj, 'from_minute1'));
				$h2 = sprintf("%02d", GetKey($obj, 'to_hour1'));
				$m2 = sprintf("%02d", GetKey($obj, 'to_minute1'));
				push @section, {	R24 => 1, 
									OBJ_VAL1 => html('From:'),
									OBJ_VAL2 => $h1.':'.$m1,
									OBJ_VAL3 => html('To:'),
									OBJ_VAL4 => $h2.':'.$m2 };
			}
			if (GetKey($obj, 'from_hour2')||GetKey($obj, 'from_minute2')||GetKey($obj, 'to_hour2')||GetKey($obj, 'to_minute2')) {
				$h1 = sprintf("%02d", GetKey($obj, 'from_hour2'));
				$m1 = sprintf("%02d", GetKey($obj, 'from_minute2'));
				$h2 = sprintf("%02d", GetKey($obj, 'to_hour2'));
				$m2 = sprintf("%02d", GetKey($obj, 'to_minute2'));
				push @section, {	R24 => 1, 
									OBJ_VAL1 => html('From:'),
									OBJ_VAL2 => $h1.':'.$m1,
									OBJ_VAL3 => html('To:'),
									OBJ_VAL4 => $h2.':'.$m2 };
			}
			if (GetKey($obj, 'from_hour3')||GetKey($obj, 'from_minute3')||GetKey($obj, 'to_hour3')||GetKey($obj, 'to_minute3')) {
				$h1 = sprintf("%02d", GetKey($obj, 'from_hour3'));
				$m1 = sprintf("%02d", GetKey($obj, 'from_minute3'));
				$h2 = sprintf("%02d", GetKey($obj, 'to_hour3'));
				$m2 = sprintf("%02d", GetKey($obj, 'to_minute3'));
				push @section, {	R24 => 1, 
									OBJ_VAL1 => html('From:'),
									OBJ_VAL2 => $h1.':'.$m1,
									OBJ_VAL3 => html('To:'),
									OBJ_VAL4 => $h2.':'.$m2 };
			}
			push @{$tobj[$cnt]{TIME_GENERAL}}, { S0 => html('Time of day (hh:mm)'), S2 => \@section };

		} elsif ($type eq 'Scheduled Event') {
			my @section;
			if (GetKey($obj, 'days_specification') ne 'seconds') {
				$var = GetKey($obj, 'hour').':'.GetKey($obj, 'minutes');
				push @section, { R2 => 1, OBJ_VAR => html('At (hh:mm):'), OBJ_SET => $var };
			} else {
				$var = GetKey($obj, 'every_seconds');
				$h=sprintf("%02d", int($var/3600));
				$m=sprintf("%02d", int(($var-$h*3600)/60));
				$s=sprintf("%02d", ($var-$h*3600-$m*60));
				$var=$h.'h '.$m.'m '.$s.'s';
				push @section, { R2 => 1, OBJ_VAR => html('Every:'), OBJ_SET => $var };
			}
			push @{$tobj[$cnt]{TIME_GENERAL}}, { S0 => html('Time of Event'), S1 => \@section };

		} elsif ($type =~ /Group/) {
			my @section;
			# Group Members ...
			@var = GetMembers($obj);
			$val = 'Members:';
			foreach $var (@var) {
				%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
				$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
				push @section, { R2 => 1,  OBJ_VAR => $val, OBJ_SET => $var };
				$val = '';
			}
			push @{$tobj[$cnt]{TIME_GENERAL}}, { S1 => \@section };
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{TIME_GENERAL}}) { push @{$tobj[$cnt]{TIME_GENERAL}}, { VOID => 1 } }
		
		#------------------------------------------------
		# DAYS tab/column
		#------------------------------------------------
		if ($type eq 'Time') {
			my @section;
			$val = stripquotes(GetKey($obj, 'days_specification'));
			push @section, { R0 => 1,
								OBJ_VAL1 => (($val eq 'none')?$MarkDot:$Mark).' None',
								OBJ_VAL2 => (($val eq 'by day in month')?$MarkDot:$Mark).' Day in month',
								OBJ_VAL3 => (($val eq 'by day in week')?$MarkDot:$Mark).' Day in week' };
								
			push @{$tobj[$cnt]{TIME_DAYS}}, { S0 => html('Days specification'), S2 => \@section };
			
			if ($val eq 'by day in month') {
				my @section;
				$var = join(', ', GetMembers(GetKey($obj, 'day_of_month')));
				if ($var) {
					push @section, { R2 => 1, OBJ_VAR => html('Days:'), OBJ_SET => $var };
				}
				$var = join(', ', GetMembers(GetKey($obj, 'month')));
				if ($var) {
					push @section, { R2 => 1, OBJ_VAR => html('Month:'), OBJ_SET => $Month{$var} };
				} else {
					push @section, { R2 => 1, OBJ_VAR => html('Month:'), OBJ_SET => html('Any') };
				}
				push @{$tobj[$cnt]{TIME_DAYS}}, { S0 => html('Days in month'), S1 => \@section };
				
			} elsif ($val eq 'by day in week') {
				my @section;
				$var = join(', ', GetMembers(GetKey($obj, 'day_of_week')));
				if ($var) {
					push @section, { R2 => 1, OBJ_VAR => html('Days:'), OBJ_SET => $var };
				}
				push @{$tobj[$cnt]{TIME_DAYS}}, { S0 => html('Days in week'), S1 => \@section };
				
			}
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{TIME_DAYS}}) { push @{$tobj[$cnt]{TIME_DAYS}}, { VOID => 1 } }
		

		#------------------------------------------------
		# COMMENTS tab/column of Resource
		#------------------------------------------------
		$tobj[$cnt]{TIME_COMMENTS} = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
	}
		
	$template->param(TIMES=>\@tobj);
}

##########################################################################
# Routine: communities2html
#
# Description:
# Routine to create the VPN Communities table
##########################################################################
sub communities2html {
	use strict;
	my (@ObjList, @tobj);
	my ($cnt, $obj, $rb_name, $name);
	my ($topo, $type, $current_type);
	my (@var, %var, $var, $val);
	my (@tmp);

	my $rulebase = shift;
	my $template = shift;
	my $CPtable  = shift;
	my $obj      = shift;
	my @ObjList  = @{$obj};
	
	# Determine rulebase name ...
	$rb_name       = GetKey($rulebase, "collection:Name");
	
	#----------------------------------------------------------------
	#---                 COMMUNITIES to HTML                      ---
	#----------------------------------------------------------------
	$cnt = -1;
	$current_type = '';
	foreach $name (sort {uc($a) cmp uc($b)} @ObjList) {
		$cnt += 1;
		(undef, $type, $name) = split(/:/, $name);
		$obj  = GetKey($CPtable.':'.$name);

		if ($type ne $current_type) {
			$tobj[$cnt]{TYPE} = 1;
			$tobj[$cnt]{COMM_TYPE} = $type;
			$current_type = $type;
		}

		$tobj[$cnt]{COMM_TEXT} = stripquotes($name);
		$tobj[$cnt]{COMM_TYPE} = $type;
		$tobj[$cnt]{COMM_NAME} = HTMLObjectList($rb_name, $obj, $name);
		
		#------------------------------------------------
		# GATEWAYS tab/column of Community
		#------------------------------------------------
		# Participating Gateways ...
		@var = GetMembers(GetKey($obj, 'participant_gateways'));
		if (($type eq 'Site to Site - Meshed') || ($type eq 'Remote Access')) {
			my @section;
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_GATEWAYS}}, { S0 => html('Participating Gateways'), S1 => \@section };
			
		} elsif ($type eq 'Site to Site - Star') {
			my @section;
			if (GetKey($obj, 'meshed_in_center') eq 'true') {
				push @section, { R1 => 1, OBJ_VAR => html('* Meshed in center') };
			} else {
				push @section, { R1 => 1, OBJ_VAR => html('* Not meshed in center') };
			}
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_GATEWAYS}}, { S0 => html('Central Gateways'), S1 => \@section };
		}

		# Satellite Gateways ...
		if ($type eq 'Site to Site - Star') {
			my @section;
			@var = GetMembers(GetKey($obj, 'satellite_gateways'));
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_GATEWAYS}}, { S0 => html('Satellite Gateways'), S1 => \@section };
		}

		# Extranet Partners ...
		if ($type eq 'Extranet Manager') {
			my @section;
			@var = GetMembers(GetKey($obj, 'extranet_partners'));
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_GATEWAYS}}, { S0 => html('Extranet Partners'), S1 => \@section };
		}

		# Participating User Groups ...
		if ($type eq 'Remote Access') {
			my @section;
			@var = GetMembers(GetKey($obj, 'participant_users_groups'));
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_GATEWAYS}}, { S0 => html('User Groups'), S1 => \@section };
		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{COMM_GATEWAYS}}) { push @{$tobj[$cnt]{COMM_GATEWAYS}}, { VOID => 1 } }

		#------------------------------------------------
		# PROPS tab/column of Community
		#------------------------------------------------
		if ($type =~ /Site to Site/) {
			# Community Traffic Security Policy
			my @section;
			$var = (GetKey($obj, 'allow_all_encrypted_traffic') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => ('Accept all encrypted traffic'), OBJ_SET => $var };

			push @{$tobj[$cnt]{COMM_PROPS}}, { S0 => html('Community Traffic Security Policy'), S1 => \@section };

			# Services in the clear
			@var = GetMembers(GetKey($obj, 'exclude_srv'));
			my @section;
			if (@var) {
				foreach $var (@var) {
					%var = %{obj2html($rb_name, GetKey($var, 'Table'), GetKey($var, 'Name'), $_HREF)};
					$var = $var{OBJ_ICON}.' '.$var{OBJ_TEXT};
					push @section, { R1 => 1, OBJ_VAR => $var };
				}
			} else {
				push @section, { R2 => 1, OBJ_VAR => $None };
			}
			push @{$tobj[$cnt]{COMM_PROPS}}, { S0 => html('Excluded Services'), S1 => \@section };
		}
		
		# VPN Settings phase 1
		if (($type =~ /Site to Site/) || ($type eq 'Extranet Partner')) {
			my @section;
			$var = html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_enc_alg')));
			push @section, { R2 => 1, OBJ_VAR => html('Perform key exchange encryption with:'), OBJ_SET => $var };
			$var = html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_hash_alg')));
			push @section, { R2 => 1, OBJ_VAR => html('Perform data integrity with:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{COMM_PROPS}}, { S0 => html('IKE (Phase 1) Properties'), S1 => \@section };
		}
		# VPN Settings phase 2
		if ($type =~ /Site to Site/) {
			my @section;
			$var = html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_enc_alg')));
			push @section, { R2 => 1, OBJ_VAR => html('Perform IPSEC data encryption with:'), OBJ_SET => $var };
			$var = html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_hash_alg')));
			push @section, { R2 => 1, OBJ_VAR => html('Perform data integrity with:'), OBJ_SET => $var };

			push @{$tobj[$cnt]{COMM_PROPS}}, { S0 => html('IPSEC (Phase 2) Properties'), S1 => \@section };
		}

		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{COMM_PROPS}}) { push @{$tobj[$cnt]{COMM_PROPS}}, { VOID => 1 } }

		#------------------------------------------------
		# ADVANCED tab/column of Community
		#------------------------------------------------
		# Advanced VPN Settings phase 1
		if (($type =~ /Site to Site/) || ($type eq 'Extranet Partner')) {
			my @section;
			$var = html(stripquotes(GetKey($obj, 'ike_p1:ike_p1_dh_grp:Name')));
			push @section, { R2 => 1, OBJ_VAR => html('Use Diffie-Hellman group:'), OBJ_SET => $var };
			$var = html(GetKey($obj, 'ike_p1:ike_p1_rekey_time').' minutes');
			push @section, { R2 => 1, OBJ_VAR => html('Renegotiate IKE SA every:'), OBJ_SET => $var };

			if ($type =~ /Site to Site/) {
				$var = (GetKey($obj, 'ike_p1:ike_p1_use_aggressive') eq 'true')?$CheckTick:$Check;
				push @section, { R3 => 1, OBJ_VAR => html('Use aggressive mode'), OBJ_SET => $var };
			}

			push @{$tobj[$cnt]{COMM_ADVANCED}}, { S0 => html('IKE (Phase 1)'), S1 => \@section };
		}
		# Advanced VPN Settings phase 2
		if ($type =~ /Site to Site/) {
			my @section;
			$var = (GetKey($obj, 'ike_p2:ike_p2_use_pfs') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Use Perfect Forward Secrecy'), OBJ_SET => $var };
			if (GetKey($obj, 'ike_p2:ike_p2_use_pfs') eq 'true') {
				$var = html(stripquotes(GetKey($obj, 'ike_p2:ike_p2_pfs_dh_grp:Name')));
				push @section, { R2 => 1, OBJ_VAR => html('DH group:'), OBJ_SET => $var };
			}
			$var = html(GetKey($obj, 'ike_p2:ike_p2_rekey_time').' seconds');
			push @section, { R2 => 1, OBJ_VAR => html('Renegotiate IPSEC SA every:'), OBJ_SET => $var };

			$var = (GetKey($obj, 'ike_p2:ike_p2_ipcomp') eq 'DEFLATE')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Support Site to Site IP compression'), OBJ_SET => $var };

			push @{$tobj[$cnt]{COMM_ADVANCED}}, { S0 => html('IPSEC (Phase 2)'), S1 => \@section };
		}
		# Advanced Properties NAT
		if ($type =~ /Site to Site/) {
			my @section;
			$var = (GetKey($obj, 'disable_NAT') eq 'true')?$CheckTick:$Check;
			push @section, { R3 => 1, OBJ_VAR => html('Disable NAT inside the VPN community'), OBJ_SET => $var };

			push @{$tobj[$cnt]{COMM_ADVANCED}}, { S0 => html('NAT'), S1 => \@section };
		}
		# Check for empty cell and correct if necessary ...
		if (!defined @{$tobj[$cnt]{COMM_ADVANCED}}) { push @{$tobj[$cnt]{COMM_ADVANCED}}, { VOID => 1 } }
		
		#------------------------------------------------
		# COMMENTS tab/column of Community
		#------------------------------------------------
		$tobj[$cnt]{COMM_COMMENTS}   = text2html(stripquotes(GetKey($obj, 'comments')), $_MAX_COMMENT_LENGTH);
			
	}
		
	$template->param(COMMUNITIES=>\@tobj);
}

sub users2html {
	print "users2html\n";
}

sub usergroups2html {
	print "usergroups2html\n";
}

sub usertemplates2html {
	print "usertemplates2html\n";
}

1;
