#!/usr/bin/perl
##########################################################################
# Package: Options
#
# Description:
# This package holds the routines to read the options specified,
# and to get the values of those options. The options data is 
# stored in the package itself and is not available outside it.
# Therefore you need to use the function: Option
#
# Routines:
# ReadOptions	Reads initialy the .ini file and the commandline
# Option		Returns the value of the option specified
# Page			Returns the page setup as defined in in CPpages.def
# Object2Page	Returns the page an object type is in
##########################################################################
package Options;

use strict;
use Strings;
use Constants;

our (@ISA, @EXPORT);
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(ReadOptions Option Page Rulebase Object2Page);

##########################################################################
# Define global (to this package) variables
my %Option;		# Hash to store the options from the .ini file or command line
my %Page;		# Hash to store the tables/sections per Rulebase per Page definition 
my %Type2Table;	# Hash to store the conversion from CheckPoint table:type to
				# html table/section
my %Table2Page;	# Hash to store per rulebase, the conversion from html table/section 
				# to html page it is in per rulebase
my %Rulebases;	# Hash to store if a rulebase should be converted to HTML

				
# Define default values ...
%Option = (
		FW1RULES	    =>	'rulebases_5_0.fws',
		FW1OBJECTS	    =>	'objects_5_0.C',
		TMPLPATH	    =>	'', # searched for in current and working dir ...
		HTMLPATH	    =>	'./html',
		
		# The following paths are hardcoded in the resuling HTML pages and
		# are therefore relative to the HTML directory of the rulebase!!!
		# i.e. (normallly) $Option{HTMLPATH}/<rulebase_name> ...
		ICONPATH	    =>	'../icons',
		CSSPATH			=>	'../css',
		SCRIPTPATH		=>	'../scripts',
		
		MAINLOGO        =>  '',
		DEBUG		    =>	0
	);
	
$Option{DEFAULT} = {
		LOGO		    =>	'',
		ALLOBJECTS  	=>	0,
		GLOBALPROPS     =>	1
	};



##########################################################################
# Routine: ReadOptions
#
# Description:
# This routine reads the options from the .INI file and from
# the command line. The data is stored in the global hash: %Option
#
# Parameters:
# $INIfile	- name of the .INI file
# $Pagefile	- name of the page layout file
# 
# Returns:
# nothing; data is stored in the global hash %Option
sub ReadOptions {
	use strict;
	my $INIFile  = shift;
	my $Pagefile = shift;
	my $working_dir = shift;

	# Define default template path by searching
	# the current and working directory ...
	if (-d './Templates') {
		$Option{TMPLPATH} = './Templates'
	} elsif (-d $working_dir.'/Templates') {
		$Option{TMPLPATH} = $working_dir.'/Templates'
	}

	# Read Ini file ...
	if (-f $INIFile) {
		&ReadINIfile($INIFile);
	} elsif (-f $working_dir.'/'.$INIFile) {
		&ReadINIfile($working_dir.'/'.$INIFile);
	} else {
		# No .INI file found ...
	}

	# Read commandline parameters ...
	&ReadCommandLine;
  
	# Read Page layout file ...
	if (-f $Pagefile) {
		&ReadPagefile($Pagefile);
	} elsif (-f $working_dir.'/'.$Pagefile) {
		&ReadPagefile($working_dir.'/'.$Pagefile);
	} else {
		# No Page layout file found ...
		die "No page layout file found: $Pagefile!!!\n\n";
	}

	# Initialize search variables
	&Initialize;
}

##########################################################################
# Routine: ReadINIfile
#
# Description:
# Reads the ini file and store the found options in the hash %Option
#
# Parameters:
# $INIfile	the name of the .INI file
#
# Returns:
# nothing; data is stored in the global hash %Option
sub ReadINIfile {
	use strict;
	my $INIFile = shift;
	my ($section, $option, $rulebase);

	open (INFILE, $INIFile)
	or die "Cannot open the .INI file $INIFile!\n\n";
	
	while (<INFILE>) {
		if (/^\[(.*)\]/) {
			$section=uc(strip($1));
		} elsif (/^\s*[;!#]/) {
			# Commentline, skip
		} elsif (/^\s*(\S+)\s*=(.*)$/) {
			if (!$section) {
				$option   = uc($1);
				if ($option eq 'INCLUDE') {
					$rulebase = uc(strip($2));
					if (!defined($Option{RULEBASES}{$rulebase})) { $Option{RULEBASES}{$rulebase} = 1 }
					if (!defined($Option{RULEBASES}{_DEFAULT_})) { $Option{RULEBASES}{_DEFAULT_} = 0 }
				} elsif ($option eq 'EXCLUDE') {
					$rulebase = uc(strip($2));
					if (!defined($Option{RULEBASES}{$rulebase})) { $Option{RULEBASES}{$rulebase} = 0 }
					if (!defined($Option{RULEBASES}{_DEFAULT_})) { $Option{RULEBASES}{_DEFAULT_} = 1 }
				} else {
					$Option{$option}=strip($2);
				}
			} else {
				$Option{$section}{uc($1)}=strip($2);
			}            
		}
	}
	if (!defined($Option{RULEBASES}{_DEFAULT_})) { $Option{RULEBASES}{_DEFAULT_} = 1 }

	close (INFILE);
}

##########################################################################
# Routine: ReadPagefile
#
# Description:
# Reads the page definition file and stores it in the hash %Page
#
# Parameters:
# $Pagefile	the name of the page layout file
#
# Returns:
# nothing; data is stored in the global hash %Page
sub ReadPagefile {
	use strict;
	my $Pagefile = shift;
	my $header;
	my ($Page, $Table);
	my ($CPtable, $key);
	
	open (INFILE, $Pagefile)
	or die "Cannot open the definition file $Pagefile!\n\n";

	$header='DEFAULT';
	while (<INFILE>) {
		if (/^\[(.*)\]/) {
			$header=uc(strip($1));
		} elsif (/^\s*[;!#]/) {
			# Commentline, skip
		} elsif (/^\s*(\S+)\s*=(.*)$/) {
			if (uc($1) eq 'PAGE')  { $Page=strip($2);  push @{$Page{$header}{PAGES}}, $Page }
			if (uc($1) eq 'TABLE') { $Table=strip($2); push @{$Page{$header}{uc($Page)}{TABLES}}, $Table }
		}
	}
	close (INFILE);
}


##########################################################################
# Routine: ReadCommandLine
#
# Description:
# Scans the commandline for options and stores them in the hash %Option
#
# Parameters:
# (none)
#
# Returns:
# nothing; data is stored in the global hash %Option
sub ReadCommandLine {
  use strict;
  my ($i, $arg);
  my $args=$#ARGV;
  
  $i=0;
  while ($i<=$args) {
	$arg="\L$ARGV[$i]";
	if (($arg eq "-t") ||
	 	($arg eq "--tmplpath")) {	    	# TmplPath=
		$i++;
		$Option{TMPLPATH}=$ARGV[$i];
	  
	} elsif (($arg eq "-h") ||
			 ($arg eq "--htmlpath")) {	    # HTMLPath=
		$i++;
		$Option{HTMLPATH}=$ARGV[$i];
	  
	} elsif (($arg eq "-i") || 
			 ($arg eq "--iconpath")) {		# IconPath=
		$i++;
		$Option{ICONPATH}=$ARGV[$i];
	  
	} elsif (($arg eq "-s") || 
			 ($arg eq "--scriptpath")) {	# ScriptPath=
		$i++;
		$Option{ICONPATH}=$ARGV[$i];
	  
	} elsif (($arg eq "-c") || 
			 ($arg eq "--csspath")) {		# CSSPath=
		$i++;
		$Option{ICONPATH}=$ARGV[$i];
	  
	} elsif (($arg eq "-m") || 
			 ($arg eq "--mainlogo")) {		# MainLogo=
		$i++;
		$Option{MAINLOGO}=$ARGV[$i];
	  
	} elsif ($arg eq "\L--Debug") {	        # Debug=
		$Option{DEBUG}=1;

	} elsif (($arg eq "-r") ||
			 ($arg eq "--fw1rules")) {	    # FW1Rules=
		$i++;
		$Option{FW1RULES}=$ARGV[$i];
	  
	} elsif (($arg eq "-o") || 
			 ($arg eq "--fw1objects")) {	# FW1Objects=
		$i++;
		$Option{FW1OBJECTS}=$ARGV[$i];
	  
	} elsif (($arg eq "-b") ||
			 ($arg eq "--rulebase")) {      # Rulebase=
		$i++;
		$Option{RULEBASES}={};
		$Option{RULEBASES}{uc($ARGV[$i])}=1;
		$Option{RULEBASES}{_DEFAULT_}=0;
		
	} elsif (($arg eq "-l") ||
			 ($arg eq "--logo")) {	        # Logo=
		$i++;
		$Option{DEFAULT}{LOGO}=$ARGV[$i];

	} elsif (($arg eq "\L-AllObjects") ||
 			 ($arg eq "\L--AllObjects")) {	# AllObjects=0
		$Option{DEFAULT}{ALLOBJECTS}=1;

	} elsif (($arg eq "\L-NoGlobalProps") ||
		 	 ($arg eq "\L--NoGlobalProps")) {	# GlobalProps=1
		$Option{DEFAULT}{GLOBALPROPS}=0;
	}
	$i++;
  }
}


##########################################################################
# Routine: Initialize
#
# Description:
# This routine initializes hashes to make searching faster during
# runtime
#
# Parameters:
# 
# Returns:
sub Initialize {
	my ($table, $CPtable, $type);
	my ($table, $page, $rulebase);
	my (@pages, @tables);
		
	foreach $table (keys %Table) {
		$CPtable = $Table{$table}{TABLE};
		foreach $type (keys %{$Table{$table}{TYPE}}) {
			$Type2Table{lc($CPtable).':'.lc($type)} = $table
		}
	}
	
	foreach $rulebase (keys %Page) {
		@pages = @{$Page{$rulebase}{'PAGES'}};
		foreach $page (@pages) {
			@tables = @{$Page{$rulebase}{uc($page)}{'TABLES'}};
			foreach $table (@tables) {
				$Table2Page{uc($rulebase)}{uc($table)} = $page;
			}
		}
	}
}


##########################################################################
# Routine: Option
#
# Description:
# Function to return the option specified to an external source
#
# Parameters:
# $rulebase	Retrieve option for specific rulebase. If omitted,
# 			the global options are assumed (optional)
# $var		Option value to return
#
# Returns:
# The value of the option requested. If the option is not defined in
# the rulebase, the default section is used
sub Option {
	use strict;
	my ($rulebase, $var);
	my ($value);

	$var = uc(shift);
	if (@_) {
		$rulebase=$var;
		$var = uc(shift);
	}

	if (!$rulebase) {
		$value = $Option{$var};
	} elsif (defined($Option{$rulebase}{$var})) {
		$value = $Option{$rulebase}{$var};
	} elsif (defined($Option{DEFAULT}{$var})) {
		$value = $Option{DEFAULT}{$var};
	} else {
		$value = '';
	}
	
	return $value;
}


##########################################################################
# Routine: Page
#
# Description:
# Function to return the page definitions to an external source
#
# Parameters:
# $rulebase	Retrieve option for specific rulebase. If omitted,
# 			the global settings are assumed (optional)
# $page		Retrieve definition of specific page (optional)
# $var		The variable requested (PAGES or TABLES)
#
# Returns:
# The list of values of the option requested. 
#
# Note: If the option is not defined in the rulebase, the default
#       values will be returned

sub Page {
	use strict;
	my ($rulebase, $page, $var);
	my ($value, @list);
	my ($var1, $var2, $var3);
	
	$rulebase='DEFAULT';
	($var1, $var2, $var3) = @_;
	# Request for PAGES
	if    (uc($var1) eq 'PAGES')  { $var='PAGES' }
	elsif (uc($var2) eq 'PAGES')  { $var='PAGES'; $rulebase=uc($var1) }
	# Request for TABLES
	elsif (uc($var2) eq 'TABLES') { $var='TABLES'; $page=uc($var1) }
	elsif (uc($var3) eq 'TABLES') { $var='TABLES'; $page=uc($var2); $rulebase=uc($var1) }

	if ($var eq 'PAGES') {
		if (defined($Page{$rulebase}{PAGES})) {
			@list = @{$Page{$rulebase}{PAGES}};
		} elsif (defined($Page{DEFAULT}{PAGES})) {
			@list = @{$Page{DEFAULT}{PAGES}};
		} else {
			@list = ();
		}
		return @list;
		
	} elsif ($var eq 'TABLES') {
		if (defined($Page{$rulebase}{$page}{TABLES})) {
			@list = @{$Page{$rulebase}{$page}{TABLES}};
		} elsif (defined($Page{DEFAULT}{$page}{TABLES})) {
			@list = @{$Page{DEFAULT}{$page}{TABLES}};
		} else {
			@list = ();
		}
		return @list;
	}
}

		
##########################################################################
# Routine: Rulebase
#
# Description:
# Function to return if the rulebase should be processed. This can be set
# by using include/exclude statements in the .ini file
#
# Parameters:
# $rulebase	Rulebase name to check for
#
# Returns:
# False(0) or True(1) depending if the rulebase should be generated
sub Rulebase {
	use strict;
	my $rulebase = uc(shift);
    
	if (!$rulebase) {
		return 0;
	} elsif (defined($Option{RULEBASES}{$rulebase})) {
		return $Option{RULEBASES}{$rulebase};
	} else {
		return $Option{RULEBASES}{_DEFAULT_}
	}
}


##########################################################################
# Routine: Object2Page
#
# Description:
# Function to return the page a table is located on
#
# Parameters:
# $rulebase	Rulebase name
# $CPtable	CheckPoint object Table
# $CPtype	CheckPoint object Type
#
# Returns:
# The name of the page the table is located on ...
sub Object2Page {
	use strict;
	my ($table, $page);
	
	my $rulebase = uc(shift);
	my $CPtable  = lc(shift);
	my $CPtype   = lc(shift);
	
	$table = $Type2Table{$CPtable.':'.$CPtype};
	if ($Table2Page{$rulebase}) {
		$page  = $Table2Page{$rulebase}{$table};
	} else {
		$page  = $Table2Page{'DEFAULT'}{$table};
	}

	return $page;
}

1;
