#!/usr/bin/perl
##########################################################################
# Package: CheckPoint.pm
#
# Description:
# This package holds the routines to read the database files from
# Check Point and store them in ons single hash structure: %Objects
# All data is kept local to this package, so the internal routines
# GetObjects, GetKey, SetKey and GetMembers need to be used to access
# the information from external.
#
# Routines:
# GetObjects	Reads an entire CheckPoint database file and stores
#				the data in the hash %Objects
# GetMembers	Returns the members of a specific key
# ObjectUsed	Marks the object as used in the given rulebase
# GetKey		Returns the value of a specific key
# SetKey        Set the value of a key and returns the ref to key
##########################################################################
# This routine is not supported by Check Point in any way!
# Use of this software is free, but is for your own risk!
##########################################################################
package CheckPoint;

use strict;
use Debug;
use CPobject;
use Strings;

our (@ISA, @EXPORT);
require Exporter;
@ISA = qw(Exporter Debug);
@EXPORT = qw(GetObjects GetMembers ObjectUsed GetKey SetKey);

# Global variables ...
my $objects;
my $_CPclass = 'CPobject';


##########################################################################
# Routine: GetObjects
#
# Description:
# This routine reads tables from a CheckPoint file and stores
# the data in a hash structure to be processed later on
#
# Parameters:
# $file   - Filename of CheckPoint databasefile
#
# Returns:
# $objects - Root of objects tree
sub GetObjects {

	use strict;
	my ($line, $level);
	my ($section, $key);

	my $file   = shift;

	if ($objects) {
		# already initialized ...
	} else {
		# first table to be read, so initialise root object!
		$objects = CPobject->new();
	}

	open (INFILE, "$file")
		or die "Cannot open the CheckPoint database $file!\n\n";

	$level = 0;
	&ReadObject(undef, undef, $objects, $level);

	close (INFILE);
	return $objects;
}

##########################################################################
# Routine: ReadObject
#
# Description:
# This routine reads the CheckPoint table from the given key
# and below that from the input file. Note this routine is
# called recursively, so effectivily it reads the entire
# file.
#
# Parameters:
# $parent		a reference to the parent object
# $parent_key	the key used in the parent object to
#				get to the current object
# $current		the reference to the current object
# $level		the level of depth in the database file;
#				Used only for debugging information
#
# Returns:
# nothing; the data is in the global hash $objects
sub ReadObject {
	use strict;
	my ($line, $level, $prev_level);
	my ($EndOfFile, $EndOfSection);
	my ($parent, $parent_key);
	my ($current, $key, $val);
	my ($child, $child_key);
	my ($grandchild);
	my ($key, $ref, $val);

	my ($parent, $parent_key, $current, $level) = @_;

	$prev_level = $level-1;
	$EndOfSection=0; $EndOfFile=0;

	while (($EndOfSection==0) && ($EndOfFile==0)) {
		if ($line = <INFILE>) {
			$line = fromdos($line);

			# Determine SYNTAX of line ...
			$key=undef; $ref=undef; $val=undef;
			if    ( $line =~ /:"(.*?)"\s\((.*)/ ) { $key='"'.$1.'"'; $ref=$2 }
			elsif ( $line =~ /:"(.*)"/ )          { $key='"'.$1.'"' }
			elsif ( $line =~ /:(.*?)\s\((.*)/ )   { $key=$1; $ref=$2 }
			elsif ( $line =~ /:(.*)/ )            { $key=$1 }

			# Determine if reference is VALUE ...
			if ( $ref =~ /\)$/ )                  { $val=$ref; $ref=undef; $val =~ s/\)$// }

			# Determine if reference is Anonymous ...
			# ': ('
			if (((defined($key)) && ($key eq '')) &&
				((defined($ref)) && ($ref eq ''))) { $ref='AnonymousObject' }


			# Found a VALUE to be set
			# ':key (value)'
			# ': (value)'
			if (defined($val)) {
				&debug("[$level]"." "x$level.":".$key." (".$val.")\n") if ($key ne '');
				&debug("[$level]"." "x$level.": (".$val.")\n")         if ($key eq '');
				if ($key) {
					$current->{$key} = $val;
				} else {
					# Special case: $parent checks out to be a list of values,
					# thus make the key an array iso a hash element ...
					if ((ref($parent->{$parent_key}) eq $_CPclass) ||
						(ref($parent->{$parent_key}) eq 'HASH')) {
						$parent->{$parent_key} = undef;

					}
					push @{$parent->{$parent_key}}, $val;
				}

			# Found a KEY of the existing object
			# ':key ('
			# ':key (ref' -> ref is used for comments only. ignored
			} elsif ($key ne '') {
				&debug("[$level]"." "x$level.":".$key." ("."\n") 		if (!$ref);
				&debug("[$level]"." "x$level.":".$key." (".$ref."\n") 	if ($ref);

				if ($ref eq 'ReferenceObject') {
					# Create new CPobject for REF
					$child = CPobject->new();
				} else {
					$child = {};
				}
				if (defined($current->{$key})) {
					# KEY already in use ...
					if ((ref($current->{$key}) eq $_CPclass) ||
						(ref($current->{$key}) eq 'HASH')) {
						# Key checks out to be a list of values, thus
						# make the key an array iso of a hash element ...
						$val = $current->{$key};
						$current->{$key}=undef;
						push @{$current->{$key}}, $val;
						$val = undef;
					}
					push @{$current->{$key}}, $child;

				} else {
					$current->{$key} = $child;
				}
				&ReadObject($current, $key, $child, $level+1);

			# Found a reference to an OBJECT called REF
			# ': (ref'
			} elsif ($ref ne '') {
				&debug("[$level]"." "x$level.": (".$ref."\n");
				# Create new CPobject for REF
				$child = CPobject->new();

				# In case of a 'ReferenceObject' or 'AnonymousObject'
				# push it on the stack, else reference
				if (($ref eq 'ReferenceObject') ||
					($ref eq 'AnonymousObject')) {
					push @{$current->{$ref}}, $child;
				} else {
					$current->{$ref} = $child;
				}
				&ReadObject($current, $ref, $child, $level+1);

			# Check for new level or termination of existing level ...
			# '('
			# ')'
			} else {
				if ( $line =~ /\((.*)$/ )  {
					&debug("[$level]"." "x$level."("."\n");
					&ReadObject($parent, $parent_key, $current, $level+1);
				} elsif ( $line =~ /\)(.*)$/ )  {
					&debug("[$level]"." "x$prev_level.")"."\n");
					$EndOfSection=1;
				}
			} # End of IF statement
		} else {
			$EndOfFile=1;
		} # End of IF statement to check for EOF
	} # Until EOF or End Of Section
}


##########################################################################
# Routine: GetKey
#
# Description:
# This routine returns the value of a key
#
# Parameters:
# $obj		Start of search object. If omitted, root is
#			assumed (optional)
# $keystr	':'-separated list of keys
#
# Returns:
# $obj		Value of key requested
sub GetKey {
	my ($obj, $keystr);
	my (@key, $key);

	$obj    = $objects;
	$keystr = shift;
	if ($keystr eq __PACKAGE__) {$keystr = shift}

	if (ref($keystr) ne '') {
		$obj=$keystr;
		$keystr=shift;
	}

	while ($keystr ne '') {
		@key = split(/:/, $keystr);
		foreach $key (@key) {
			if ((ref($obj) ne '') && exists($obj->{$key})) {
				$obj=$obj->{$key}
			} else {
				return "";
			}
		}
		$keystr = shift;
	}
	return $obj;
}

##########################################################################
# Routine: SetKey
#
# Description:
# This routine enables changing the global hash
#
# Parameters:
# $obj   	Object/key to define (new) key in
# $key		Name of key to define
# Optional:
# $val      Reference(!) to value (might be SCALAR, ARRAY, HASH, $_CPclass)
#           If ommitted the key will be destroyed
#  - or -
# $table	Table name of the object
# $name		Name of the object
# $key		Name of key to define
# Optional:
# $val      Reference(!) to value (might be SCALAR, ARRAY, HASH, $_CPclass)
#           If ommitted the key will be destroyed
#
# Returns:
# if the key has been changed, the reference to new or changed key
# else a reference to the object itself
sub SetKey {
	use strict;
	my ($obj, $table, $name);
	my ($key, $val);

	$obj = shift;
	if ($obj eq __PACKAGE__) {$obj = shift}

	if (ref($obj) eq '') {
	    $table = $obj;
	    $name  = shift;
	    $obj   = $objects->{$table}{$name};
	}
	$key = shift;	# Name of new key
	$val = shift;	# Reference to value (SCALAR, ARRAY, HASH or $_keyclass)

   	# Check if object is referenced (not a value)
   	if ((ref($obj) eq $_CPclass) ||
   		(ref($obj) eq 'HASH')) {
   		# Check if key is legal ...
		if ($key ne '') {
    		# Check if key already exists; if so destroy ...
    		if (exists($obj->{$key})) {
    			$obj->{$key} = undef;
    		}
    	}
    	# Check if value is supplied ...
    	if ($val) {
    		# Write key ...
    		if (ref($val) eq 'SCALAR') {
    			$obj->{$key} = ${$val};
    		} elsif (ref($val) eq 'ARRAY') {
    			push @{$obj->{$key}}, @{$val};
    		} elsif (ref($val) eq 'HASH') {
    			$obj->{$key}=$val;
    		} elsif (ref($val) eq $_CPclass) {
    			$obj->{$key}=$val;
    		} else {
    			$obj->{$key}=$val;
    		}
    		$obj = $obj->{$key};
    	}
    }

	return $obj;
}

##########################################################################
# Routine: GetMembers
#
# Description:
# This routine returns a list of members of the requested
# key. The term members is used loosely. See inline comments
# for more details
#
# Parameters:
# $hash		Key to search for members
#
# Returns:
# @members	Array of members. Use with care as these can be
#			other references of values in itself
sub GetMembers {
	my $hash = shift;
	my (@members, $i);
	if ($hash eq __PACKAGE__) {$hash = shift}

	if ($hash eq '') {
		# Empty element ...
		@members = ();
	} elsif (ref($hash) eq 'ARRAY') {
		@members = @{$hash}
	} elsif (ref($hash) eq 'SCALAR') {
		@members = ($hash)
	} elsif (exists $hash->{"ReferenceObject"}) {
		@members = @{$hash->{"ReferenceObject"}};
	} elsif (exists $hash->{"AnonymousObject"}) {
		@members = @{$hash->{"AnonymousObject"}};
	} elsif (exists $hash->{0}) {
		# This object holds a numbered list of elements, f.i. interfaces
		$i=0;
		while (exists $hash->{$i}) {
			push @members, $hash->{$i};
			$i+=1;
		}
	} elsif (ref($hash) eq $_CPclass) {
		# This is an actual CPobject without members; f.i. an empty group ...
		@members=();
	} elsif (ref($hash) eq 'HASH') {
		# This is a weird case, return reference to itself;
		# Might be a a list of one object, and is therefore note converted
		# to a list yet. F.i. a single rule in a rulebase,
		# or an action rulebase object
		push @members, $hash
	}

	return @members;
}

##########################################################################
# Routine: ObjectUsed
#
# Description:
# This routine flags the Check Point object(!) requested as used. If this
# object has members, those are also flagged as being used.
#
# Parameters:
# $object  Object/key to flag used
# $key     Reference to where it is used (f.i. rulebase)
#
# Returns:
# n  The number of objects changed recursively, not used before
# 0  The object referenced is no CheckPoint object (0 objects changed)
sub ObjectUsed {
	use strict;
	my ($object, $key, $n);
	my ($is_used, $obj, $newobj, $nrobj);
	my ($is_used_before);

	$object = shift;
	if ($object eq __PACKAGE__) {$object = shift; print "*** object shifted\n"}
	$key = shift;
	$n = shift;

	if (!$n) { $n = 0 }
	$nrobj = 0;
	$is_used_before = 0;

	# Check if object referenced is a CheckPoint object ...
	if (ref($object) eq $_CPclass) {
		# Create hash 'is_used' if necessary ...
		$is_used = GetKey($object, 'is_used');
		if (!$is_used) {
			$is_used = SetKey($object, 'is_used', {});
		}
		# Check if object is used before ...
		if (GetKey($is_used, $key) eq 'true') {
			$is_used_before = 1
		} else {
			# Set is_used parameter ...
			SetKey($is_used, $key, 'true');
			$nrobj += 1;
		}
	}

	if (!$is_used_before) {
       	my ($k, $l);
       	# Check recursively for used objects ...
    	if ($n < 30) {
        	if ((ref($object) eq 'HASH') ||
        		(ref($object) eq $_CPclass)) {
        		foreach $k (keys %{$object}) {
        			if (ref($object->{$k}) eq $_CPclass) {
						# Check if object is a ReferenceObject and not an Anonymous object ...
        				if ($object->{$k}{Name} && $object->{$k}{Table}) {
	        				$newobj = GetKey($object->{$k}{Table}, $object->{$k}{Name});
    	    				$nrobj += ObjectUsed($newobj, $key, $n+1);
   						}
        			} elsif (ref($object->{$k}) eq 'HASH') {
        				$newobj = $object->{$k};
        				$nrobj += ObjectUsed($newobj, $key, $n+1);
        			} elsif (ref($object->{$k}) eq 'ARRAY') {
        				foreach $l (@{$object->{$k}}) {
        					if (ref($l) eq $_CPclass) {
        						# Check if object is a ReferenceObject and not an Anonymous object ...
		        				if ($l->{Name} && $l->{Table}) {
	        						$newobj = GetKey($l->{Table}, $l->{Name});
    	    						$nrobj += ObjectUsed($newobj, $key, $n+1);
   								}
        					} elsif (ref($l) eq 'HASH') {
        						$newobj = $l;
        						$nrobj += ObjectUsed($newobj, $key, $n+1);
        					}
        				}
        			}
        		}
        	}
    	} else {
    		print "To many recursions found, skipping ...\n";
    	}
    }
	return $nrobj;
}

1;

__END__

=head1 Name

B<CheckPoint.pm> - Perl package to read and use CheckPoint database files

=head1 Description

This package is created to support programs that need information from the CheckPoint databases. It is written originally to support CPRULES, a tool to translate CheckPoint FW-1 rulebases to html format. However other purposes might come up ...

The package contains routines to read the provided files and stores them in memory. The I<GetKey> and I<GetMembers> routine makes the data easily accessible for the hostprogram.

B<Warning:> To access the data from the database, one has to know the exact location/syntax of the information in the Check Point tables. This requires thorough understanding of the CheckPoint database structure.

=head1 Technical description

The data read from the CheckPoint files is retained within the package. That means that the main variable (%objects) is not exported (not known outside the package). Routines are provided to allow the data to be read and/or changed by external programs.

Internal to the package, the database is one big hash structure. When reading a file, the contents of this file is added to the same structure over and over again. This will not pose a problem, as long as all CheckPoint data is stored in different tables. For example:

  $objects{network_objects}

is the ref to the network_objects table. And to get to a specific value, in this case of a service, one would write:

  $port = $objects{services}{http}{port};

This will normally hold the value '80', as http uses tcp port 80 to do its magic :-)


A special case worth mentioning is:

  $objects{rule-base}

This will hold a reference to all rulebases defined. If one checks the rulebases file, one could be puzzled. For in the CheckPoint database all rulebases are stored using the same key 'rule-base' (without a proper rulebase name). The name of the rulebase is actually stored much deeper in the structure (see the coding example below). That obviously makes it hard to distinguish between the different rulebases.

To solve this problem, the routine converts the value C<$objects{rule-base}> to an array element. Within this array every member holds the reference to one rulebase.

So the ref to the 'Standard' rulebase might actually be:

  $objects{rule-base}[2]

and to make it a bit more confusing, the fourth rule in this rulebase would be:

  $objects{rule-base}[2]{rule}[3]

as all the rules are stored as 'rule'.

This is however NOT how this package should be used. In such cases the I<GetMembers> routine will proof its worth.

To start you of, the code below will read the rulebases file and get all comments as noted in the individual rules of all rulebases:

  use CheckPoint;

  my ($rulebase, $rule);
  my ($rulebase_name, $rule_comment);
  my (@rulebases, @rules);

  # Read the rulebases
  GetObjects('rulebases_5_0.fws');

  # Get a list of all rulebases
  @rulebases=GetMembers(GetKey('rule-base'));
  foreach $rulebase (@rulebases) {
    # Get the name of the rulebase
    $rulebase_name = GetKey($rulebase, 'collection:Name');

    # Get a list of all rules in the rulebase
    @rules=GetMembers(GetKey($rulebase, 'rule'));
    foreach $rule (@rules) {
      # Get the comments of this rule
      $rule_comment = GetKey($rule, 'comments');

      # Print the comments to stdout
      if ($rule_comment) {
        print $rulebase_name.': '.$rule_comment."\n";
      }
    }
  }

B<Note:> All the data is retained within the package and can only be accessed through the GetKey and GetMembers routines. A few adaptations could be made to export the hash variables themselves, but that is not recommended.

=head1 Routines

=head2 GetObjects

The first step of any program would be to read the CheckPoint database files in to memory. The most important file is currently 'objects_5_0.C' which holds all objects and its parameters. Other files are 'rulebases_5_0.fws' and 'user.C' (although it is not clear to me what this file exactly is meant to do).

B<Syntax:> GetObjects(I<filename>)

The routine reads the file and stores it in one single hash. The reference to this hash-structure is returned. If more consecutive files are read, the tables are stored in the same hash-structure!

B<Examples:>

To read the objects database in memory you would use the command below. The returned value is a reference to the root of the database.

  $objects=GetObjects("objects_5_0.C");

Next the rulebases will be added to the same hash

  $objects=GetObjects("rulebases_5_0.fws");


=head2 GetKey

Now the information is stored in memory, the actual work can begin. To retrieve the information from the database, one uses the I<GetKey> function or I<GetMembers> as described later on.

B<Syntax:> GetKey([I<object_reference>], 'I<key>[I<:more_keys>]')

- where the I<object_reference> is optional. If it is not specified, the root object is assumed.

- where the I<key>-part is a colon-delimited list of keys enclosed within quotes (see examples).

B<Returns:> The value of the key requested, or an empty string if the key could not be found.

B<Examples:>

To get the type of an object called 'InternalNetwork':

  $type = GetKey('network_objects:InternalNetwork:type');

which will probably return the scalar value 'network'

... or if the reference to an object is first stored in a variable:

  $obj   = GetKey('network_objects:InternalNetwork');
  $color = GetKey($obj, 'color');

which might f.i. return the scalar value 'blue'

  $class = GetKey($rule, 'AdminInfo:ClassName');

which will return the class of a rule (referenced by C<$rule>), f.i. the scalar value 'security_rule'

... the same principle is applied here

  $obj  = GetKey('network_objects:MailServer');
  $smtp = GetKey($obj, 'smtp');

which will return a reference to the hash that holds all SMTP specific parameters. So now you would be able to do:

  $rcpt = GetKey($smtp, 'maxrecipients');

which might return the scalar value of '50'


=head2 SetKey

To change, add or even remove information to/from the database, one can use the I<SetKey> function.

B<Syntax:> SetKey([I<object_reference>], 'I<key>', ['I<value>' | I<value_reference>])

- where the I<object_reference> is optional. If it is not specified, the root object is assumed.

- where the I<key>-part is the name of the (new) key enclosed within quotes (see examples). Note this can NOT be a colon delimited list as is allowed in the I<GetKey> routine.

- where the I<value> is a scalar, or the I<value_reference> is the reference to a datastructure such as a hash or array. If neither is specified, the key will be destroyed.

B<Returns:> The reference to the newly created key if a value was set successfully, or the reference to the original object if no value was set

B<Examples:>

To set the color of an object called 'InternalNetwork' to 'blue':

  $obj = GetKey('network_objects:InternalNetwork');
  SetKey($obj, 'color', 'blue');
  - or combined in one line -
  SetKey(GetKey('network_objects:InternalNetwork'), 'color', 'blue');

To add a list of names to an object (note the reference to the array!):

  @names = ('jane', 'mark', 'misterX', 'paul');
  $obj = GetKey('network_objects:ServerA');
  SetKey($obj, 'Names', \@names);

To remove the above list of names:

  $obj = GetKey('network_objects:ServerA');
  SetKey($obj, 'Names');

To create a new hash in an object called 'is_used' and set a value within that hash (such as used in the I<ObjectUsed> routine):

   $obj = GetKey('network_objects:MyServer');
   $obj = SetKey($obj, 'is_used', {});
   SetKey($obj, 'byMe', 'true');


=head2 GetMembers

This routine is to retrieve the members of an object. This term is used loosely to accomodate different situations, but members should always be organized in some kind of list. For example the members of a group, or a numbered list within the object. So B<members are NOT all the subkeys of an object/key>. This definition makes the I<GetMembers> routine unsuitable to get a list of all (f.i.) services, or network-objects. These are not stored in any kind of list, but in a hash structure (see below for an example on this).

B<Syntax:> GetMembers([I<object_reference>])

The routine returns an array with the members of the requested object (specified by the object_reference). The members are determined in the order specified below.

1. The first method CheckPoint uses to store a list is to use several items with the same name (f.i. rules). As described before, these elements will
have been placed in a list already. So if one asks for all rules in a rulebase, the corresponding list will be returned.

2. If the object referenced only holds a single value (SCALAR), this value is returned as a list with only one member.

3. The normal way for CheckPoint to store members is in one or more occurrences of a I<'ReferenceObject'> (referencing a different object by I<Table> and I<Name>). These objects are stored in a list called I<'ReferenceObjects'>. Sometimes a key holds a list of unnamed values or objects. These unnamed values/objects are stored in a list called I<'AnonymousObjects'>. In both cases the list is returned as being the memberlist.

4. Then it can occur that a numbered list is used. This happens f.i. when enumerating the interfaces of an object. All numbered items are returned in the list.

5. If the object referenced has no members, but it is defined as a CheckPoint object, an empty list is returned. This is f.i. the case when querying an empty group.

6. Finally if all tests fail, it is concluded that this key is a list in itself. The function will return the object reference itself as a list with only one member. This can occur f.i where the rulebase has only one rule. During the import of the database there was no indication that this rule was actually the start of a list of rules. Therefore it has not been converted to a list (yet)!

A special known situation is the ACTION key of a rule. It behaves like the start of a list of I<actions>, so it is guessed there might be more ACTIONS in one rule in the future.

B<Examples:>

To get the members of a simple group:

  @members = GetMembers(GetKey('network_objects:ServerGroup'));

To get all rules of a given rulebase:

  @rules = GetMembers(GetKey($rulebase, 'rule'));

B<Note:> To get a list of all services, this WILL NOT work:

  @services = GetMembers(GetKey('services'));

to get a list of services, one should use:

  @services = (keys %{GetKey('services')});


=head2 ObjectUsed

The routine I<ObjectUsed> is to flag a CheckPoint object as being 'used' in a rulebase. This is usefull to filter out the objects which are related to in a specific rulebase, or to determine which objects are not used at all.

Some objects use other objects. For instance on the interface of a gateway, one can configure an anti-spoofing group. This group is not necessarily used anywhere else, but because the gateway object is, the anti-spoofing group is used too. So it should me marked as such. This is accomplished by recursively calling the routine for those objects that do have 'children'.

The routine uses the function I<SetKey> to add the elements to the database.

B<Syntax:> ObjectUsed([I<object_reference>], I<where_used>)

- where the I<object_reference> can be any object in the database. However, only CheckPoint objects will be marked as 'used'.

- the I<where_used> defines the rulebase the object is used in. The function will create a new key call 'is_used' which is a hash holding the names of rulebases the object is used in.

B<Returns:> The number of objects marked as being used recursively. Note however that every object can only me marked 'used' once. So this number only reports the objects marked that has not been marked before.

B<Examples:>

  $NrObjectsChanged = ObjectUsed(GetKey('network_objects:MyIntranet'), 'ThisRulebase');


=head1 History

This is version 1.0. There is no history yet, as this is the first release...

=head1 Bug Reports

Please send bug reports to the author. Please include a detailed description of the problem and some test data to reproduce the problem. 

=head1 Author

Peter-Paul Worm, (Peter-Paul.Worm@wormnet.nl)

=cut
