#!/usr/bin/perl

#############################################################################
# CPrules.pl: Analyze and print a report of Firewall-1 rules and objects
#
# Note: this script is unsupported by Checkpoint or representatives.
#
# Copyright (C) 2003 Peter-Paul Worm
#               originally based on fw1rules from Volker Tanger
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#############################################################################

my $SCR_NAME 	= 'CPRules';
my $VERSION  	= '1.03';

my $lib_dir;
my $working_dir;
BEGIN {
        if ($0 =~ /(.*)\\(\S*)$/) {
                $working_dir=$1;
                $lib_dir=$working_dir.'\Lib'
        } elsif ($0 =~ /(.*)\/(\S*)$/) {
                $working_dir=$1;
                $lib_dir=$working_dir.'/Lib'
        } else {
                $lib_dir='Lib'
        }
}
use lib $lib_dir; 


# Constant definitions ...
use Constants;

# Special purpose packages which store their own data ...
use CheckPoint;
use Options;

# CPRules specific library ...
use tables2html;
use settings2html;

# General purpose libraries used ...
use Template;
use Strings;
use Debug;

my $INIFile  	= 'CPrules.ini';
my $PageFile	= 'CPpages.def';

##########################################################################
#                            MAIN ROUTINE                                #
##########################################################################
# Read options ...
&ReadOptions($INIFile, $PageFile, $working_dir);

# Set debugging on if requested ...
if (Option('Debug')) {&debugging(1, 'debug.info')}

# Read Objects ...
print "Reading objects ...\n";
my $object = &GetObjects(Option('FW1Objects'));

# Read Rulebases ...
print "Reading rulebases ...\n";
my $rule = &GetObjects(Option('FW1Rules'));

# Output HTML pages ...
print "Generating HTML pages ...\n";
&OutputHTMLPages;

# Turn debugging off ...
if (Option('Debug')) {&debugging(0)}

##########################################################################
#                       END OF MAIN ROUTINE                              #
##########################################################################


##########################################################################
# Routine: OutputHTMLPages
#
# Description:
# Write Objects to HTML files. Relating to the page definitions
# in CPpages.def, the different pages are generated
#
# Parameters:
# (none)
#
# Returns: nothing, html pages are written to file
sub OutputHTMLPages {
	use strict;
	my ($htmlpage, $filename, $file, @files);
	my (@rulebases, $rulebase, $rb_name, $rb_dir);
	my (@pages, $page, @tables, $table);
	my (%NrTables, %NrObjects);

	#--------------------------------------------------------------------
	# Create GLOBAL PROPERTIES html code for this management server ...
	#--------------------------------------------------------------------
	print "Processing Global Properties\n";

	$htmlpage = Template->new(filename=>Option('TmplPath').'/globalproperties.tmpl');
    		
	# Remove old file ...
	unlink Option('HTMLPath').'/_globalproperties.html';

	# Open the output file ...
	open (FH, ">",Option('HTMLPath').'/_globalproperties.html')
		or die "Cannot open the temp OutputFile ".Option('HTMLPath')."/_globalproperties.html!\n\n";
   
	$htmlpage = HTMLGlobalProperties($htmlpage);

	# Write filled template file to output ...
	if ($htmlpage ne '') {
		print FH $htmlpage->output;
	}
	# Close the html page ...
	close (FH);


	#--------------------------------------------------------------------
	# Create Rulebase dependent html pages ...
	#--------------------------------------------------------------------
	$table = $Table{RULES}{TABLE};
	@rulebases=GetMembers(GetKey($table));
	foreach $rulebase (@rulebases) {
		$rb_name = GetKey($rulebase, 'collection:Name'); 
		# Check if this rulebase should be converted to HTML ...
		if (Rulebase($rb_name)) {
    		print "Processing rulebase: ".$rb_name."\n";
    
    		# Create directory for rulebase (will return error if already exists, but this is ignored)
    		$rb_dir = Option('HTMLPath')."/".$rb_name; 
    		mkdir $rb_dir;
    
    		#--------------------------------------------------------------------
    		# Remove all *.html files from the HTML directory ...
    		#--------------------------------------------------------------------
    		@files = glob($rb_dir.'/*.html');
    		unlink @files;
    		@files = glob($rb_dir.'/*.tmpl');
    		unlink @files;
    		
    		#--------------------------------------------------------------------
    		# Copy Global Properties html code if requested ...
    		#--------------------------------------------------------------------
    		if (Option($rb_name, 'GlobalProps')) {
        		open (FH, ">", $rb_dir.'/globalproperties.html')
        			or die "Cannot open the output file $rb_dir/globalproperties.html!\n\n";

    			open (INFILE, Option('HTMLPath').'/_globalproperties.html');

    			while (<INFILE>) {
    				print FH strip($_)."\n" if /\S/
    			}
        		close INFILE;
    			close FH;
    		}
    		
    		#--------------------------------------------------------------------
    		# Create HEADER html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(filename=>Option('TmplPath').'/header.tmpl');
    
    		# Open the file ...
    		open (FH, ">",$rb_dir.'/header.html')
    			or die "Cannot open the the OutputFile $rb_dir/header.html!\n\n";
    
    		$htmlpage = HTMLHeader($rulebase, $htmlpage);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    
    		#--------------------------------------------------------------------
    		# Create FOOTER html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(filename=>Option('TmplPath').'/footer.tmpl');
    
    		# Open the file ...
    		open (FH, ">",$rb_dir.'/footer.html')
    			or die "Cannot open the the OutputFile $rb_dir/footer.html!\n\n";
    
    		$htmlpage = HTMLFooter($rulebase, $htmlpage);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    
    		#--------------------------------------------------------------------
    		# Create POLICY INSTALLATION TARGETS html code ...
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(filename=>Option('TmplPath').'/targets.tmpl');
    		
    		# Open the file ...
    		open (FH, ">",$rb_dir.'/targets.html')
    			or die "Cannot open the OutputFile $rb_dir/targets.html!\n\n";
    
    		$htmlpage = HTMLTargets($rulebase, $htmlpage);

    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);

       		#--------------------------------------------------------------------
    		# Create template files for the PAGES to generate ...
    		#--------------------------------------------------------------------
    		@pages=Page($rb_name, 'PAGES');
    		foreach $page (@pages) {
    			open (FH, ">", $rb_dir.'/'.lc($page).'.tmpl')
    				or die "Cannot open the output-template file $rb_dir/$page.tmpl!\n\n";
    
    			@tables=Page($rb_name, $page, 'TABLES');
    			foreach $table (@tables) {
    				print FH '<TMPL_INCLUDE "'.Option('TmplPath').'/'.lc($table).'.tmpl">'."\n";
    			}
    			
    			close FH;
    		}		
    
    		#--------------------------------------------------------------------
    		# Create html code for all PAGES ...
    		#--------------------------------------------------------------------
    		@pages=Page($rb_name, 'PAGES');
    		foreach $page (@pages) {
    			# Assume page is empty ...
    			$NrTables{$page}=0;
    			
    			# Determine template file to use ...
    			$htmlpage = Template->new(filename=>$rb_dir.'/'.lc($page).'.tmpl');
    			
    			@tables = Page($rb_name, $page, 'TABLES');
    			foreach $table (@tables) {
    				$NrObjects{$table} = table2html($rulebase, $htmlpage, $table);
    				
    				# Increase nr. of tables if appropiate ...
    				if ($NrObjects{$table}) {$NrTables{$page} += 1 }
    			}

				if ($NrTables{$page}) {
        			# Open the file ...
        			open (FH, ">",$rb_dir.'/_'.lc($page).'.html')
        				or die "Cannot open the the OutputFile $rb_dir/_".lc($page).".html!\n\n";
        
        			# Write filled template file to output ...
        			if ($htmlpage ne '') {
        				print FH $htmlpage->output;
        			}
        			# Close the html page ...
        			close FH;
        		}
    
    		} # END foreach $page

    		#--------------------------------------------------------------------
    		# Create MENU code for this rulebase ...
    		# (this code can only be generated after the pages have been completed)
    		#--------------------------------------------------------------------
    		$htmlpage = Template->new(filename=>Option('TmplPath').'/menu.tmpl');
    
    		# Open the file ...
    		open (FH, ">",$rb_dir.'/menu.html')
    			or die "Cannot open the the OutputFile $rb_dir/menu.html!\n\n";
    
    		$htmlpage = HTMLMenu($rulebase, $htmlpage, \%NrTables, \%NrObjects);
    
    		# Write filled template file to output ...
    		if ($htmlpage ne '') {
    			print FH $htmlpage->output;
    		}
    		# Close the html page ...
    		close (FH);
    		
    		#--------------------------------------------------------------------
    		# Create final html pages from the section files ...
    		#--------------------------------------------------------------------
    		@pages=Page($rb_name, 'PAGES');
    		
    		foreach $page (@pages) {
				if ($NrTables{$page}) {
        			open (FH, ">", $rb_dir.'/'.lc($page).'.html')
        				or die "Cannot open the output file $page.html!\n\n";
    
    				open (INFILE, $rb_dir.'/header.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $rb_dir.'/menu.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $rb_dir.'/_'.lc($page).'.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
    				open (INFILE, $rb_dir.'/footer.html');
    			   	while (<INFILE>) {
    			   		print FH strip($_)."\n" if /\S/;
    			   	}
        			close INFILE;
       			
        			close FH;
        		}
    		}		

    		#--------------------------------------------------------------------
    		# Clean up temp files ...
    		#--------------------------------------------------------------------
    		@files = glob($rb_dir.'/*.tmpl');
    		unlink @files;
    		@files = glob($rb_dir.'/_*.html');
    		unlink @files;
    		unlink $rb_dir.'/header.html', $rb_dir.'/footer.html', $rb_dir.'/menu.html';
		}    		
	} # END foreach $rulebase
	
	#--------------------------------------------------------------------
	# Clean up Global Properties file ...
	#--------------------------------------------------------------------
	unlink Option('HTMLPath').'/_globalproperties.html';
}


##########################################################################
# Routine: HTMLHeader
#
# Description:
# Write HEADER to HTML file. The output file is included 
# in the actual output later on.
#
# Parameters:
# $rulebase	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 

sub HTMLHeader {
	use strict;
	my ($rb_name);
	
	my $rulebase = shift;
	my $template = shift;	# HEADER.TMPL
	
	# Determine rulebase name ...
	$rb_name = GetKey($rulebase, 'collection:Name');
	
	# Fill header with general variables of rulebase
	$template->param(	RULEBASE     => $rb_name,
						LASTMODIFIED => stripquotes(GetKey($rulebase, 'AdminInfo:LastModified:Time')),
						CSSPATH      => Option('CSSPath'),
						SCRIPTPATH   => Option('ScriptPath'),
						MAINLOGO     => Option('MainLogo'),
						LOGO         => Option($rb_name, 'Logo'),
						
						GLOBPROP       => Option($rb_name, 'GlobalProps'),
						GLOBPROP_PAGE  => 'globalproperties.html',
						GLOBPROP_TEXT  => 'Global Properties',
						TARGET       => 1,
						TARGET_PAGE  => 'targets.html',
						TARGET_TEXT  => 'Installation Targets'
				);

	return $template;
}


##########################################################################
# Routine: HTMLFooter
#
# Description:
# Write FOOTER to HTML file. 
#
# Parameters:
# $rulebase	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
#
# Returns: 
# $template	Filled template 

sub HTMLFooter {
	use strict;
	my $rulebase = shift;
	my $template = shift;	# HEADER.TMPL
	
	# Fill header with general variables of rulebase
	my $now = gmtime();

	$template->param(	CreatedDate  => $now,
						Script       => $SCR_NAME,
						Version      => $VERSION
				);
				
	return $template;
}


##########################################################################
# Routine: HTMLMenu
#
# Description:
# Write Menu to HTML file. This file is later added to the actual output
#
# Parameters:
# $rulebase	Reference to the rulebase in the database structure
# $template	Empty template to be filled for the html page
# \%ActiveTables  Reference to hash which holds if tables
#                 are empty or not
#
# Returns: 
# $template	Filled template 

sub HTMLMenu {
	use strict;
	my ($rb_name, $cnt);
	my (@pages, $page);
	my (@tables, $table);
	my (%style, %menu, %submenu);
	my (@styles, @menus);
	my (%NrTables, %NrObjects);
	
	my $rulebase = shift;
	my $template = shift;	# MENU.TMPL
	my $ref1   = shift;
	my $ref2   = shift;
	
	%NrTables  = %{$ref1};
	%NrObjects = %{$ref2};
	
	# Determine rulebase name ...
	$rb_name = GetKey($rulebase, 'collection:Name');
	
	$template->param( SCRIPTPATH   => Option('ScriptPath') );

	# Discover pages to be displayed ....
	@pages=Page($rb_name, 'PAGES');

	# Define styles for the menus ...
	$cnt=25;
	foreach $page (@pages) {
		if ($NrTables{$page}) {
    		my %style;
    		%style = ( 	MENU_TXT	 => $page,
    					MENU_LEFT	 => $cnt,
    					SUBMENU_LEFT => $cnt+5
    				);
    		push @styles, \%style;
    		$cnt += 100;
		}
	}
	$template->param(STYLES=>\@styles);

	foreach $page (@pages) {
		if ($NrTables{$page}) {
    		my %menu;
    		%menu = (	PAGE_TXT	=> $page,
    					PAGE_FILE 	=> lc($page)
    				);
    		@tables=Page($rb_name, $page, 'TABLES');
    		foreach $table (@tables) {
    			if ($NrObjects{$table}) {
        			my %submenu;
        			%submenu = ( SECTION_TXT => $table,
        						 SECTION_FILE => lc($page)
        					);
        			push @{$menu{SUBMENUS}}, \%submenu;
    			}
    		}
			push @menus, \%menu;
    	}
	}
	$template->param(MENUS=>\@menus);
	
	return $template;
}

__END__

=head1 Name

B<CPRules.pl> - Perl program to convert a Check Point FW-1 configuration to HTML

=head1 Description

As the management of a FW is a security-sensitive subject, one can not allow others to access the management servers. With 'others' being non-administrators of the firewall, like internal or external customers. In many cases it is even prohibited by the security policy of your company and sometimes the mgmt LAN is even physically disconnected!

However in many cases it is very convenient to have a copy of the configuration available for viewing by a select group of 'others'. This can be for a helpdesk for trouble shooting purposes (not needing to call you every five minutes), a customer demanding to have an insight in their FW configuration or just for backup purposes.

On the other hand, it's not wise to leave this kind of configurations lying around for everybody to see (you're auditors would have a field day). So you might not want everything being published, but only a subset of the configuration (f.i. only the FW rulebase itself). Such a subset would enable the customer (being either internal or external) to request meaningfull changes without having to bother the administrators first.

To be able to do all this, we need a tool to convert the Check Point files to a readable format. This tool would need to be configurable to allow administrators to define what to publish and how. CPRULES is designed to accomodate in just that wish.

Searching the internet the only tool available to convert Check Point configurations is a perl script called FW1RULES, written and maintained by Volker Tanger <volker.tanger@wyae.de>. Admittedly this program did serve very well and is widely used. This program is originally written to cope with Check Point FW-1 version 4.1 code and has later been adapted to support NG configurations as well. It drastically needed a rewrite of the code due to additions and changes made in the past. That is basically where CPRULES has taken of.

=head1 Configurable Options

There are two ways to configure the program. The first and easiest way is through the configuration file called B<CPRules.ini>. The second option is to use the commandline parameters. The advantage of the .ini file is that many parameters can be set per rulebase. This is not possible through the commandline switches.

The software will read the CPRules.ini file first and adds the commandline switches later on. This enables you to use a default setup and overwrite it at will from the commandline. If you are handling more management stations, this would enable you to call the program for every management server in turn, only differing the location of the Check Point files.

Default behaviour is to process all rulebases found in the rulebases file. In some cases you just want to get the results for one rulebase or a subset of rulebases. Both is possible through the (include and exclude) commands in the CPRules.ini file. Through the commandline it is only possible to process one (-b switch) or all rulebases.

=head2 CPRules.ini

As discussed above, the CPRules.ini is the preferred way to configure the behaviour of the program. The CPRules.ini is searched for in the current directory and if not found in the working directory of CPRules, i.e. the folder where the script itself is located. 

The configuration options are divided in three sections: I<CPRules file locations>, I<Check Point file locations> and I<Rulebase specific settings>. The next three corresponding paragraphs will outline the commands.

=head3 CPRules file locations

The first section of the CPRules.ini file holds all the path and file names needed by the software itself:

B<TmplPath> - The program uses templates to build the HTML pages. Templating allows you to make changes to the output without messing with the program. For more details about the syntax, see the documentation of the 'I<template.pm>' file. This parameter defines the location of the template files. The program searches for the folder 'Templates' in both the current- and the working directory. 

  TmplPath=./tmpl

B<HTMLPath> - The output of the program are HTML pages per rulebase. This will result in subfolders in the HTML folder per rulebase (f.i. ./html/MyRulebase). This parameter defines the location of the HTML files.

  HTMLPath=./html

=head3 HTML file locations

The second section of the CPRules.ini file holds all the path and file names needed to view the produced webpages correctly. Note that these paths are hardcoded in the HTML code produced! So you'd better check the files are there when viewing the results of the software.

B<IconPath> - The objects all have different icons per type or class. These icons have been extracted from the Check Point files and stored in a separate folder. This folder is needed for correct displaying of the HTML pages, so the path is relative to the HTML directory. The default value assumes the 'icons'-folder is a subfolder of the HTML folder. This path will be hardcoded in the HTML pages, so be sure the icons are actual there!

  IconPath=../icons

B<CSSPath> - The location of the stylesheets used. There is currently only one stylesheet used, called I<cprules.css>.

  CSSPath=../css

B<ScriptPath> - The HTML output uses three java scripts: I<menu.js>, I<back2top.js> and I<visible.js>. This parameter defines the location of those scripts relative to the HTML files.

  ScriptPath=../scripts

B<MainLogo> - This defines the location of the main logo of the program. This could be the logo of the ISP providing the service, or a self-designed CPRules logo. Note that this will be hardcoded in the HTML files, so be sure it can be found during viewing.

=head3 Check Point file locations

The third section of the CPRules.ini file holds the location of the Check Point files needed. How the files get there is completely up to you, as long as they are accessible. B<Note:> Check Point files are case-sensitive, so be careful not to misspell the filenames.

B<FW1Rules> - Check Point stores all the rulebases in one file, called 'rulebases_5_0.fws'. This is the only rulebase file needed. This parameter defines the location and name of this file relative to the work folder.

  FW1Rules=./rulebases_5_0.fws

B<FW1Objects> - Check Point stores all the objects, services, etc in one database file called 'objects_5_0.C'.  This parameter defines the location and name of this file relative to the work folder.

  FW1Objects=./objects_5_0.C

=head3 Rulebase specific settings

Default behaviour of the program is to process all rulebases found in the rulebases file. The I<Include> and I<Exclude> commands can be used to alter this behaviour. You should never have to combine both statements, but you are allowed to.

B<Exclude> - Do NOT generate HTML for this rulebase. If the first (include/exclude) command found is an Exclude command, all other rulebases are INcluded automatically.

B<Include> - Do generate HTML for this rulebase. If the first (include/exclude) command found is an Include command all other rulebases are EXcluded automatically.

B<Note:> If a rulebase is both included and excluded (???), the first entry will prevail.

B<Example:>

  Include=MyCompany
  Exclude=Standard

The rulebase specific section holds additional commands to alter the output of the program. As the program creates HTML files per rulebase, these settings must be adaptable per rulebase. To accomplish this, section headers with the name of the rulebase are used to define the setting for this specific rulebases.

To make life easier, a B<[Default]> section can (and will) be used to hold all 'default' settings. If a specific setting is not specified in a rulebase section, the corresponding setting in the [Default] section will be used. So in many practical situations, there will be no other rulebase sections at all!

B<Logo> - The HTML pages always have a logo in the left top of the page (see I<MainLogo>). The Logo parameter is used to define the image in the right top of the pages. This is designed to be the logo of the customer which rulebase is shown here. Note that this will be hardcoded in the HTML files, so be sure it can be found during viewing.

  Logo=./html/MyCompany.jpg

B<AllObjects> - By default (0) the HTML pages will only list those objects actually used in the rulebase. This parameter (if set to '1') will change this behaviour and will list all objects available. That might come in handy during testing or when documenting the configuration.

  AllObjects=0

B<GlobalProps> - By default (1) the Global Properties will be included in the output files. To suppress the Global Properties, set the value to '0'.

  GlobalProps=1

B<Example:>

  [Default]
  Logo=./html/MyCompany.jpg
  AllObjects=0
  GlobalProps=0

  [CompanyB]
  Logo=./html/CompanyB/OtherLogo.gif
  AllObjects=1
  GlobalProps=1

=head2 Commandline switches

All the default values that can be set in the CPRules.ini can also be set using a commandline switch. For a more detailed description see that section. There is always a verbose version and a short version available and the switches are case-insensitive.

To set the path to the template folder (default: ./tmpl):

  --TmplPath <path>
  -t <path>

To set the path to the HTML folder (default: ./html):

  --HTMLPath <path>
  -h <path>

To set the path to the folder with icons relative to the HTML folder (default: ../icons):

  --IconPath <path>
  -i <path>

To set the path to the folder which holds the css stylesheets (default: ../css):

  --CSSPath <path>
  -c <path>

To set the path to the folder which holds the scripts used (default: ../scripts):

  --ScriptPath <path>
  -s <path>

To define the logo in the upper left top of the HTML pages (default: (none)):

  --MainLogo <path and filename>
  -m <path and filename>
  
To define the location of the Check Point rulebase file (default: ./rulebases_5_0.fws):

  --FW1Rules <path and filename>
  -r <path and filename>

To define the location of the Check Point object database (default: ./objects_5_0.C):

  --FW1Objects <path and filename>
  -o <path and filename>

To specify one specific rulebase only (default: all rulebases):

  --Rulebase <rulebase name>
  -b <rulebase name>

To define the logo in the upper right top of the HTML pages (default: (none)):

  --Logo <path and filename>
  -l <path and filename>

To define if all objects should be listed by default, the following switch should be used. This switch turns the setting ON (it's OFF by default):

  --AllObjects

To define if the Global Properties should be excluded from the output, the following switch should be used. This switch turns the setting OFF (it's ON by default):

  --NoGlobalProps

B<Examples:>

The typical installation of the CPRules will often be static. The icons, templates and html files are in a fixed location, so only the location of the Check Point files needs to be defined. All other parameters are defined in the .ini file. Such a setup would result in a commandline as shown below (displayed over more lines for readability):

  cprules --FW1Objects /data/fwmgmt2/objects_5_0.C 
          --FW1Rules /data/fwmgmt2/rulebases_5_0.C 
          --AllObjects

To run the program for one rulebase only with the corresponding logo:

  cprules --Rulebase MyCompany --Logo ./MyCompany.gif

=head1 Page and Table definitions

This chapter describes how the output of CPRules is setup and how it can be altered. The default setup creates all HTML pages with all object types converted to HTML. This might well be not what you want. The page and table definitions enables you to change the output to your taste. 

Run a report and check it out before you start modifying the page definitions. Changing the table definitions is however not advised, because it requires extended knowledge of the Check Point database structure.

=head2 Page definitions

Which tables are presented on which HTML pages, is defined in a file called: B<CPPages.def>. It also defines which tables are presented at all! So this file allows you to show just what you want your customers to see. 

In CPPages.def the page setup can be defined per rulebase by creating a separate section per rulebase. However there is also a section called B<[Default]> which will be used if there is no section defined for a rulebase. This enables you to specify a different layout for every rulebase!

A (rulebase)section consists of pages and tables. The page names are free to choose, but the tables have to be defined in the I<Constants.pm> file. In the next paragraph more about that. Every page definition starts with a B<Page=> command, followed by one or more B<Table=> commands.

B<Example:>

  [MyRulebase]
  Page=MyPage
  Table=Hosts
  Table=Services

  Page=My2ndPage
  Table=Resources
  Table=OPSEC

=head2 Table definitions

I<B<Warning:> Changing table definitions is not straight-forward. You need to have thorough understanding of the Check Point database structure! However it provides an manageable way (for the author) to cope with future changes/additions Check Point comes up with.>

The table definitions are in a package called B<Constants.pm>. This Perl package holds all Check Point related constants used in CPRules.

The hash used for the table definitions is %Table and it is a bit complex. The purpose is to define which objects are to be displayed in which table and to define the order in which they are presented within the table. The objects within a HTML table have to be member of the same Check Point table, which is reflected in the value of B<TABLE>. I<At this time it is not possible to combine objects from different Check Point tables into one HTML table.> 

All Check Point objects do have a 'type'. Objects of the same type are normally grouped together in one table (not strictly necessary), but several types can be used in one table. The best table to show this grouping is the I<Services> table:

  SERVICES => {
    TABLE => 'services',
    TYPE  => {
      'tcp'            => '1:TCP',
      'tcp_subservice' => '2:Compound TCP',
      'tcp_citrix'     => '3:Citrix TCP',
      'udp'            => '4:UDP',
      'rpc'            => '5:RPC',
      'icmp'           => '6:ICMP',
      'other'          => '7:Other',
      'dcerpc'         => '8:DCE-RPC',
      'gtp'            => '9:GTP'
    }
  },

The SERVICES table above is getting the objects from the Check Point table 'services'. The numbers followed by a colon (':')in the text define the order in which they will be presented in the HTML page. For instance the services with type 'tcp' are presented before the services with type 'udp'. The remainder of the text is passed to the routine that creates the HTML pages and is currently used as header text for the service subsections (see the examples).

If the TYPE of an object is not sufficient to determine the different members, it is possible to define other keys to select on. An example to explain this is seen in the gateway table definition:

  GATEWAYS =>	{
    TABLE => 'network_objects',
    TYPE  => {
      'gateway'	=> {
        'AdminInfo:ClassName' => {
          'gateway_ckp'          => '1:Gateway',
          'gateway_plain'        => '6:Interoperable Device',
          'gateway_profile'      => '7:Gateway Profile'
        }
      },
      'gateway_cluster'          => '2:Gateway Cluster',
      'cluster_member'           => '3:Cluster Member',
      'sofaware_gateway_profile' => '4:Sofaware Profile',
      'router'                   => '5:Router' 
    }
  },

The type 'gateway' holds three different types of gateways. To distinguish between the three we need to look at the 'ClassName' of the objects. See above how that is defined in the hash %Table.

There is one final possibility. The rules in a rulebase are no separate objects, but are part of a list of rules. This case is reflected in the %Table by not defining a TYPE, but by defining a LIST:

  RULES => {
    TABLE => 'rule-base',
    LIST  => 'rule'
  },

For more specific details check the code, to see what is passed exactly to the corresponding routines and how that can be handled.

=head1 Version and Bug reports

CPRules version 1.0  d.d. 01/02/2004

CPRules version 1.01 d.d. 20/05/2004

=over 1

=item * 
Added working directory to the equation to search for the .ini and page layout file; 

=item * 
Fixed error when empty tables were processed;

=back

Bug reports and requests for modifications can be send to Peter-Paul.Worm@wormnet.nl

=head1 Author

Peter-Paul Worm (Peter-Paul.Worm@wormnet.nl)

